<?php

namespace Topdata\TopdataVariantsInProductBoxesSW6\Service;

use Shopware\Core\Content\Product\Exception\ProductNotFoundException;
use Shopware\Core\Content\Product\ProductEntity;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Content\Property\Aggregate\PropertyGroupOption\PropertyGroupOptionEntity;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityCollection;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsAnyFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Struct\ArrayStruct;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Symfony\Contracts\Cache\CacheInterface;
use Symfony\Contracts\Cache\ItemInterface;
use Topdata\TopdataQueueHelperSW6\Util\UtilDebug;

/**
 * 09/2024 created
 */
class ProductVariantsFetcher
{
    private const  MAX_IMAGES_PER_PRODUCT  = 99;
    private const  DISABLE_CACHE           = false; // when developing, set to true to disable caching
    private const  CACHE_FALLBACK_LIFETIME = 3600; // 1 hour

    private SalesChannelRepository $productRepository;
    private EntityRepository $seoUrlRepository;
    private ProductMediaFetcher $productMediaFetcher;
    private CacheInterface $cache;
    private SystemConfigService $systemConfig;

    public function __construct(
        SalesChannelRepository $productRepository,
        EntityRepository       $seoUrlRepository,
        ProductMediaFetcher    $productMediaFetcher,
        CacheInterface         $cache,
        SystemConfigService    $systemConfig
    )
    {
        $this->productRepository = $productRepository;
        $this->seoUrlRepository = $seoUrlRepository;
        $this->productMediaFetcher = $productMediaFetcher;
        $this->cache = $cache;
        $this->systemConfig = $systemConfig;
    }

    /**
     * Load multiple products and enrich them with their variants and media.
     *
     * @param SalesChannelProductEntity[] $products Array of product entities to be enriched.
     */
    public function loadMany(array $products, SalesChannelContext $salesChannelContext)
    {
        $context = $salesChannelContext->getContext();

        // Fetch media map (we need for image sliders)
        $productIds = array_map(function (SalesChannelProductEntity $product) {
            return $product->getId();
        }, $products);
        $mediaMap = $this->productMediaFetcher->fetchMediaOfMany($productIds, $context, self::MAX_IMAGES_PER_PRODUCT);

        // Add variants and media to products
        foreach ($products as $product) {
            $hasVariants = !empty($product->getParentId()) || $product->getChildCount() > 0;
            $groupedVariants = [];

            if ($product->getChildCount() > 0) {
                $groupedVariants = $this->getCachedProductVariants($product->getId(), $salesChannelContext);
            } elseif ($product->getParentId() !== null) {
                $groupedVariants = $this->getCachedProductVariants($product->getParentId(), $salesChannelContext);
            }

            $product->addExtension('topdataVariantsInProductBoxesSW6', new ArrayStruct([
                'hasVariants'     => $hasVariants,
                'childCount'      => $product->getChildCount(),
                'parentId'        => $product->getParentId(),
                'groupedVariants' => $groupedVariants,
                'media'           => $mediaMap[$product->getId()] ?? [],
            ]));
        }
    }

    /**
     * Get cached product variants or fetch them if not cached
     */
    private function getCachedProductVariants(string $parentId, SalesChannelContext $salesChannelContext): array
    {
        $isCachingEnabled = $this->systemConfig->get('TopDataVariantsInProductBoxesSW6.config.enableCaching') ?? true;

        if (!$isCachingEnabled) {
            return $this->fetchAndGroupProductVariants($parentId, $salesChannelContext);
        }

        $cacheKey = sprintf('product_variants_%s_%s', $parentId, $salesChannelContext->getSalesChannel()->getId());

        return $this->cache->get($cacheKey, function (ItemInterface $item) use ($parentId, $salesChannelContext) {
            $item->expiresAfter($this->getCacheLifetime());
//            $item->tag([
//                self::CACHE_TAG_PRODUCT . '-' . $parentId,
//                self::CACHE_TAG_PRODUCT . '-variants'
//            ]);

            return $this->fetchAndGroupProductVariants($parentId, $salesChannelContext);
        });
    }

    /**
     * 09/2024 created
     */
    public function fetchAndGroupProductVariants(string $parentId, SalesChannelContext $salesChannelContext): array
    {
        // Fetch variants
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('parentId', $parentId));
        $criteria->addFilter(new EqualsFilter('active', 1));
        $criteria->addAssociation('options.group');
        $criteria->addAssociation('prices');
        $criteria->addAssociation('media');
        $variants = $this->productRepository->search($criteria, $salesChannelContext)->getEntities();

        // Fetch seo urls for the variants
        $seoUrls = $this->getCachedSeoUrls($variants->getIds(), $salesChannelContext->getContext());

        return $this->groupVariants($variants, $seoUrls);
    }

    /**
     * Get cached SEO URLs or fetch them if not cached
     */
    private function getCachedSeoUrls(array $productIds, Context $context): array
    {
        $isCachingEnabled = $this->systemConfig->get('TopDataVariantsInProductBoxesSW6.config.enableCaching') ?? true;

        if (!$isCachingEnabled) {
            return $this->fetchSeoUrls($productIds, $context);
        }

        $cacheKey = 'seo_urls_' . md5(implode('_', $productIds));

        return $this->cache->get($cacheKey, function (ItemInterface $item) use ($productIds, $context) {
            $item->expiresAfter($this->getCacheLifetime());
//            $tags = [self::CACHE_TAG_SEO];
//            foreach ($productIds as $productId) {
//                $tags[] = self::CACHE_TAG_SEO . '-' . $productId;
//            }
//            $item->tag($tags);

            return $this->fetchSeoUrls($productIds, $context);
        });
    }

    /**
     * Fetch a single product and its grouped variants
     */
    public function fetchProductWithGroupedVariants(string $productId, SalesChannelContext $salesChannelContext)
    {
        $context = $salesChannelContext->getContext();

        $criteria = new Criteria([$productId]);
        $criteria->setLimit(1);

        $product = $this->productRepository->search($criteria, $salesChannelContext)->first();
        if (!$product) {
            return null;
            // throw new ProductNotFoundException($productId);
        }

        $groupedVariants = [];
        if ($product->getChildCount() > 0) {
            $groupedVariants = $this->getCachedProductVariants($product->getId(), $salesChannelContext);
        } elseif ($product->getParentId() !== null) {
            $groupedVariants = $this->getCachedProductVariants($product->getParentId(), $salesChannelContext);
            // In this case, also fetch the parent product
            // this one crashed (if the parent product was was not active)
            $parentProduct = $this->fetchProductWithGroupedVariants($product->getParentId(), $salesChannelContext);
            $product->addExtension('parent', new ArrayStruct(['product' => $parentProduct]));
        }

        $product->addExtension('topdataVariantsInProductBoxesSW6', new ArrayStruct([
            'hasVariants'     => !empty($groupedVariants),
            'childCount'      => $product->getChildCount(),
            'parentId'        => $product->getParentId(),
            'groupedVariants' => $groupedVariants,
            'media'           => $this->productMediaFetcher->fetchMediaOfOne($productId, $context, self::MAX_IMAGES_PER_PRODUCT),
        ]));

        return $product;
    }

    private function getCacheLifetime(): int
    {
        if (self::DISABLE_CACHE) {
            return 0;
        }

        return $this->systemConfig->get('TopDataVariantsInProductBoxesSW6.config.cacheLifetime') ?? self::CACHE_FALLBACK_LIFETIME;
    }

    /**
     * Group variants by their property group options
     */
    private function groupVariants(EntityCollection $variants, array $seoUrlsMap): array
    {
        $groupedVariants = [];

        /** @var ProductEntity $variant */
        foreach ($variants as $variant) {
            $options = $variant->getOptions();
            if (!$options) {
                continue;
            }

            /** @var PropertyGroupOptionEntity $option */
            foreach ($options as $option) {
                $groupId = $option->getGroupId();
                $groupName = $option->getGroup() ? $option->getGroup()->getName() : 'Unknown Group';
                $optionId = $option->getId();
                $optionName = $option->getName();

                if (!isset($groupedVariants[$groupId])) {
                    $groupedVariants[$groupId] = [
                        'groupName' => $groupName,
                        'options'   => [],
                    ];
                }

                if (!isset($groupedVariants[$groupId]['options'][$optionId])) {
                    $groupedVariants[$groupId]['options'][$optionId] = [
                        'optionName' => $optionName,
                        'variants'   => [],
                    ];
                }

                $groupedVariants[$groupId]['options'][$optionId]['variants'][] = [
                    'id'        => $variant->getId(),
                    'available' => $variant->getAvailable(),
                    'stock'     => $variant->getStock(),
                    'seoUrl'    => $seoUrlsMap[$variant->getId()] ?? '/detail/' . $variant->getId(),
                ];
            }
        }

        return $groupedVariants;
    }

    /**
     * 09/2024 created
     */
    private function fetchSeoUrls(array $productIds, Context $context): array
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsAnyFilter('foreignKey', $productIds));
        $criteria->addFilter(new EqualsFilter('routeName', 'frontend.detail.page'));

        $seoUrlEntities = $this->seoUrlRepository->search($criteria, $context)->getEntities();

        $seoUrls = [];
        foreach ($seoUrlEntities as $seoUrlEntity) {
            $seoUrls[$seoUrlEntity->getForeignKey()] = '/' . $seoUrlEntity->getSeoPathInfo();
        }

        return $seoUrls;
    }
}
