export default class SimpleProductSlider {
    constructor(element, options = {}) {
        this.element = element;
        this.options = {
            sliderContainerSelector: '.product-image-slider',
            slideSelector:           '.product-image-slide',
            prevButtonSelector:      '.slider-prev',
            nextButtonSelector:      '.slider-next',
            miniThumbnailSelector:   '.mini-thumbnail',
            miniThumbnailsSelector:  '.mini-thumbnails',
            activeClass:             'active',
            visibleThumbnails:       3,
            ...options
        };
        this.currentIndex = 0;
        this.thumbnailIndex = 0;
        this.initializeSlider();
    }

    initializeSlider() {
        const container = this.element.querySelector(this.options.sliderContainerSelector);
        if (!container) return;

        this.slides = container.querySelectorAll(this.options.slideSelector);
        this.prevButton = container.querySelector(this.options.prevButtonSelector);
        this.nextButton = container.querySelector(this.options.nextButtonSelector);
        this.miniThumbnails = container.querySelectorAll(this.options.miniThumbnailSelector);
        this.miniThumbnailsContainer = container.querySelector(this.options.miniThumbnailsSelector);

        if (this.slides.length === 0) return;

        this.setupEventListeners();
        this.showSlide(this.currentIndex);
        this.updateThumbnails();
    }

    setupEventListeners() {
        if (this.prevButton) {
            this.prevButton.addEventListener('click', () => this.prevSlide());
        }
        if (this.nextButton) {
            this.nextButton.addEventListener('click', () => this.nextSlide());
        }

        this.miniThumbnails.forEach((thumbnail, index) => {
            thumbnail.addEventListener('click', () => {
                this.currentIndex = index;
                this.showSlide(this.currentIndex);
                this.updateThumbnails();
            });
        });
    }

    showSlide(index) {
        this.slides.forEach((slide, i) => {
            if (i === index) {
                slide.classList.add(this.options.activeClass);
                if (this.miniThumbnails[i]) {
                    this.miniThumbnails[i].classList.add(this.options.activeClass);
                }
            } else {
                slide.classList.remove(this.options.activeClass);
                if (this.miniThumbnails[i]) {
                    this.miniThumbnails[i].classList.remove(this.options.activeClass);
                }
            }
        });
    }

    nextSlide() {
        this.currentIndex = (this.currentIndex + 1) % this.slides.length;
        this.showSlide(this.currentIndex);
        this.updateThumbnails();
    }

    prevSlide() {
        this.currentIndex = (this.currentIndex - 1 + this.slides.length) % this.slides.length;
        this.showSlide(this.currentIndex);
        this.updateThumbnails();
    }

    updateThumbnails() {
        const totalThumbnails = this.miniThumbnails.length;
        if (totalThumbnails <= this.options.visibleThumbnails) return;

        // Ensure the current thumbnail is visible
        if (this.currentIndex < this.thumbnailIndex) {
            this.thumbnailIndex = this.currentIndex;
        } else if (this.currentIndex >= this.thumbnailIndex + this.options.visibleThumbnails) {
            this.thumbnailIndex = this.currentIndex - this.options.visibleThumbnails + 1;
        }

        // Clamp thumbnailIndex to valid range
        this.thumbnailIndex = Math.max(0, Math.min(this.thumbnailIndex, totalThumbnails - this.options.visibleThumbnails));

        // Update thumbnail container position
        const thumbnailWidth = this.miniThumbnails[0].offsetWidth;
        const offset = -this.thumbnailIndex * thumbnailWidth;
        this.miniThumbnailsContainer.style.transform = `translateX(${offset}px)`;
    }

    static init(element) {
        return new SimpleProductSlider(element);
    }
}

// Make SimpleProductSlider available globally
window.SimpleProductSlider = SimpleProductSlider;