<?php

namespace Topdata\TopdataVariantsInProductBoxesSW6\Service;

use Shopware\Core\Content\Product\Exception\ProductNotFoundException;
use Shopware\Core\Content\Product\ProductEntity;
use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Content\Property\Aggregate\PropertyGroupOption\PropertyGroupOptionEntity;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityCollection;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsAnyFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Struct\ArrayStruct;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;

/**
 * 09/2024 created
 */
class ProductVariantsFetcher
{
    const MAX_IMAGES_PER_PRODUCT = 99;


    private SalesChannelRepository $productRepository;
    private EntityRepository $seoUrlRepository;
    private ProductMediaFetcher $productMediaFetcher;

    public function __construct(
        SalesChannelRepository $productRepository,
        EntityRepository       $seoUrlRepository,
        ProductMediaFetcher    $productMediaFetcher
    )
    {
        $this->productRepository = $productRepository;
        $this->seoUrlRepository = $seoUrlRepository;
        $this->productMediaFetcher = $productMediaFetcher;
    }

    /**
     * Load multiple products and enrich them with their variants and media.
     *
     * @param SalesChannelProductEntity[] $products Array of product entities to be enriched.
     */
    public function loadMany(array $products, SalesChannelContext $salesChannelContext)
    {
        $context = $salesChannelContext->getContext();

        // ---- fetch media map (we need for image sliders)
        $productIds = array_map(function (SalesChannelProductEntity $product) {
            return $product->getId();
        }, $products);
        $mediaMap = $this->productMediaFetcher->fetchMediaOfMany($productIds, $context, self::MAX_IMAGES_PER_PRODUCT);

        // ---- add variants and media to products
        foreach ($products as $product) {
            $hasVariants = !empty($product->getParentId()) || $product->getChildCount() > 0;
            $groupedVariants = [];

            if ($product->getChildCount() > 0) {
                // Fetch and group variants if the product has children
                $groupedVariants = $this->fetchAndGroupProductVariants($product->getId(), $salesChannelContext);
            } elseif ($product->getParentId() !== null) {
                // Fetch and group variants if the product is a child
                $groupedVariants = $this->fetchAndGroupProductVariants($product->getParentId(), $salesChannelContext);
            }

//            dd("dskjfhdskjfhkjdshf kjdshfkjsd hfkjdshf ksdhf", $products);

            // Add custom extension with variants and media information
            $product->addExtension('topdataVariantsInProductBoxesSW6', new ArrayStruct([
                'hasVariants'     => $hasVariants,
                'childCount'      => $product->getChildCount(),
                'parentId'        => $product->getParentId(),
                'groupedVariants' => $groupedVariants,
                'media'           => $mediaMap[$product->getId()] ?? [],
            ]));
        }
    }

    /**
     * 09/2024 created
     */
    public function fetchAndGroupProductVariants(string $parentId, SalesChannelContext $salesChannelContext)
    {
        // ---- fetch variants
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('parentId', $parentId));
        $criteria->addAssociation('options.group');
        $criteria->addAssociation('prices');
        $criteria->addAssociation('media');
        $variants = $this->productRepository->search($criteria, $salesChannelContext)->getEntities();

        // ---- fetch seo urls for the variants map, format: {variantId: seoUrl}
        $seoUrls = $this->fetchSeoUrls($variants->getIds(), $salesChannelContext->getContext());

        return $this->groupVariants($variants, $seoUrls);
    }

    /**
     * Fetch a single product and its grouped variants
     */
    public function fetchProductWithGroupedVariants(string $productId, SalesChannelContext $salesChannelContext)
    {
        $context = $salesChannelContext->getContext();

        $criteria = new Criteria([$productId]);
        $criteria->setLimit(1);
//        $criteria->addAssociation('prices');
//        $criteria->addAssociation('media');
//        $criteria->addAssociation('options.group');

// FIXME: infinite recursion here because of MyEventSubscriber:
        $product = $this->productRepository->search($criteria, $salesChannelContext)->first();

        if (!$product) {
            throw new ProductNotFoundException($productId);
        }

        $groupedVariants = [];
        if ($product->getChildCount() > 0) {
            $groupedVariants = $this->fetchAndGroupProductVariants($product->getId(), $salesChannelContext);
        } elseif ($product->getParentId() !== null) {
            $groupedVariants = $this->fetchAndGroupProductVariants($product->getParentId(), $salesChannelContext);
            // In this case, also fetch the parent product
            $parentProduct = $this->fetchProductWithGroupedVariants($product->getParentId(), $salesChannelContext);
            $product->addExtension('parent', new ArrayStruct(['product' => $parentProduct]));
        }

        $product->addExtension('topdataVariantsInProductBoxesSW6', new ArrayStruct([
            'hasVariants'     => !empty($groupedVariants),
            'childCount'      => $product->getChildCount(),
            'parentId'        => $product->getParentId(),
            'groupedVariants' => $groupedVariants,
            'media'           => $this->productMediaFetcher->fetchMediaOfOne($productId, $context, self::MAX_IMAGES_PER_PRODUCT),
        ]));

        return $product;
    }

    /**
     * Group variants by their property group options
     */
    private function groupVariants(EntityCollection $variants, array $seoUrlsMap): array
    {
        $groupedVariants = [];

        /** @var ProductEntity $variant */
        foreach ($variants as $variant) {
            $options = $variant->getOptions();
            if (!$options) {
                continue;
            }

            /** @var PropertyGroupOptionEntity $option */
            foreach ($options as $option) {
                $groupId = $option->getGroupId();
                $groupName = $option->getGroup() ? $option->getGroup()->getName() : 'Unknown Group';
                $optionId = $option->getId();
                $optionName = $option->getName();

                if (!isset($groupedVariants[$groupId])) {
                    $groupedVariants[$groupId] = [
                        'groupName' => $groupName,
                        'options'   => [],
                    ];
                }

                if (!isset($groupedVariants[$groupId]['options'][$optionId])) {
                    $groupedVariants[$groupId]['options'][$optionId] = [
                        'optionName' => $optionName,
                        'variants'   => [],
                    ];
                }

                $groupedVariants[$groupId]['options'][$optionId]['variants'][] = [
                    'id'        => $variant->getId(),
                    'available' => $variant->getAvailable(),
                    'stock'     => $variant->getStock(),
                    'seoUrl'    => $seoUrlsMap[$variant->getId()] ?? '/detail/' . $variant->getId(),
                ];
            }
        }

        return $groupedVariants;
    }


    /**
     * 09/2024 created
     */
    private function fetchSeoUrls(array $productIds, Context $context): array
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsAnyFilter('foreignKey', $productIds));
        $criteria->addFilter(new EqualsFilter('routeName', 'frontend.detail.page'));

        $seoUrlEntities = $this->seoUrlRepository->search($criteria, $context)->getEntities();

        $seoUrls = [];
        foreach ($seoUrlEntities as $seoUrlEntity) {
            $seoUrls[$seoUrlEntity->getForeignKey()] = '/' . $seoUrlEntity->getSeoPathInfo();
        }

        return $seoUrls;
    }

}