<?php

namespace Topdata\TopdataVariantsInProductBoxesSW6\Service;

use Shopware\Core\Content\Product\Aggregate\ProductMedia\ProductMediaEntity;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsAnyFilter;

/**
 * Class ProductMediaFetcher
 *
 * This service class is responsible for fetching media entities associated with products.
 */
class ProductMediaFetcher
{
    /**
     * @var EntityRepository
     */
    private EntityRepository $productMediaRepository;

    /**
     * ProductMediaFetcher constructor.
     *
     * @param EntityRepository $productMediaRepository The repository to fetch product media entities.
     */
    public function __construct(EntityRepository $productMediaRepository)
    {
        $this->productMediaRepository = $productMediaRepository;
    }

    /**
     * Fetches media entities for multiple products.
     *
     * This method retrieves media entities associated with the given product IDs.
     * It returns a map where each key is a product ID and the value is an array of media entities.
     *
     * @param string[] $productIds An array of product IDs to fetch media for.
     * @param Context $context The context of the current request.
     * @param int|null $maxPerProduct Optional. The maximum number of media entities per product. If null, all media entities are returned.
     * @return array A map with format: [productId => MediaEntity[]]
     */
    public function fetchMediaOfMany(array $productIds, Context $context, ?int $maxPerProduct = null): array
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsAnyFilter('productId', $productIds));
        $criteria->addAssociation('media');

        /** @var ProductMediaEntity[] $productMediaCollection */
        $productMediaCollection = $this->productMediaRepository->search($criteria, $context)->getEntities();

        $result = [];
        foreach ($productMediaCollection as $productMedia) {
            $productId = $productMedia->getProductId();
            if (!isset($result[$productId])) {
                $result[$productId] = [];
            }
            if($maxPerProduct !== null && count($result[$productId]) >= $maxPerProduct) {
                continue; // the map entry for this product is already full
            }
            $result[$productId][] = $productMedia->getMedia();
        }

        return $result;
    }

    /**
     * Fetches media entities for a single product.
     *
     * This method retrieves media entities associated with the given product ID.
     * It returns an array of media entities.
     *
     * @param string $productId The product ID to fetch media for.
     * @param Context $context The context of the current request.
     * @param int $max The maximum number of media entities to fetch.
     * @return array An array of media entities.
     */
    public function fetchMediaOfOne(string $productId, Context $context, int $max): array
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsAnyFilter('productId', [$productId]));
        $criteria->addAssociation('media');
        $criteria->setLimit($max);

        /** @var ProductMediaEntity[] $productMediaCollection */
        $productMediaCollection = $this->productMediaRepository->search($criteria, $context)->getEntities();

        $result = [];
        foreach ($productMediaCollection as $productMedia) {
            $result[] = $productMedia->getMedia();
        }

        return $result;
    }
}