/**
 * VariantSelector class handles the variant selection functionality.
 * It uses event delegation to manage click events on variant selector buttons.
 */
export default class VariantSelector {
    constructor() {
        this.addEventListeners();
    }

    /**
     * Adds event listeners to the document for handling variant selection.
     * Event delegation is used to attach a single event listener to the parent element.
     */
    addEventListeners() {
        document.addEventListener('click', event => {
            if (event.target.matches('.variant-selector')) {
                this.handleVariantSelection(event);
            }
            if (event.target.matches('.topdata-variants-in-product-boxes__show-more')) {
                this.handleShowMore(event);
            }
        });
    }

    /**
     * Handles the variant selection event.
     * Fetches the variant data from the server and replaces the product box.
     *
     * @param {Event} event - The click event triggered by selecting a variant.
     */
    handleVariantSelection(event) {
        const button = event.target;
        const url = button.dataset.variantUrl;
        const productBox = button.closest('.card.product-box');

        if (url && productBox) {
            fetch(url, { method: 'GET' })
                .then(response => response.text())
                .then(response => {
                    const tempContainer = document.createElement('div');
                    tempContainer.innerHTML = response;
                    const newProductBox = tempContainer.firstElementChild;

                    if (newProductBox) {
                        productBox.replaceWith(newProductBox);
                        SimpleProductSlider.init(newProductBox);
                        // Reinitialize PayPal buttons in the new product box
                        window.PayPalReinitializer.init(newProductBox);
                    } else {
                        console.error('New product box element not found in the response');
                    }
                })
                .catch(error => {
                    console.error('Topdata Variants Ajax Error:', error);
                });
        }
    }

    /** @var PointerEvent event */
    handleShowMore(event) {
        if (event.target) {
            // ---- change display from none to block for all variants switch buttons
            const cls = event.target.getAttribute("data-css-class-to-show");
            document.querySelectorAll(`.${cls}`).forEach(el => el.style.display = "block");
            // ---- hide the "show more" button
            event.target.style.display = "none";
        }
    }

}
