<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\Popup;

use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataTopFinderProSW6\Service\TopdataBrandService;
use Topdata\TopdataTopFinderProSW6\Service\TopdataDeviceDataService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\List\ListPage;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\List\TRASH____ListPageLoader;

/**
 * Class PopupBrandsPageLoader
 *
 * This class is responsible for loading the popup brands page.
 * It retrieves all enabled brands and prepares the page for display in a popup.
 */
class PopupBrandsPageLoader
{

    public function __construct(
        private readonly GenericPageLoader        $genericPageLoader,
        private readonly TopdataDeviceDataService $deviceDataService,
        private readonly SettingsService          $settingsService,
        private readonly Translator               $translator,
        private readonly UrlGeneratorInterface    $router,
        private readonly LoggerInterface          $logger,
        private readonly TopdataBrandService         $topdataBrandService,
    )
    {
    }

    /**
     * Load the popup brands page
     *
     * This method loads the generic page, retrieves all enabled brands,
     * and configures the page for display in a popup.
     *
     * @param mixed $request The current request
     * @param mixed $salesChannelContext The sales channel context
     * @return ListPage The loaded popup brands page
     */
    public function load($request, $salesChannelContext): ListPage
    {
        try {
            // Load the generic page
            $page = $this->genericPageLoader->load($request, $salesChannelContext);

            // Create the list page from the generic page
            $listPage = new ListPage();
            // Copy all properties from the generic page to the list page
            foreach (get_object_vars($page) as $property => $value) {
                $listPage->$property = $value;
            }

            // Set the page type to brands
            $listPage->setPageType(TRASH____ListPageLoader::TYPE_BRANDS);

            // Set the page to show brands
            $listPage->setShow(['brands' => true]);

            // Retrieve all enabled brands from the brand data service
            $brands = $this->topdataBrandService->getAllEnabledBrands();

            // Set the retrieved brands to the page
            $listPage->setBrands($brands);

            // Set the title for the popup brands page
            $listPage->setTitle($this->translator->trans('topdata-topfinder.popup.allBrands'));

            // Set the popup path
            $listPage->popupPath[] = [
                'name' => $this->translator->trans('topdata-topfinder.popup.allBrands'),
                'path' => $this->router->generate('frontend.top-finder-api.popup-all-brands')
            ];

            return $listPage;
        } catch (\Exception $e) {
            $this->logger->error('Error loading popup brands page: ' . $e->getMessage());
            throw $e;
        }
    }
}