<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\List;

use Exception;
use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataTopFinderProSW6\Component\BreadcrumbsTrait;
use Topdata\TopdataTopFinderProSW6\Service\TopdataBrandService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\List\ListPage;

/**
 * This class is responsible for loading the brands list page.
 * It retrieves all enabled brands and prepares the page with appropriate metadata and breadcrumbs.
 */
class BrandsListPageLoader
{
    use BreadcrumbsTrait;

    public function __construct(
        private GenericPageLoader     $genericPageLoader,
        private TopdataBrandService      $topdataBrandService,
        private SettingsService       $settingsService,
        private Translator            $translator,
        private UrlGeneratorInterface $router,
        private LoggerInterface       $logger
    )
    {
    }

    /**
     * Load the brands list page
     *
     * This method loads the generic page, retrieves all enabled brands,
     * and configures the page with appropriate metadata and breadcrumbs.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return ListPage The loaded brands list page
     */
    /**
     * Load the brands list page
     *
     * This method loads the generic page, retrieves all enabled brands,
     * and configures the page with appropriate metadata and breadcrumbs.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return ListPage The loaded brands list page
     */
    public function load($request, $salesChannelContext): ListPage
    {
        try {
            // Load the generic page
            $page = $this->genericPageLoader->load($request, $salesChannelContext);

            // Create the list page from the generic page
            $listPage = new ListPage();
            // Copy all properties from the generic page to the list page
            foreach (get_object_vars($page) as $property => $value) {
                $listPage->$property = $value;
            }

            // Set the page type to brands
            $listPage->setPageType(TRASH____ListPageLoader::TYPE_BRANDS);

            // Set the page to show brands
            $listPage->setShow(['brands' => true]);

            // Retrieve all enabled brands from the brand data service
            $brands = $this->topdataBrandService->getAllEnabledBrands();

            // Set the retrieved brands to the page
            $listPage->setBrands($brands);

            // Set the SEO title for the brands page
            $listPage->setTitle($this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'));

            // For now, we'll skip setting meta information since we're having issues with the method
            // In a real implementation, this would need to be fixed by understanding the Shopware Page structure
            // The original code uses $this->listPage->getMetaInformation() which suggests it's a method on the Page class

            // Log a warning about this issue
            $this->logger->warning('Meta information could not be set in BrandsListPageLoader. This needs to be fixed.');

            // Add breadcrumbs for the brands page
            $listPage->addBreadcrumb(
                $listPage->getTitle(),
                $this->router->generate('frontend.top-finder.brands')
            );

            // Apply compact mode limit if configured
            if ($this->settingsService->getInt('searchCompactLimit', true)) {
                $listPage->setCompactModeLimit($this->settingsService->getInt('searchCompactLimit'));
            }

            return $listPage;
        } catch (Exception $e) {
            $this->logger->error('Error loading brands list page: ' . $e->getMessage());
            throw $e;
        }
    }
}