<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Service;

use Doctrine\DBAL\Connection;
use Psr\Log\LoggerInterface;

/**
 * Service responsible for counting devices and related entities.
 * Extracted from OneBrandLetterPageLoader to improve separation of concerns.
 */
class DeviceCountService
{
    public function __construct(
        private readonly Connection $connection,
        private readonly LoggerInterface $logger
    ) {
    }

    /**
     * Gets the total count of enabled devices for a specific brand.
     *
     * @param string $brandCode The code of the brand
     * @return int The total device count
     */
    public function getTotalDeviceCount(string $brandCode): int
    {
        $count = $this->connection->fetchOne(
            'SELECT COUNT(*)
             FROM topdata_device
             INNER JOIN topdata_brand ON topdata_brand.id = topdata_device.brand_id
             WHERE topdata_brand.code = :brandCode
             AND topdata_device.is_enabled = 1',
            ['brandCode' => $brandCode]
        );

        return (int) $count;
    }

    /**
     * Gets the count of enabled devices grouped by series for a specific brand.
     *
     * @param string $brandCode The code of the brand
     * @return array Associative array with series names as keys and counts as values
     */
    public function getDeviceCountBySeries(string $brandCode): array
    {
        $results = $this->connection->fetchAllAssociative(
            'SELECT s.label as series_name, COUNT(d.id) as device_count
             FROM topdata_device d
             INNER JOIN topdata_brand b ON b.id = d.brand_id
             LEFT JOIN topdata_series s ON s.id = d.series_id
             WHERE b.code = :brandCode
             AND d.is_enabled = 1
             AND s.is_enabled = 1
             GROUP BY s.label',
            ['brandCode' => $brandCode]
        );

        $counts = [];
        foreach ($results as $row) {
            if ($row['series_name']) {
                $counts[$row['series_name']] = (int) $row['device_count'];
            }
        }

        return $counts;
    }

    /**
     * Gets the count of enabled devices grouped by type for a specific brand.
     *
     * @param string $brandCode The code of the brand
     * @return array Associative array with type names as keys and counts as values
     */
    public function getDeviceCountByType(string $brandCode): array
    {
        $results = $this->connection->fetchAllAssociative(
            'SELECT t.label as type_name, COUNT(d.id) as device_count
             FROM topdata_device d
             INNER JOIN topdata_brand b ON b.id = d.brand_id
             LEFT JOIN topdata_device_type t ON t.id = d.type_id
             WHERE b.code = :brandCode
             AND d.is_enabled = 1
             AND t.is_enabled = 1
             GROUP BY t.label',
            ['brandCode' => $brandCode]
        );

        $counts = [];
        foreach ($results as $row) {
            if ($row['type_name']) {
                $counts[$row['type_name']] = (int) $row['device_count'];
            }
        }

        return $counts;
    }

    /**
     * Gets all device counts for a brand in a single method call.
     *
     * @param string $brandCode The code of the brand
     * @return array Associative array with all counts
     */
    public function getAllDeviceCounts(string $brandCode): array
    {
        $startTime = microtime(true);
        
        $counts = [
            'total' => $this->getTotalDeviceCount($brandCode),
            'bySeries' => $this->getDeviceCountBySeries($brandCode),
            'byType' => $this->getDeviceCountByType($brandCode),
        ];

        $this->logger->info('DeviceCountService::getAllDeviceCounts - Completed', [
            'brandCode' => $brandCode,
            'totalDevices' => $counts['total'],
            'seriesWithDevices' => count($counts['bySeries']),
            'typesWithDevices' => count($counts['byType']),
            'duration_ms' => (microtime(true) - $startTime) * 1000
        ]);

        return $counts;
    }
}