<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Controller;

use Doctrine\DBAL\Connection;
use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Controller\StorefrontController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Throwable;
use Topdata\TopdataTopFinderProSW6\Service\TopdataBrandService;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\OneBrandLetterPageLoader;
use Topdata\TopdataTopFinderProSW6\Util\ControllerUtil;

/**
 * Controller for handling TopFinder device-related API requests.
 *
 * This controller provides endpoints for retrieving devices, series, and types
 * for a specific brand.
 *
 * 04/2025 created (extracted from TopFinderApiController)
 */
#[Route(defaults: ['_routeScope' => ['storefront']])]
class ApiController_TopdataDevice extends StorefrontController
{
    private const VIEW_PATHS = [
        'devices' => '@Storefront/storefront/page/topfinder/one-brand-letter-devices.html.twig',
        'series'  => '@Storefront/storefront/page/topfinder/one-brand-letter-series.html.twig',
        'types'   => '@Storefront/storefront/page/topfinder/one-brand-letter-types.html.twig',
    ];

    private const CACHE_DURATION    = 3600; // 1 hour
    private const ALLOWED_GROUPINGS = [  // TODO uppercase and use of a constants class
        'none',
        'series',
        'type',
    ];

    public function __construct(
        private readonly OneBrandLetterPageLoader $oneBrandLetterPageLoader,
        private readonly LoggerInterface          $logger,
        private readonly Connection               $connection,
        private readonly EntityRepository         $mediaRepository,
        private readonly TopdataBrandService         $topdataBrandService,
    )
    {
    }

    /**
     * List devices for a specific brand.
     *
     * @param string $brandCode The code of the brand
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response containing the HTML for the brand devices list
     */
    #[Route(
        path: '/top-finder-api/brand/{brandCode}/devices',
        name: 'frontend.top-finder-api.brand-devices-list',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET', 'POST']
    )]
    public function listDevices(string $brandCode, Request $request, SalesChannelContext $context): JsonResponse
    {
        return $this->listEntities('devices', $brandCode, $request, $context);
    }

    /**
     * List series for a specific brand.
     *
     * @param string $brandCode The code of the brand
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response containing the HTML for the brand series list
     */
    #[Route(
        path: '/top-finder-api/brand/{brandCode}/series',
        name: 'frontend.top-finder-api.brand-series-list',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET', 'POST']
    )]
    public function listSeries(string $brandCode, Request $request, SalesChannelContext $context): JsonResponse
    {
        return $this->listEntities('series', $brandCode, $request, $context);
    }

    /**
     * List types for a specific brand.
     *
     * @param string $brandCode The code of the brand
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response containing the HTML for the brand types list
     */
    #[Route(
        path: '/top-finder-api/brand/{brandCode}/types',
        name: 'frontend.top-finder-api.brand-types-list',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET', 'POST']
    )]
    public function listTypes(string $brandCode, Request $request, SalesChannelContext $context): JsonResponse
    {
        return $this->listEntities('types', $brandCode, $request, $context);
    }

    /**
     * Load more devices for a specific brand with AJAX pagination.
     *
     * @param string $brandCode The code of the brand
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response containing the HTML for new devices and pagination info
     */
    #[Route(
        path: '/top-finder-api/brand/{brandCode}/load-more',
        name: 'frontend.top-finder-api.brand-load-more',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function loadMore(string $brandCode, Request $request, SalesChannelContext $context): JsonResponse
    {
        ControllerUtil::assertValidBrandcode($brandCode);

        // Get pagination parameters
        $offset = max(0, (int)$request->get('offset', 0));
        $limit = max(1, (int)$request->get('limit', 20));
        $grouping = $request->get('grouping', '');
        $groupValue = $request->get('groupValue', '');

        // Validate grouping parameter
        if (!empty($grouping) && !in_array($grouping, self::ALLOWED_GROUPINGS)) {
            return new JsonResponse([
                'success' => false,
                'error'   => 'Invalid grouping parameter, expected one of: ' . implode(', ', self::ALLOWED_GROUPINGS) . ' got: ' . $grouping
            ], 400);
        }

        // Get brand information
        $brand = $this->topdataBrandService->getTopdataBrand($brandCode);
        if (!$brand) {
            return new JsonResponse([
                'success' => false,
                'error'   => 'Brand not found'
            ], 404);
        }

        // Get series and types for filtering
        $seriesData = $this->getSeries();
        $typesData = $this->getTypes();

        // Build and execute query based on grouping
        $devices = $this->getDevicesForBrandWithGrouping(
            $brandCode,
            $offset,
            $limit,
            $grouping,
            $groupValue
        );

        if (empty($devices)) {
            return new JsonResponse([
                'success'   => true,
                'html'      => '',
                'hasMore'   => false,
                'newOffset' => $offset
            ]);
        }

        // Process devices with series and type information
        $processedDevices = $this->processDevices($devices, $seriesData, $typesData, $context);

        // Get total count for remaining calculation
        $totalCount = $this->getTotalDeviceCountWithGrouping($brandCode, $grouping, $groupValue);
        $loadedCount = count($processedDevices);
        $newOffset = $offset + $loadedCount;
        $hasMore = $newOffset < $totalCount;

        // Render HTML fragment
        $html = $this->renderStorefront('@Storefront/storefront/page/topfinder/one-brand-letter-devices.html.twig', [
            'devices' => $processedDevices
        ])->getContent();

        return new JsonResponse([
            'success'   => true,
            'html'      => $html,
            'hasMore'   => $hasMore,
            'newOffset' => $newOffset
        ]);
    }

    /**
     * Generic method to list entities for a brand.
     *
     * @param string $entityType The type of entity (devices, series, types)
     * @param string $brandCode The code of the brand
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response containing the HTML for the entities list
     */
    private function listEntities(string $entityType, string $brandCode, Request $request, SalesChannelContext $context): JsonResponse
    {
        ControllerUtil::assertValidBrandcode($brandCode);

        // Get view path from the map
        $view = self::VIEW_PATHS[$entityType] ?? throw new \InvalidArgumentException("Invalid entity type: {$entityType}");

        // Load page data
        $page = $this->_loadPageData($brandCode, $request, $context);

        // Render view
        $html = $this->_renderEntityView($view, $page);

        // Prepare response
        return $this->_createCachedResponse(['success' => true, 'html' => $html]);
    }

    /**
     * Load page data using the page loader.
     *
     * @param string $brandCode The code of the brand
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return mixed The loaded page object
     * @throws Throwable If loading fails
     */
    private function _loadPageData(string $brandCode, Request $request, SalesChannelContext $context): mixed
    {
        try {
            return $this->oneBrandLetterPageLoader->load($request, $context, $brandCode);
        } catch (Throwable $e) {
            $this->logger->error('Failed to load page data', [
                'brandCode' => $brandCode,
                'error'     => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Render the entity view with the provided page data.
     *
     * Note: Named differently from parent class's protected renderView method
     *
     * @param string $view The view path
     * @param mixed $page The page data
     * @return string The rendered HTML
     * @throws Throwable If rendering fails
     */
    private function _renderEntityView(string $view, mixed $page): string
    {
        return $this->renderStorefront($view, ['page' => $page])->getContent();
    }

    /**
     * Create a cached JSON response.
     *
     * @param array $data The response data
     * @return JsonResponse The JSON response with cache headers
     */
    private function _createCachedResponse(array $data): JsonResponse
    {
        $response = new JsonResponse($data);
        $response->setPublic();
        $response->setMaxAge(self::CACHE_DURATION);
        $response->setSharedMaxAge(self::CACHE_DURATION);

        return $response;
    }


    /**
     * Get all enabled series.
     *
     * @return array Array of series data
     */
    private function getSeries(): array
    {
        $series = $this->connection->fetchAllAssociative('
            SELECT label as name,
                   LOWER(HEX(id)) as id,
                   code as code
            FROM topdata_series
            WHERE is_enabled = 1
            ORDER BY label
        ');

        $result = [];
        foreach ($series as $serie) {
            $result[$serie['id']] = $serie;
        }

        return $result;
    }

    /**
     * Get all enabled device types.
     *
     * @return array Array of device type data
     */
    private function getTypes(): array
    {
        $types = $this->connection->fetchAllAssociative('
            SELECT label as name,
                   LOWER(HEX(id)) as id,
                   code as code
            FROM topdata_device_type
            WHERE is_enabled = 1
            ORDER BY label
        ');

        $result = [];
        foreach ($types as $type) {
            $result[$type['id']] = $type;
        }

        return $result;
    }

    /**
     * Get devices for a brand with pagination and filtering.
     *
     * @param string $brandCode The brand code
     * @param int $offset Query offset
     * @param int $limit Query limit
     * @param string $seriesFilter Series filter (optional)
     * @param string $typeFilter Type filter (optional)
     * @return array Array of device data
     */
    private function getDevicesForBrand(
        string $brandCode,
        int    $offset,
        int    $limit,
        string $seriesFilter = '',
        string $typeFilter = ''
    ): array
    {
        $params = ['brandCode' => $brandCode, 'offset' => $offset, 'limit' => $limit];

        $sql = '
            SELECT topdata_device.model as name,
                   LOWER(HEX(topdata_device.id)) as id,
                   LOWER(HEX(topdata_device.brand_id)) as brand_id,
                   LOWER(HEX(topdata_device.series_id)) as series_id,
                   LOWER(HEX(topdata_device.type_id)) as type_id,
                   topdata_device.code as code,
                   topdata_brand.label as brand_name,
                   topdata_brand.code as brand_code,
                   LOWER(HEX(topdata_device.media_id)) as media_id
            FROM topdata_brand
            JOIN topdata_device ON topdata_brand.id = topdata_device.brand_id
            WHERE topdata_brand.code = :brandCode
              AND topdata_device.is_enabled = 1
        ';

        // Add series filter if provided
        if (!empty($seriesFilter)) {
            $sql .= ' AND topdata_device.series_id IN (
                SELECT id FROM topdata_series WHERE label = :seriesFilter AND is_enabled = 1
            )';
            $params['seriesFilter'] = $seriesFilter;
        }

        // Add type filter if provided
        if (!empty($typeFilter)) {
            $sql .= ' AND topdata_device.type_id IN (
                SELECT id FROM topdata_device_type WHERE label = :typeFilter AND is_enabled = 1
            )';
            $params['typeFilter'] = $typeFilter;
        }

        $sql .= ' ORDER BY topdata_device.code ASC LIMIT :limit OFFSET :offset';

        return $this->connection->fetchAllAssociative($sql, $params);
    }

    /**
     * Get total device count for pagination.
     *
     * @param string $brandCode The brand code
     * @param string $seriesFilter Series filter (optional)
     * @param string $typeFilter Type filter (optional)
     * @return int Total device count
     */
    private function getTotalDeviceCount(
        string $brandCode,
        string $seriesFilter = '',
        string $typeFilter = ''
    ): int
    {
        $params = ['brandCode' => $brandCode];

        $sql = '
            SELECT COUNT(*) as total
            FROM topdata_brand
            JOIN topdata_device ON topdata_brand.id = topdata_device.brand_id
            WHERE topdata_brand.code = :brandCode
              AND topdata_device.is_enabled = 1
        ';

        // Add series filter if provided
        if (!empty($seriesFilter)) {
            $sql .= ' AND topdata_device.series_id IN (
                SELECT id FROM topdata_series WHERE label = :seriesFilter AND is_enabled = 1
            )';
            $params['seriesFilter'] = $seriesFilter;
        }

        // Add type filter if provided
        if (!empty($typeFilter)) {
            $sql .= ' AND topdata_device.type_id IN (
                SELECT id FROM topdata_device_type WHERE label = :typeFilter AND is_enabled = 1
            )';
            $params['typeFilter'] = $typeFilter;
        }

        $result = $this->connection->fetchAssociative($sql, $params);

        return (int)($result['total'] ?? 0);
    }

    /**
     * Get devices for a brand with grouping support.
     *
     * @param string $brandCode The brand code
     * @param int $offset Query offset
     * @param int $limit Query limit
     * @param string $grouping Grouping type (series|type)
     * @param string $groupValue Group value to filter by
     * @return array Array of device data
     */
    private function getDevicesForBrandWithGrouping(
        string $brandCode,
        int    $offset,
        int    $limit,
        string $grouping = '',
        string $groupValue = ''
    ): array
    {
        $params = ['brandCode' => $brandCode, 'offset' => $offset, 'limit' => $limit];

        $sql = '
            SELECT topdata_device.model as name,
                   LOWER(HEX(topdata_device.id)) as id,
                   LOWER(HEX(topdata_device.brand_id)) as brand_id,
                   LOWER(HEX(topdata_device.series_id)) as series_id,
                   LOWER(HEX(topdata_device.type_id)) as type_id,
                   topdata_device.code as code,
                   topdata_brand.label as brand_name,
                   topdata_brand.code as brand_code,
                   LOWER(HEX(topdata_device.media_id)) as media_id
            FROM topdata_brand
            JOIN topdata_device ON topdata_brand.id = topdata_device.brand_id
            WHERE topdata_brand.code = :brandCode
              AND topdata_device.is_enabled = 1
        ';

        // Add grouping filter if provided
        if (!empty($grouping) && !empty($groupValue)) {
            if ($grouping === 'series') {
                $sql .= ' AND topdata_device.series_id IN (
                    SELECT id FROM topdata_series WHERE label = :groupValue AND is_enabled = 1
                )';
                $params['groupValue'] = $groupValue;
            } elseif ($grouping === 'type') {
                $sql .= ' AND topdata_device.type_id IN (
                    SELECT id FROM topdata_device_type WHERE label = :groupValue AND is_enabled = 1
                )';
                $params['groupValue'] = $groupValue;
            }
        }

        $sql .= ' ORDER BY topdata_device.code ASC LIMIT :limit OFFSET :offset';

        $types = [
            'limit'  => \Doctrine\DBAL\ParameterType::INTEGER,
            'offset' => \Doctrine\DBAL\ParameterType::INTEGER,
        ];

        return $this->connection->fetchAllAssociative($sql, $params, $types);
    }

    /**
     * Get total device count with grouping support.
     *
     * @param string $brandCode The brand code
     * @param string $grouping Grouping type (series|type)
     * @param string $groupValue Group value to filter by
     * @return int Total device count
     */
    private function getTotalDeviceCountWithGrouping(
        string $brandCode,
        string $grouping = '',
        string $groupValue = ''
    ): int
    {
        $params = ['brandCode' => $brandCode];

        $sql = '
            SELECT COUNT(*) as total
            FROM topdata_brand
            JOIN topdata_device ON topdata_brand.id = topdata_device.brand_id
            WHERE topdata_brand.code = :brandCode
              AND topdata_device.is_enabled = 1
        ';

        // Add grouping filter if provided
        if (!empty($grouping) && !empty($groupValue)) {
            if ($grouping === 'series') {
                $sql .= ' AND topdata_device.series_id IN (
                    SELECT id FROM topdata_series WHERE label = :groupValue AND is_enabled = 1
                )';
                $params['groupValue'] = $groupValue;
            } elseif ($grouping === 'type') {
                $sql .= ' AND topdata_device.type_id IN (
                    SELECT id FROM topdata_device_type WHERE label = :groupValue AND is_enabled = 1
                )';
                $params['groupValue'] = $groupValue;
            }
        }

        $result = $this->connection->fetchAssociative($sql, $params);

        return (int)($result['total'] ?? 0);
    }

    /**
     * Process devices with series, type, and media information.
     *
     * @param array $devices Raw device data
     * @param array $seriesData Series information
     * @param array $typesData Types information
     * @param SalesChannelContext $context Sales channel context
     * @return array Processed device data
     */
    private function processDevices(
        array               $devices,
        array               $seriesData,
        array               $typesData,
        SalesChannelContext $context
    ): array
    {
        if (empty($devices)) {
            return [];
        }

        $mediaIds = [];
        $processedDevices = [];

        // Process each device
        foreach ($devices as $device) {
            // Add series information
            if ($device['series_id'] && isset($seriesData[$device['series_id']])) {
                $device['series_name'] = $seriesData[$device['series_id']]['name'];
                $device['series_code'] = $seriesData[$device['series_id']]['code'];
            } else {
                $device['series_name'] = '';
                $device['series_code'] = '';
            }

            // Add type information
            if ($device['type_id'] && isset($typesData[$device['type_id']])) {
                $device['type_name'] = $typesData[$device['type_id']]['name'];
                $device['type_code'] = $typesData[$device['type_id']]['code'];
            } else {
                $device['type_name'] = '';
                $device['type_code'] = '';
            }

            // Add brand entity structure for compatibility
            $device['brand'] = new \Shopware\Core\Framework\Struct\ArrayEntity([
                'name' => $device['brand_name']
            ]);

            if ($device['media_id']) {
                $mediaIds[] = $device['media_id'];
            }

            $processedDevices[] = $device;
        }

        // Fetch and attach media URLs
        if (!empty($mediaIds)) {
            $criteria = new \Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria($mediaIds);
            $medias = $this->mediaRepository->search($criteria, $context->getContext());

            foreach ($processedDevices as &$device) {
                if ($device['media_id'] && $medias->has($device['media_id'])) {
                    $device['media'] = $medias->get($device['media_id']);
                    $device['media_url'] = $medias->get($device['media_id'])->getUrl();
                } else {
                    $device['media'] = null;
                    $device['media_url'] = null;
                }
            }
        }

        return $processedDevices;
    }
}