<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader;

use Doctrine\DBAL\Connection;
use Psr\Log\LoggerInterface;
use Shopware\Core\Checkout\Customer\CustomerEntity;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\Struct\ArrayEntity;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataConnectorSW6\Core\Content\Device\Agregate\DeviceCustomer\DeviceCustomerEntity;
use Topdata\TopdataTopFinderProSW6\Service\DeviceToCustomerService;
use Topdata\TopdataTopFinderProSW6\Service\DeviceCountService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\OneBrandLetter\OneBrandLetterPage;


/**
 * This class is responsible for loading the data required for the OneBrandLetter page.
 * It fetches brand details, devices associated with the brand, and prepares the page for display.
 */
class OneBrandLetterPageLoader
{

    const DEFAULT_INITIAL_LOAD_LIMIT = 20;

    public function __construct(
        private readonly GenericPageLoader       $genericLoader,
        private readonly Connection              $connection,
        private readonly UrlGeneratorInterface   $router,
        private readonly Translator              $translator,
        private readonly EntityRepository        $mediaRepository,
        private readonly LoggerInterface         $logger,
        private readonly DeviceToCustomerService $deviceToCustomerService,
        private readonly SettingsService         $settingsService,
        private readonly DeviceCountService      $deviceCountService,
    )
    {
    }

    /**
     * Loads the OneBrandLetterPage with the necessary data.
     *
     * @param Request $request The current HTTP request.
     * @param SalesChannelContext $salesChannelContext The current sales channel context.
     * @param string $brandCode The code of the brand to load.
     * @return OneBrandLetterPage The populated OneBrandLetterPage.
     * @throws \Exception
     */
    public function load(Request $request, SalesChannelContext $salesChannelContext, string $brandCode): OneBrandLetterPage
    {
        // ---- Load generic page
        $page = $this->genericLoader->load($request, $salesChannelContext);
        $page = OneBrandLetterPage::createFrom($page);

        // ---- Set brand and device elements
        $this->setElements($page, $brandCode, $salesChannelContext);

        // ---- Set SEO information if not an AJAX request
        if ($request->isXmlHttpRequest() === false) {
            $page->setTitle($this->translator->trans('topdata-topfinder.SEO.brandPageTitle', [
                '%brand%' => $page->brand['name'],
            ]));

            $page->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.brandMetaTitle', [
                '%brand%' => $page->brand['name'],
            ]));

            $page->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.brandMetaDescription', [
                '%brand%' => $page->brand['name'],
            ]));

            $page->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.brandMetaRobots'));

            $page->addBreadcrumb(
                $this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'),
                $this->router->generate('frontend.top-finder.brands')
            );

            $page->addBreadcrumb(
                $page->getTitle(),
                $this->router->generate('frontend.top-finder.show-brand', ['brandCode' => $page->brand['code']])
            );
        }

        return $page;
    }

    /**
     * Sets the brand, devices, series, and types elements for the page.
     *
     * @param OneBrandLetterPage $page The page to set the elements on.
     * @param string $brandCode The code of the brand.
     * @throws \Exception
     */
    private function setElements(OneBrandLetterPage $page, string $brandCode, SalesChannelContext $salesChannelContext): void
    {
        $startTime = microtime(true);
        $this->logger->info('OneBrandLetterPageLoader::setElements - Start', ['brandCode' => $brandCode]);

        // ---- Get initial load limit from configuration
        $page->initialLoadLimit = $this->settingsService->getInt('brandPageInitialLoadLimit');
        if ($page->initialLoadLimit <= 0) {
            $page->initialLoadLimit = self::DEFAULT_INITIAL_LOAD_LIMIT;
        }

        // ---- Get brand information
        $page->brand = $this->getBrand($brandCode);
        if (!$page->brand) {
            throw new \Exception('Device brand not found!');
        }

        $this->logger->info('OneBrandLetterPageLoader::setElements - Brand loaded', ['brandCode' => $brandCode, 'duration_ms' => (microtime(true) - $startTime) * 1000]);

        // ---- Get series and types (needed for all views)
        $series = $this->getSeries();
        $types = $this->getTypes();

        // ---- Get total counts for each grouping using DeviceCountService
        $totalCountsStartTime = microtime(true);
        
        $deviceCounts = $this->deviceCountService->getAllDeviceCounts($brandCode);
        $page->devicesTotal = $deviceCounts['total'];
        $page->devicesBySeriesTotal = $deviceCounts['bySeries'];
        $page->devicesByTypeTotal = $deviceCounts['byType'];

        $this->logger->info('OneBrandLetterPageLoader::setElements - Total counts calculated', [
            'brandCode' => $brandCode,
            'totalDevices' => $page->devicesTotal,
            'seriesWithDevices' => count($page->devicesBySeriesTotal),
            'typesWithDevices' => count($page->devicesByTypeTotal),
            'duration_ms' => (microtime(true) - $totalCountsStartTime) * 1000
        ]);

        if ($page->devicesTotal === 0) {
            $this->logger->warning('OneBrandLetterPageLoader::setElements - No devices found for brand', ['brandCode' => $brandCode]);
            // Set empty arrays and return early if no devices found to avoid errors
            $page->series = [];
            $page->types = [];
            $page->devices = [];
            $page->seriesList = [];
            $page->typesList = [];
            $page->devicesBySeries = [];
            $page->devicesByType = [];
            $endTime = microtime(true);
            $this->logger->info('OneBrandLetterPageLoader::setElements - End (no devices)', ['brandCode' => $brandCode, 'total_duration_ms' => ($endTime - $startTime) * 1000]);

            return;
        }

        // ---- Get paginated devices for initial load (no grouping view)
        $devices = $this->getDevicesByBrandPaginated($brandCode, $page->initialLoadLimit);
        $page->devices = $devices;

        // ---- Fetch devices for series view
        $devicesBySeries = [];
        $seriesForBrand = $this->getSeriesForBrand($brandCode);
        foreach ($seriesForBrand as $seriesName => $seriesData) {
            $devicesForSeries = $this->getDevicesBySeriesPaginated($brandCode, $seriesData['id'], $page->initialLoadLimit);
            $devicesBySeries[$seriesName] = $devicesForSeries;
        }
        $page->devicesBySeries = $devicesBySeries;

        // ---- Fetch devices for type view
        $devicesByType = [];
        $typesForBrand = $this->getTypesForBrand($brandCode);
        foreach ($typesForBrand as $typeName => $typeData) {
            $devicesForType = $this->getDevicesByTypePaginated($brandCode, $typeData['id'], $page->initialLoadLimit);
            $devicesByType[$typeName] = $devicesForType;
        }
        $page->devicesByType = $devicesByType;

        // ---- Collect all unique device objects for media hydration
        $allDevices = array_merge($devices, ...array_values($devicesBySeries), ...array_values($devicesByType));
        $mediaIds = [];
        foreach ($allDevices as $device) {
            if (!empty($device['media_id'])) {
                $mediaIds[] = $device['media_id'];
            }
        }

        // ---- Create series and types lists with counts (using total counts)
        $seriesList = [];
        foreach (array_keys($page->devicesBySeriesTotal) as $seriesName) {
            $seriesList[] = [
                'name'  => $seriesName,
                'count' => $page->devicesBySeriesTotal[$seriesName]
            ];
        }

        $typesList = [];
        foreach (array_keys($page->devicesByTypeTotal) as $typeName) {
            $typesList[] = [
                'name'  => $typeName,
                'count' => $page->devicesByTypeTotal[$typeName]
            ];
        }

        // ---- Assign data to the page
        // Fetch media entities and merge them into all device arrays
        if (!empty($mediaIds)) {
            $medias = $this->mediaRepository->search(new Criteria($mediaIds), $salesChannelContext->getContext());

            // Update devices array
            foreach ($devices as &$device) {
                if (!empty($device['media_id']) && $medias->has($device['media_id'])) {
                    $device['media'] = $medias->get($device['media_id']);
                } else {
                    $device['media'] = null;
                }
            }
            unset($device); // break reference

            // Update devicesBySeries arrays
            foreach ($devicesBySeries as &$seriesDevices) {
                foreach ($seriesDevices as &$device) {
                    if (!empty($device['media_id']) && $medias->has($device['media_id'])) {
                        $device['media'] = $medias->get($device['media_id']);
                    } else {
                        $device['media'] = null;
                    }
                }
            }
            unset($device); // break reference

            // Update devicesByType arrays
            foreach ($devicesByType as &$typeDevices) {
                foreach ($typeDevices as &$device) {
                    if (!empty($device['media_id']) && $medias->has($device['media_id'])) {
                        $device['media'] = $medias->get($device['media_id']);
                    } else {
                        $device['media'] = null;
                    }
                }
            }
            unset($device); // break reference
        }

        $page->series = array_keys($page->devicesBySeriesTotal);
        $page->types = array_keys($page->devicesByTypeTotal);
        $page->seriesList = $seriesList;
        $page->typesList = $typesList;

        $endTime = microtime(true);
        $this->logger->info('OneBrandLetterPageLoader::setElements - End', [
            'brandCode'         => $brandCode,
            'totalDevices'      => $page->devicesTotal,
            'displayedDevices'  => count($page->devices),
            'initialLoadLimit'  => $page->initialLoadLimit,
            'total_duration_ms' => ($endTime - $startTime) * 1000
        ]);
    }

    /**
     * Retrieves all enabled series from the database.
     *
     * @return array An array of series data.
     */
    private function getSeries(): array
    {
        $mediaIds = [];
        $processedDevices = [];

        // ---- Fetch series data
        $series = $this->connection->fetchAllAssociative('
SELECT label as name, 
       LOWER(HEX(id)) as id, 
       code as code
  FROM topdata_series
  WHERE is_enabled=1
  ORDER BY label
        ');

        // ---- Format series data
        foreach ($series as $serie) {
            $return[$serie['id']] = $serie;
        }

        return $return;
    }

    /**
     * Retrieves all enabled device types from the database.
     *
     * @return array An array of device type data.
     */
    private function getTypes(): array
    {
        $mediaIds = [];
        $processedDevices = [];

        // ---- Fetch device type data
        $types = $this->connection->fetchAllAssociative('
SELECT label as name,
       LOWER(HEX(id)) as id,
       code as code
  FROM topdata_device_type
  WHERE is_enabled=1
  ORDER BY label
        ');

        // ---- Format device type data
        foreach ($types as $type) {
            $return[$type['id']] = $type;
        }

        return $return;
    }

    /**
     * Retrieves all enabled series that have devices for a specific brand.
     *
     * @param string $brandCode The code of the brand.
     * @return array An array of series data keyed by name.
     */
    private function getSeriesForBrand(string $brandCode): array
    {
        // ---- Fetch series data for the brand
        $series = $this->connection->fetchAllAssociative('
            SELECT DISTINCT topdata_series.label as name,
                   LOWER(HEX(topdata_series.id)) as id,
                   topdata_series.code as code
              FROM topdata_device
              INNER JOIN topdata_brand ON topdata_brand.id = topdata_device.brand_id
              LEFT JOIN topdata_series ON topdata_series.id = topdata_device.series_id
              WHERE topdata_brand.code = :brandCode
              AND topdata_device.is_enabled = 1
              AND topdata_series.id IS NOT NULL
              AND topdata_series.is_enabled = 1
              ORDER BY topdata_series.label
        ', [
            'brandCode' => $brandCode
        ]);

        // ---- Format series data keyed by name
        $return = [];
        foreach ($series as $serie) {
            $return[$serie['name']] = $serie;
        }

        return $return;
    }

    /**
     * Retrieves all enabled device types that have devices for a specific brand.
     *
     * @param string $brandCode The code of the brand.
     * @return array An array of device type data keyed by name.
     */
    private function getTypesForBrand(string $brandCode): array
    {
        // ---- Fetch device type data for the brand
        $types = $this->connection->fetchAllAssociative('
            SELECT DISTINCT topdata_device_type.label as name,
                   LOWER(HEX(topdata_device_type.id)) as id,
                   topdata_device_type.code as code
              FROM topdata_device
              INNER JOIN topdata_brand ON topdata_brand.id = topdata_device.brand_id
              LEFT JOIN topdata_device_type ON topdata_device_type.id = topdata_device.type_id
              WHERE topdata_brand.code = :brandCode
              AND topdata_device.is_enabled = 1
              AND topdata_device_type.id IS NOT NULL
              AND topdata_device_type.is_enabled = 1
              ORDER BY topdata_device_type.label
        ', [
            'brandCode' => $brandCode
        ]);

        // ---- Format device type data keyed by name
        $return = [];
        foreach ($types as $type) {
            $return[$type['name']] = $type;
        }

        return $return;
    }

    /**
     * Fetches devices for a specific brand with pagination
     */
    private function getDevicesByBrandPaginated(string $brandCode, int $limit, int $offset = 0): array
    {
        return $this->connection->fetchAllAssociative('
            SELECT topdata_device.model as name,
                   LOWER(HEX(topdata_device.id)) as id,
                   LOWER(HEX(topdata_device.brand_id)) as brand_id,
                   LOWER(HEX(topdata_device.series_id)) as series_id,
                   LOWER(HEX(topdata_device.type_id)) as type_id,
                   topdata_device.code as code,
                   topdata_brand.label as brand_name,
                   topdata_brand.code as brand_code,
                   LOWER(HEX(topdata_device.media_id)) as media_id,
                   topdata_series.label as series_name,
                   topdata_device_type.label as type_name
              FROM topdata_device
              INNER JOIN topdata_brand ON topdata_brand.id = topdata_device.brand_id
              LEFT JOIN topdata_series ON topdata_series.id = topdata_device.series_id
              LEFT JOIN topdata_device_type ON topdata_device_type.id = topdata_device.type_id
              WHERE topdata_brand.code = :brandCode
              AND topdata_device.is_enabled = 1
              ORDER BY topdata_device.code
              LIMIT :limit OFFSET :offset
        ', [
            'brandCode' => $brandCode,
            'limit' => $limit,
            'offset' => $offset
        ], [
            'limit' => \Doctrine\DBAL\ParameterType::INTEGER,
            'offset' => \Doctrine\DBAL\ParameterType::INTEGER
        ]);
    }

    /**
     * Fetches devices for a specific series with pagination
     */
    private function getDevicesBySeriesPaginated(string $brandCode, string $seriesId, int $limit, int $offset = 0): array
    {
        return $this->connection->fetchAllAssociative('
            SELECT topdata_device.model as name,
                   LOWER(HEX(topdata_device.id)) as id,
                   LOWER(HEX(topdata_device.brand_id)) as brand_id,
                   LOWER(HEX(topdata_device.series_id)) as series_id,
                   LOWER(HEX(topdata_device.type_id)) as type_id,
                   topdata_device.code as code,
                   topdata_brand.label as brand_name,
                   topdata_brand.code as brand_code,
                   LOWER(HEX(topdata_device.media_id)) as media_id,
                   topdata_series.label as series_name,
                   topdata_device_type.label as type_name
              FROM topdata_device
              INNER JOIN topdata_brand ON topdata_brand.id = topdata_device.brand_id
              LEFT JOIN topdata_series ON topdata_series.id = topdata_device.series_id
              LEFT JOIN topdata_device_type ON topdata_device_type.id = topdata_device.type_id
              WHERE topdata_brand.code = :brandCode
              AND topdata_device.series_id = UNHEX(:seriesId)
              AND topdata_device.is_enabled = 1
              ORDER BY topdata_device.code
              LIMIT :limit OFFSET :offset
        ', [
            'brandCode' => $brandCode,
            'seriesId' => $seriesId,
            'limit' => $limit,
            'offset' => $offset
        ], [
            'limit' => \Doctrine\DBAL\ParameterType::INTEGER,
            'offset' => \Doctrine\DBAL\ParameterType::INTEGER
        ]);
    }

    /**
     * Fetches devices for a specific type with pagination
     */
    private function getDevicesByTypePaginated(string $brandCode, string $typeId, int $limit, int $offset = 0): array
    {
        return $this->connection->fetchAllAssociative('
            SELECT topdata_device.model as name,
                   LOWER(HEX(topdata_device.id)) as id,
                   LOWER(HEX(topdata_device.brand_id)) as brand_id,
                   LOWER(HEX(topdata_device.series_id)) as series_id,
                   LOWER(HEX(topdata_device.type_id)) as type_id,
                   topdata_device.code as code,
                   topdata_brand.label as brand_name,
                   topdata_brand.code as brand_code,
                   LOWER(HEX(topdata_device.media_id)) as media_id,
                   topdata_series.label as series_name,
                   topdata_device_type.label as type_name
              FROM topdata_device
              INNER JOIN topdata_brand ON topdata_brand.id = topdata_device.brand_id
              LEFT JOIN topdata_series ON topdata_series.id = topdata_device.series_id
              LEFT JOIN topdata_device_type ON topdata_device_type.id = topdata_device.type_id
              WHERE topdata_brand.code = :brandCode
              AND topdata_device.type_id = UNHEX(:typeId)
              AND topdata_device.is_enabled = 1
              ORDER BY topdata_device.code
              LIMIT :limit OFFSET :offset
        ', [
            'brandCode' => $brandCode,
            'typeId' => $typeId,
            'limit' => $limit,
            'offset' => $offset
        ], [
            'limit' => \Doctrine\DBAL\ParameterType::INTEGER,
            'offset' => \Doctrine\DBAL\ParameterType::INTEGER
        ]);
    }

    /**
     * Retrieves a brand from the database by its code.
     *
     * @param string $brandCode The code of the brand to retrieve.
     * @return array The brand data, or an empty array if not found.
     */
    private function getBrand(string $brandCode): array
    {
        // ---- Fetch brand data
        $brand = $this->connection->fetchAllAssociative('
SELECT LOWER(HEX(id)) as id, 
       code, 
       label as name
  FROM topdata_brand
  WHERE (is_enabled=1)
        AND(code = "' . $brandCode . '")
            ');

        return isset($brand[0]) ? $brand[0] : [];
    }

    /**
     * Loads the OneBrandLetterPage for JSON response, including device-specific data and media URLs.
     *
     * @param Request $request The current HTTP request.
     * @param SalesChannelContext $salesChannelContext The current sales channel context.
     * @param string $brandCode The code of the brand to load.
     * @param string $displayMode The display mode ('all', 'series', or 'types').
     * @return OneBrandLetterPage The populated OneBrandLetterPage.
     */
    public function loadJson(
        Request             $request,
        SalesChannelContext $salesChannelContext,
        string              $brandCode,
        string              $displayMode = 'all'
    ): OneBrandLetterPage
    {
        // ---- Load generic page and set elements
        $page = $this->genericLoader->load($request, $salesChannelContext);
        /** @var OneBrandLetterPage $page */
        $page = OneBrandLetterPage::createFrom($page);
        $this->setElements($page, $brandCode, $salesChannelContext);

        $page->setTitle($this->translator->trans('topdata-topfinder.SEO.brandPageTitle', [
            '%brand%' => $page->brand['name'],
        ]));

        // ---- Determine the letter for the popup path
        $letter = $page->brand['code'][0];

        if (preg_match('/^[0-9]{1}$/', $letter)) {
            $letter = '0';
            $letterStr = '0-9';
        } else {
            $letterStr = strtoupper($letter);
        }

        $page->popupPath[] = [
            'name' => $letterStr,
            'path' => $this->router->generate('frontend.top-finder-api.popup-letter', ['letter' => $letter]),
        ];

        $page->popupPath[] = [
            'name' => $page->brand['name'],
        ];

        // ---- Determine the list type based on display mode
        $listType = 'brand';
        if ($displayMode == 'series') {
            usort($page->devices, function ($a, $b) {
                if ($a['series_name'] === '' && $b['series_name']) {
                    return 1;
                }
                if ($a['series_name'] && $b['series_name'] === '') {
                    return -1;
                }

                return $a['series_name'] . $a['code'] <=> $b['series_name'] . $b['code'];
            });

            $page->popupPath[] = [
                'name' => $this->translator->trans('topdata-topfinder.popup.brandSeries'),
            ];
            $listType = 'series';
        } elseif ($displayMode == 'types') {
            usort($page->devices, function ($a, $b) {
                if ($a['type_name'] === '' && $b['type_name']) {
                    return 1;
                }
                if ($a['type_name'] && $b['type_name'] === '') {
                    return -1;
                }

                return $a['type_name'] . $a['code'] <=> $b['type_name'] . $b['code'];
            });
            $page->popupPath[] = [
                'name' => $this->translator->trans('topdata-topfinder.popup.brandTypes'),
            ];
            $listType = 'types';
        } else {
            $page->popupPath[] = [
                'name' => $this->translator->trans('topdata-topfinder.popup.allModels'),
            ];
        }

        $mediaIds = [];
        $devicelist = $this->deviceToCustomerService->getDevicesOfCustomer($salesChannelContext->getCustomer());

        // ---- Process devices to include path, media URL, and device list count
        foreach ($page->devices as $key => $device) {
            $page->devices[$key]['path'] = $this->router->generate('frontend.top-finder-api.popup-device-new', [
                'deviceCode' => $device['code'],
                'listType'   => $listType,
            ]);
            if ($device['media_id']) {
                $mediaIds[] = $device['media_id'];
            }

            if (isset($devicelist[$device['id']]) && isset($devicelist[$device['id']]['devices'])) {
                $page->devices[$key]['devicelist'] = count($devicelist[$device['id']]['devices']);
            } else {
                $page->devices[$key]['devicelist'] = -1;
            }
        }

        // ---- Fetch media URLs for devices
        if (count($mediaIds)) {
            $medias = $this->mediaRepository->search(new Criteria($mediaIds), $salesChannelContext->getContext());

            foreach ($page->devices as $key => $device) {
                if ($device['media_id'] && $medias->get($device['media_id'])) {
                    $page->devices[$key]['media_url'] = $medias->get($device['media_id'])->getUrl();
                }
            }
        }

        return $page;
    }


}