<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Service;

use Doctrine\DBAL\Connection;
use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\System\SalesChannel\SalesChannelContext;

/**
 * Class TopdataBrandService
 *
 * This service is responsible for handling brand data operations.
 * It provides methods for retrieving and processing brand data.
 */
class TopdataBrandService
{
    public function __construct(
        private readonly Connection $connection,
        private readonly EntityRepository $topdataBrandRepository,
        private readonly LoggerInterface $logger
    ) {
    }

    /**
     * Get all enabled brands
     *
     * This method retrieves all enabled brands from the database,
     * sorted by sort value (descending) and then by name (ascending).
     *
     * @return array The array of enabled brands with code, name, and sort values
     */
    public function getAllEnabledBrands(): array
    {
        try {
            $brands = $this->connection->fetchAllAssociative(
                'SELECT code, label as name, sort FROM `topdata_brand`'
                . ' WHERE is_enabled = 1'
                . ' ORDER BY sort DESC, name ASC'
            );

            return $brands;
        } catch (\Exception $e) {
            $this->logger->error('Error fetching all enabled brands: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get a brand by its code
     *
     * This method retrieves a specific brand entity based on the provided code.
     * It returns the brand entity if found, or null if not found.
     *
     * @param string $code The brand code to search for
     * @param SalesChannelContext $context The sales channel context
     * @return \Topdata\TopdataTopFinderProSW6\Core\Content\TopdataBrand\TopdataBrandEntity|null The brand entity or null if not found
     */
    public function getBrandByCode(string $code, SalesChannelContext $context): ?\Topdata\TopdataTopFinderProSW6\Core\Content\TopdataBrand\TopdataBrandEntity
    {
        try {
            $criteria = new Criteria();
            $criteria->addFilter(new EqualsFilter('enabled', true));
            $criteria->addFilter(new EqualsFilter('code', $code));

            $brand = $this->topdataBrandRepository->search($criteria, $context->getContext())->first();

            return $brand;
        } catch (\Exception $e) {
            $this->logger->error('Error fetching brand by code: ' . $e->getMessage(), [
                'code' => $code
            ]);
            return null;
        }
    }

    /**
     * Get all enabled brands with ID and name
     *
     * This method retrieves all enabled brands from the database with their IDs and names,
     * sorted by sort value (descending) and then by label.
     *
     * @return array An array of brand data, each containing 'id' and 'name'
     */
    public function getBrands(): array
    {
        try {
            return $this->connection->createQueryBuilder()
                ->select('LOWER(HEX(id)) as id, label as name')
                ->from('topdata_brand')
                ->where('is_enabled = 1')
                ->orderBy('sort DESC, label')
                ->executeQuery()
                ->fetchAllAssociative();
        } catch (\Exception $e) {
            $this->logger->error('Error fetching brands: ' . $e->getMessage());
            return [];
        }
    }


    /**
     * Get brand information by code.
     *
     * @param string $brandCode The brand code
     * @return array|null The brand data or null if not found
     */
    public function getTopdataBrand(string $brandCode): ?array
    {
        $brand = $this->connection->fetchAssociative('
            SELECT LOWER(HEX(id)) as id,
                   code,
                   label as name
            FROM topdata_brand
            WHERE is_enabled = 1
              AND code = :brandCode
        ', ['brandCode' => $brandCode]);

        return $brand ?: null;
    }


}