<?php declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Service;

use Doctrine\DBAL\Connection;
use Psr\Log\LoggerInterface;
use Shopware\Core\Checkout\Cart\Cart;
use Shopware\Core\Checkout\Cart\CartBehavior;
use Shopware\Core\Checkout\Cart\CartProcessorInterface;
use Shopware\Core\Checkout\Cart\LineItem\CartDataCollection;
use Shopware\Core\Checkout\Cart\LineItem\LineItem;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\SalesChannel\SalesChannelContext;

/**
 * This class is responsible for associating a device with a cart line item based on the customer's device history.
 * It implements the CartProcessorInterface to hook into the cart calculation process.
 * 04/2025 created
 */
class CartDeviceAssociator implements CartProcessorInterface
{
    private array $fetchedDevicesCache = []; // Simple cache for fetched devices in this request

    public function __construct(
        private readonly DeviceHistoryService_Session $deviceHistoryService_Session,
        private readonly EntityRepository             $topdataDeviceRepository,
        private readonly LoggerInterface              $logger,
        private readonly SettingsService              $settingsService,
        private readonly Connection                   $connection,
    )
    {
    }

    /**
     * Processes the cart to associate devices with line items based on the customer's device history.
     *
     * This method iterates through the cart's line items, checks if they are products, and then attempts to find a compatible device
     * from the customer's device history. If a compatible device is found, it's added as an extension to the line item.
     */
    public function process(CartDataCollection $data, Cart $original, Cart $toCalculate, SalesChannelContext $context, CartBehavior $behavior): void
    {
        $this->logger->debug('CartDeviceAssociator: process started.');

        // ---- Check if the feature is enabled in plugin config
        // Assuming SettingsService uses 'showDeviceInCart' key
        $isEnabled = $this->settingsService->getBool('showDeviceInCart', $context->getSalesChannelId()); // <-- Pass SalesChannelId
        $this->logger->debug('CartDeviceAssociator: Feature enabled check.', ['isEnabled' => $isEnabled]);
        if (!$isEnabled) {
            return; // Do nothing if disabled
        }

        // ---- Get the device history from the session
        $deviceHistoryIds = $this->deviceHistoryService_Session->getHistory();
        $this->logger->debug('CartDeviceAssociator: Fetched device history.', ['history' => $deviceHistoryIds]);
        if (empty($deviceHistoryIds)) {
            $this->logger->debug('CartDeviceAssociator: Device history is empty. Exiting.');
            return; // No history, nothing to associate
        }

        // ---- Reset cache for each process run
        $this->fetchedDevicesCache = [];

        // ---- Add the device entity as an extension to each line item
        foreach ($toCalculate->getLineItems()->getFlat() as $lineItem) {
            $this->logger->debug('CartDeviceAssociator: Processing line item.', ['lineItemId' => $lineItem->getId(), 'productId' => $lineItem->getReferencedId(), 'type' => $lineItem->getType()]);
            if ($lineItem->getType() !== LineItem::PRODUCT_LINE_ITEM_TYPE || !$lineItem->getReferencedId()) {
                continue; // Skip non-products or items without referenced ID
            }

            $productId = $lineItem->getReferencedId();

            // ---- Iterate through history (most recent first) to find the first compatible device
            foreach ($deviceHistoryIds as $historyDeviceId) {
                $this->logger->debug('CartDeviceAssociator: Checking history device ID.', ['productId' => $productId, 'historyDeviceId' => $historyDeviceId]);
                if (!Uuid::isValid($historyDeviceId)) { // Basic validation
                    $this->logger->warning('CartDeviceAssociator: Invalid UUID found in history.', ['invalidId' => $historyDeviceId]);
                    continue;
                }

                $isCompatible = $this->isCompatible($productId, $historyDeviceId);
                $this->logger->debug('CartDeviceAssociator: Compatibility check result.', ['productId' => $productId, 'historyDeviceId' => $historyDeviceId, 'isCompatible' => $isCompatible]);

                if ($isCompatible) {
                    // ---- Found the most recent compatible device for this product
                    $compatibleDeviceEntity = $this->fetchDeviceEntity($historyDeviceId, $context);
                    $this->logger->debug('CartDeviceAssociator: Fetched compatible device entity.', ['deviceId' => $historyDeviceId, 'entityFound' => !is_null($compatibleDeviceEntity)]);

                    if ($compatibleDeviceEntity) {
                        // ---- Add extension and break inner loop (history check) for this line item
                        $lineItem->addExtension('topfinderLastDevice', $compatibleDeviceEntity);
                        $this->logger->info('CartDeviceAssociator: Added extension "topfinderLastDevice" to line item.', ['lineItemId' => $lineItem->getId(), 'deviceId' => $historyDeviceId]);
                        break;
                    }
                    // If fetching failed, maybe log it, but continue checking older history items
                }
            }
        }
        $this->logger->debug('CartDeviceAssociator: process finished.');
    }

    /**
     * Checks if a product and device are compatible using the mapping table.
     *
     * This method queries the database to determine if a relationship exists between the given product and device.
     *
     * @param string $productId The ID of the product.
     * @param string $deviceId  The ID of the device.
     *
     * @return bool True if the product and device are compatible, false otherwise.
     */
    private function isCompatible(string $productId, string $deviceId): bool
    {
        // ---- Convert IDs to binary for the query
        $this->logger->debug('CartDeviceAssociator::isCompatible - Checking compatibility.', ['productId' => $productId, 'deviceId' => $deviceId]);
        $productIdBytes = Uuid::fromHexToBytes($productId);
        $deviceIdBytes = Uuid::fromHexToBytes($deviceId);

        $sql = "SELECT 1 FROM `topdata_device_to_product` WHERE `product_id` = :productId AND `device_id` = :deviceId LIMIT 1";
        $result = $this->connection->fetchOne($sql, [
            'productId' => $productIdBytes,
            'deviceId'  => $deviceIdBytes,
        ]);

        $this->logger->debug('CartDeviceAssociator::isCompatible - DB Result.', ['result' => $result]);
        return $result !== false; // Returns '1' if found, false otherwise
    }

    /**
     * Fetches a device entity by ID, using a simple cache.
     *
     * This method first checks if the device entity is already in the cache. If not, it fetches the entity from the database
     * and stores it in the cache for future use.
     *
     * @param string                $deviceId The ID of the device to fetch.
     * @param SalesChannelContext $context  The sales channel context.
     *
     * @return object|null The device entity, or null if not found.
     */
    private function fetchDeviceEntity(string $deviceId, SalesChannelContext $context): ?object // Adjust return type if needed (e.g., DeviceEntity)
    {
        $this->logger->debug('CartDeviceAssociator::fetchDeviceEntity - Attempting to fetch device.', ['deviceId' => $deviceId]);
        if (isset($this->fetchedDevicesCache[$deviceId])) {
            $this->logger->debug('CartDeviceAssociator::fetchDeviceEntity - Returning from cache.', ['deviceId' => $deviceId, 'foundInCache' => !is_null($this->fetchedDevicesCache[$deviceId])]);
            return $this->fetchedDevicesCache[$deviceId];
        }

        // ---- Fetch device from repository
        $criteria = new Criteria([$deviceId]);
        $criteria->addAssociation('brand');
        $criteria->addAssociation('media');
        $this->logger->debug('CartDeviceAssociator::fetchDeviceEntity - Created criteria with associations.', ['deviceId' => $deviceId]);

        $deviceEntity = $this->topdataDeviceRepository->search($criteria, $context->getContext())->get($deviceId);

        $this->logger->debug('CartDeviceAssociator::fetchDeviceEntity - Repository search result.', ['deviceId' => $deviceId, 'entityFound' => !is_null($deviceEntity)]);
        // Cache the result (even if null) to avoid repeated lookups
        $this->fetchedDevicesCache[$deviceId] = $deviceEntity;

        return $deviceEntity;
    }
}