import Plugin from 'src/plugin-system/plugin.class';
import DomAccess from 'src/helper/dom-access.helper';
import HttpClient from 'src/service/http-client.service';
import CookieStorage from 'src/helper/storage/cookie-storage.helper';
import TopdataTopFinderProDevice from "./topdata-top-finder-pro.device.class";
import SlimSelect from "../slimselect/slimselect.min";
declare global {
    interface Window {
        popupDeviceReturnPath?: string;
    }
}

declare namespace JSX {
    interface IntrinsicElements {
        [elem: string]: any;
    }
}

export default class TopdataTopFinderProPopupFinder extends Plugin {
    // Define properties with initial values to satisfy TypeScript
    private _client: HttpClient;
    private BrandId: string | null = null;
    private SeriesId: string | null = null;
    private TypeId: string | null = null;
    private el: HTMLElement | null = null;
    private popupHistory: string[] = [];

    // Define method types
    private options: typeof TopdataTopFinderProPopupFinder.options;

    static options = {
        switchSelector: 'select.top-finder-switch',
        brandSelector: 'select.top-finder-brand',
        seriesSelector: 'select.top-finder-series',
        typeSelector: 'select.top-finder-types',
        deviceSelector: 'select.top-finder-devices',
        //deviceHistorySelector: 'select.top-finder-device-history',
        deviceListDevicesSelector: 'select.top-finder-device',
        seriesContainer: '.top-finder-series-container',
        typeContainer: '.top-finder-types-container',
    };

    constructor() {
        super();
        this.options = TopdataTopFinderProPopupFinder.options;
    }

    /**
     * Handles dynamic content updates by reinitializing SlimSelect on new elements.
     * This method ensures that SlimSelect is properly initialized on all elements
     * with the top-finder-slimselect class, including those added dynamically.
     * @private
     */
    private _handleDynamicContent = (): void => {
        const slimSelectElements = this.el?.querySelectorAll('.top-finder-slimselect') as NodeListOf<HTMLSelectElement> | null;
        if (slimSelectElements) {
            slimSelectElements.forEach(element => {
                // Destroy existing SlimSelect instance if it exists to prevent duplicates
                if ((element as any).slim) {
                    (element as any).slim.destroy();
                }
                this._initializeSlimSelect(element);
            });
        }
    };

    init = () => {
        console.log('Load TopdataTopFinderProPopupFinder Plugin');
        //this._client = new HttpClient(window.accessKey, window.contextToken);
        this._client = new HttpClient();
        this.BrandId = DomAccess.querySelector(this.el, this.options.brandSelector)?.value ?? null;
        this.SeriesId = DomAccess.querySelector(this.el, this.options.seriesSelector)?.value ?? null;
        this.TypeId = DomAccess.querySelector(this.el, this.options.typeSelector)?.value ?? null;

        // Initialize SlimSelect for all popup dropdown elements with top-finder-slimselect class
        const slimSelectElements = this.el?.querySelectorAll('.top-finder-slimselect') as NodeListOf<HTMLSelectElement> | null;
        if (slimSelectElements) {
            slimSelectElements.forEach(element => {
                // Ensure we properly initialize SlimSelect even if it was previously initialized
                if ((element as any).slim) {
                    (element as any).slim.destroy();
                }
                this._initializeSlimSelect(element);
            });
        }

        // Initialize SlimSelect for the popup device selector
        const popupDeviceSelect = this.el?.querySelector('.top-finder-device-popup-new') as HTMLSelectElement | null;
        if (popupDeviceSelect) {
            this._initializeSlimSelect(popupDeviceSelect);
        }

        this._registerEvents();

        let btn = document.querySelector("a[data-bs-target='#topdata-finder-popup']");

        if(btn) {
            btn.addEventListener('click', this.openModal.bind(this));
        }

        // Add event listener for dynamic content loading
        const popupContentContainer = this.el?.querySelector('.popup-content-container');
        if (popupContentContainer) {
            popupContentContainer.addEventListener('content:loaded', this._handleDynamicContent.bind(this));
        }

        // Add event listener for AJAX content updates
        document.body.addEventListener('ajaxContentUpdated', this._handleDynamicContent.bind(this));

        // Add event listener for custom event when AJAX content is loaded in the popup
        document.body.addEventListener('popupContentLoaded', this._handleDynamicContent.bind(this));

        //window.addEventListener('click', this.closeModal.bind(this));
        //window.addEventListener('touchend', this.closeModal.bind(this));
    };

    /**
     * Initializes SlimSelect for a given select element with consistent configuration.
     * @param {HTMLSelectElement} element The select element to initialize SlimSelect on.
     * @private
     */
    private _initializeSlimSelect = (element: HTMLSelectElement): void => {
        new SlimSelect({
            select: element,
            settings: {
                searchText: 'Search...',
                searchPlaceholder: 'Search options',
                searchHighlight: true,
                closeOnSelect: true,
                showSearch: true,
                searchFilter: (option: any, search: string) => {
                    return option.text.toLowerCase().includes(search.toLowerCase());
                }
            }
        });
    };

    /**
     * Register events
     * @private
     */
    _registerEvents = () => {

        if(DomAccess.querySelector(this.el, this.options.brandSelector) !== null)
            DomAccess.querySelector(this.el, this.options.brandSelector).addEventListener('change', this.onChangeBrandSelection);

        if(DomAccess.querySelector(this.el, this.options.typeSelector) !== null)
            DomAccess.querySelector(this.el, this.options.typeSelector).addEventListener('change', this.onChangeTypeSelection);

        if(DomAccess.querySelector(this.el, this.options.seriesSelector) !== null)
            DomAccess.querySelector(this.el, this.options.seriesSelector).addEventListener('change', this.onChangeSeriesSelection);

        if(DomAccess.querySelector(this.el, this.options.deviceSelector) !== null)
            DomAccess.querySelector(this.el, this.options.deviceSelector).addEventListener('change', this.onChangeDeviceSelection);

        if(DomAccess.querySelector(this.el, '.top-finder-toggle-class') !== null)
            DomAccess.querySelector(this.el, '.top-finder-toggle-class').addEventListener('click', this.onClickToggleMenu);

        if(DomAccess.querySelector(this.el, '.topdata-finder-popup .modal-body') !== null)
            DomAccess.querySelector(this.el, '.topdata-finder-popup .modal-body').addEventListener('click', this.onClickCloseMenu);

        if(DomAccess.querySelector(this.el, '.topdata-finder-popup-back') !== null)
            DomAccess.querySelector(this.el, '.topdata-finder-popup-back').addEventListener('click', this.onClickHistoryBack);

        if(DomAccess.querySelector(this.el, '.topdata-finder-popup-reset') !== null)
            DomAccess.querySelector(this.el, '.topdata-finder-popup-reset').addEventListener('click', this.onClickReset);

        // changePopupMode
        if(DomAccess.querySelector(this.el, '.topdata-finder-popup-mode-abc-trigger') !== null)
            DomAccess.querySelector(this.el, '.topdata-finder-popup-mode-abc-trigger')
                .addEventListener('click', this.changePopupMode);

        if(DomAccess.querySelector(this.el, '.topdata-finder-popup-mode-selectboxes-trigger') !== null)
            DomAccess.querySelector(this.el, '.topdata-finder-popup-mode-selectboxes-trigger')
                .addEventListener('click', this.changePopupMode);

        /*
        if(DomAccess.querySelector(this.el, '.topdata-finder-popup-mode-search-trigger') !== null)
            DomAccess.querySelector(this.el, '.topdata-finder-popup-mode-search-trigger')
                .addEventListener('click', this.changePopupMode.bind(this));
        */

        //if(DomAccess.querySelector(this.el, this.options.deviceHistorySelector) !== null)
        //    DomAccess.querySelector(this.el, this.options.deviceHistorySelector).addEventListener('change', this.onChangeDeviceSelection.bind(this));

        var elements;

        if(DomAccess.querySelectorAll(this.el, '.top-finder-device-popup-new') !== null) {
            elements = DomAccess.querySelectorAll(this.el, '.top-finder-device-popup-new');
            elements.forEach((element: HTMLElement) => {
                element.addEventListener('change', this.onChangeDeviceHistory);
            });
        }

        if(DomAccess.querySelectorAll(this.el, '.top-finder-hiddable-trigger') !== null) {
            elements = DomAccess.querySelectorAll(this.el, '.top-finder-hiddable-trigger');
            elements.forEach((element: HTMLElement) => {
                element.addEventListener('click', this.onClickHiddableTrigger);
            });
        }

        if(DomAccess.querySelectorAll(this.el, '.popup-navigate-right') !== null) {
            elements = DomAccess.querySelectorAll(this.el, '.popup-navigate-right');
            elements.forEach((element: HTMLElement) => {
                element.addEventListener('click', this.onClickNavigateRight);
            });
        }

        if(DomAccess.querySelectorAll(this.el, '.empty-content-show-link') !== null) {
            elements = DomAccess.querySelectorAll(this.el, '.empty-content-show-link');
            elements.forEach((element: HTMLElement) => {
                element.addEventListener('click', this.onClickEmptyContentShow);
            });
        }

        elements.forEach((element: HTMLElement) => {
            element.addEventListener('click', TopdataTopFinderProDevice.removeDevice);
        });
    };

    openModal = (event: Event) => {
        //console.log('open');
        this.popupResetAll();
        this.popupHistory = [];
        this.popupHistoryAdd(window.location.pathname);
        this._registerEvents();
        this._handleDynamicContent();
    };

    onClickReset = (event: Event) => {
        if ((event.currentTarget as HTMLElement).classList.contains('inactive')) {
            return;
        }
        this.popupResetAll();
    };

    popupHistoryAdd = (path: string) => {
        let lastPath = this.popupHistory.pop()
        if((path !== lastPath) && lastPath) {
            this.popupHistory.push(lastPath);
        }
        this.popupHistory.push(path);

        if(DomAccess.querySelectorAll(this.el, '.topdata-finder-popup-back') !== null) {
            var elements = DomAccess.querySelectorAll(this.el, '.topdata-finder-popup-back');
            elements.forEach((element: HTMLElement) => {
                element.classList.remove('inactive');
            });
        }
    }

    onClickHiddableTrigger = (event: Event) => {
        var elements
        var group = (event.currentTarget as HTMLElement).dataset.group;
        console.log(group);
        if(group) {
            elements = DomAccess.querySelectorAll(this.el, '.top-finder-hiddable[data-group="'+group+'"]');
            elements.forEach((element: HTMLElement) => {
                element.classList.toggle('hidden');
            });
        }else{
            elements = DomAccess.querySelectorAll(this.el, '.top-finder-hiddable');
            elements.forEach((element: HTMLElement) => {
                element.classList.toggle('hidden');
            });
        }
    }

    /**
     * @param {Event} event
     */
    onChangeBrandSelection = (event: Event) => {
        event.preventDefault();

        var path
        this.BrandId = (event.currentTarget as HTMLSelectElement).value;

        var brandSelector = DomAccess.querySelector(this.el, this.options.brandSelector);
        var seriesSelector = DomAccess.querySelector(this.el, this.options.seriesSelector);
        var typeSelector = DomAccess.querySelector(this.el, this.options.typeSelector);
        var deviceSelector = DomAccess.querySelector(this.el, this.options.deviceSelector);

        var emptySeriesLink = DomAccess.querySelector(this.el, '.empty-series-link');
        var emptyTypesLink = DomAccess.querySelector(this.el, '.empty-types-link');
        var emptyDeviceLink = DomAccess.querySelector(this.el, '.empty-devices-link');

        if(brandSelector.length !== 0) {
            path = (event.currentTarget as HTMLSelectElement).dataset.pathgotobrand;
            if (path && this.BrandId) {
                path = path.replace('brandcode', this.BrandId);
                this.popupButtonDevicesEnable(path);
            }
        }

        if(typeSelector.length !== 0) {
            path = (event.currentTarget as HTMLSelectElement).dataset.pathloadtypes;
            if (path && this.BrandId) {
                this.resetSelectOptions(typeSelector);
                this.loadNewSelectOptions(path.replace('brandcode', this.BrandId), typeSelector);
                typeSelector.classList.remove('hidden');
                emptySeriesLink.classList.add('hidden');
            }
        }

        if(seriesSelector.length !== 0) {
            path = (event.currentTarget as HTMLSelectElement).dataset.pathloadseries;
            if (path && this.BrandId) {
                this.resetSelectOptions(seriesSelector);
                this.loadNewSelectOptions(path.replace('brandcode', this.BrandId), seriesSelector);
                seriesSelector.classList.remove('hidden');
                emptyTypesLink.classList.add('hidden');
            }
        }

        if(deviceSelector.length !== 0) {
            path = (event.currentTarget as HTMLSelectElement).dataset.pathloaddevices;
            if (path && this.BrandId) {
                this.resetSelectOptions(deviceSelector);
                this.loadNewSelectOptions(path.replace('brandcode', this.BrandId), deviceSelector);
                deviceSelector.classList.remove('hidden');
                emptyDeviceLink.classList.add('hidden');
            }
        }

        DomAccess.querySelector(this.el, '.topdata-finder-popup-reset').classList.remove('inactive');
    }

    /**
     * @param {Event} event
     */
    onChangeTypeSelection = (event: Event) => {
        event.preventDefault();

        var path
        this.TypeId = (event.currentTarget as HTMLSelectElement).value;
        var deviceSelector = DomAccess.querySelector(this.el, this.options.deviceSelector);

        if(deviceSelector.length !== 0) {

            path = (event.currentTarget as HTMLSelectElement).dataset.pathgototype;
            if (path && this.TypeId) {
                path = path.replace('typecode', this.TypeId);
                this.popupButtonDevicesEnable(path);

                path = (event.currentTarget as HTMLSelectElement).dataset.pathloaddevices;
                if (path && this.BrandId) {
                    path = path.replace('brandcode', this.BrandId);
                    if (path) {
                        path = path.replace('typecode', this.TypeId);

                        this.resetSelectOptions(deviceSelector);
                        this.loadNewSelectOptions(path, deviceSelector);
                    }
                }
            }
        }
    }

    /**
     * @param {Event} event
     */
    onChangeSeriesSelection = (event: Event) => {
        event.preventDefault();

        var path
        this.SeriesId = (event.currentTarget as HTMLSelectElement).value;
        var deviceSelector = DomAccess.querySelector(this.el, this.options.deviceSelector);

        if(deviceSelector.length !== 0) {

            path = (event.currentTarget as HTMLSelectElement).dataset.pathgotoseries;
            if (path && this.SeriesId) {
                path = path.replace('seriescode', this.SeriesId);
                this.popupButtonDevicesEnable(path);

                path = (event.currentTarget as HTMLSelectElement).dataset.pathloaddevices;
                if (path && this.BrandId) {
                    path = path.replace('brandcode', this.BrandId);
                    if (path) {
                        path = path.replace('seriescode', this.SeriesId);

                        this.resetSelectOptions(deviceSelector);
                        this.loadNewSelectOptions(path, deviceSelector);
                    }
                }
            }
        }
    }

    /**
     * @param {Event} event
     */
    onChangeDeviceSelection = (event: Event) => {
        //event.preventDefault();
        event.stopImmediatePropagation();

        var DeviceId = (event.currentTarget as HTMLSelectElement).value;

        if(DeviceId !== '' ){
            var path = (event.currentTarget as HTMLSelectElement).dataset.path;
            var listType = (event.currentTarget as HTMLSelectElement).dataset.listtype;
            var pathLisDevices = (event.currentTarget as HTMLSelectElement).dataset.pathlisdevices;

            if (path) {
                path += "?deviceCode=" + DeviceId + "&listType" + listType;
                this.popupLoadDevice(path, '.topdata-finder-popup-content-right');

                if(pathLisDevices){
                    this.popupButtonDevicesEnable(pathLisDevices);
                }else{
                    this.popupButtonDevicesDisable()
                }
            }

            //window.location.href = path;
        }
        return false;
    }

    onChangeDeviceHistory = (event: Event) => {
        event.stopImmediatePropagation();

        let DeviceCode = (event.currentTarget as HTMLSelectElement).value;
        window.popupDeviceReturnPath = '';

        if(DeviceCode === '' || DeviceCode === null || DeviceCode === (event.currentTarget as HTMLSelectElement).dataset.plcaholder) {
            return false;
        }
        let path = (event.currentTarget as HTMLSelectElement).dataset.path;
        let listType = (event.currentTarget as HTMLSelectElement).dataset.listtype;
        if (path) {
            path += "?deviceCode="+DeviceCode+"&listType="+listType;
            this.popupLoadDevice(path,'.topdata-finder-popup-content-right');
            let pathListDevice = (event.currentTarget as HTMLSelectElement).dataset.pathlisdevices;
            if(pathListDevice) {
                this.popupButtonDevicesEnable(pathListDevice);
            }
        }
        return false;

    }

    onClickNavigateRight = (event: Event) => {
        event.stopPropagation();

        if((event.currentTarget as HTMLElement).classList.contains('inactive')){
            return;
        }

        var path = (event.currentTarget as HTMLElement).dataset.path;
        if(!path) {
            return;
        }

        this.popupNavigateRight(path);
        this.popupHistoryAdd(path);

        this.popupButtonProductsDisable();
        this.popupButtonsListReset();

        this.onClickCloseMenu(event);

        DomAccess.querySelector(this.el, '.topdata-finder-popup-reset').classList.remove('inactive');
    }

    popupNavigateRight = (path: string) => {
        var container = DomAccess.querySelector(this.el, '.topdata-finder-popup-content-right');

        this._client.get(path, (responseText: string, request: XMLHttpRequest) => {
            if (request.status >= 400) {
                console.log(responseText);
            }
            try {
                let success = true;
                let response;
                let returnPath = '';
                let responseHtml = '';
                let loadOffset = '';

                if(request.getResponseHeader('content-type') === 'application/json') {
                    response = JSON.parse(responseText);
                    responseHtml = response.html;
                    success = response.success;
                } else {
                    returnPath = request.getResponseHeader('returnPath') || '';
                    responseHtml = responseText;
                    loadOffset = request.getResponseHeader('loadOffset') || '';
                }

                if (success === true) {
                    if(returnPath) {
                        window.popupDeviceReturnPath = returnPath;
                    }
                    container.innerHTML = responseHtml;

                    // Reinitialize SlimSelect on the newly loaded content
                    this._handleDynamicContent();

                    this._registrateEventsForPopupNavigateRight(container);

                    if(loadOffset) {
                        if(path.indexOf('?')>=0) {
                           this.popupLoadMoreDevices(path + '&offset=' + loadOffset);
                        }
                        else {
                          this.popupLoadMoreDevices(path + '?offset=' + loadOffset);
                        }
                    }
                }else {
                    console.log(responseText);
                }
            } catch (error) {
                console.log(error);
            }
        });
    }

    // Implement missing methods
    private onClickToggleMenu(event: Event) {}
    private onClickCloseMenu(event: Event) {}
    private onClickHistoryBack(event: Event) {}
    private changePopupMode(event: Event) {}
    private onClickEmptyContentShow(event: Event) {}
    private popupResetAll() {}
    private _registrateEventsForPopupNavigateRight(container: HTMLElement) {}
    private popupButtonDevicesEnable(path: string) {}
    private popupButtonDevicesDisable() {}
    private resetSelectOptions(selector: HTMLElement) {}
    private loadNewSelectOptions(path: string, selector: HTMLElement) {}
    private popupLoadDevice(path: string, container: string) {}
    private popupButtonProductsDisable() {}
    private popupButtonsListReset() {}
    private popupLoadMoreDevices(path: string) {}
}
