<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader;

use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Topdata\TopdataTopFinderProSW6\Component\Collection;
use Topdata\TopdataTopFinderProSW6\Service\BrandDataService;
use Topdata\TopdataTopFinderProSW6\Service\DeviceHistoryService_Cookie;
use Topdata\TopdataTopFinderProSW6\Service\DeviceToCustomerService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\PopupFormWidget\PopupFormWidgetPage;

/**
 * Loads the necessary data for the Popup Form Widget page.
 * This includes fetching brands, checking settings for displaying series, types, and device lists,
 * and potentially loading the user's device list or device history.
 */
class PopupFormWidgetPageLoader
{


    public function __construct(
        private readonly GenericPageLoader           $genericPageLoader,
        private readonly SettingsService             $settingsService,
        private readonly BrandDataService            $brandDataService,
        private readonly DeviceToCustomerService     $deviceToCustomerService,
        private readonly DeviceHistoryService_Cookie $deviceHistoryService_Cookie
    )
    {
    }

    /**
     * Loads the data required for the popup form widget page.
     *
     * @param Request $request The current request.
     * @param SalesChannelContext $salesChannelContext The current sales channel context.
     * @return PopupFormWidgetPage The populated page object for the popup form widget.
     */
    public
    function load(Request $request, SalesChannelContext $salesChannelContext): PopupFormWidgetPage
    {
        // ---- Load generic page data
        $page = $this->genericPageLoader->load($request, $salesChannelContext);
        /** @var PopupFormWidgetPage $page */
        $page = PopupFormWidgetPage::createFrom($page);

        // ---- Prepare data structure for the finder
        $struct = new Collection();

        // ---- Add basic finder data
        $struct->set('brands', $this->brandDataService->getBrands());
        $struct->set('showSeries', $this->settingsService->getBool('showSeries'));
        $struct->set('showTypes', $this->settingsService->getBool('showTypes'));
        $struct->set('showDevicelist', $this->settingsService->getBool('showDevicelist'));

        // ---- Add device list if enabled and user is logged in
        if ($this->settingsService->getBool('showDevicelist')) {
            if ($salesChannelContext->getCustomer() && !($salesChannelContext->getCustomer()->getGuest())) {
                $struct->set('userLogged', true);
                $struct->set('deviceList', $this->deviceToCustomerService->getDeviceListForCustomer($salesChannelContext));
            } else {
                $struct->set('userLogged', false);
                $struct->set('deviceList', false);
            }
        }

        // ---- Add device history if enabled
        if ($this->settingsService->getBool('showDeviceHistory') && $this->settingsService->getBool('showDeviceHistorySelect')) {
            $struct->set(
                'deviceHistory',
                $this->deviceHistoryService_Cookie->getDeviceEntities(
                    $request,
                    $salesChannelContext->getContext()
                )
            );
        }

        // ---- Assign the prepared data structure to the page
        $page->finder = $struct;

        return $page;
    }
}