import Plugin from 'src/plugin-system/plugin.class';
import DomAccess from 'src/helper/dom-access.helper';
import HttpClient from 'src/service/http-client.service';
import CookieStorage from 'src/helper/storage/cookie-storage.helper';
import PseudoModalUtil from 'src/utility/modal-extension/pseudo-modal.util';
import SlimSelect from "../slimselect/slimselect.min";

/**
 * TopdataTopFinderProFinder Plugin
 *
 * This plugin handles the functionality for the Topdata Top Finder Pro feature.
 * It manages various selectors for brands, series, types, and devices, and handles
 * their respective change events.
 */
export default class TopdataTopFinderProFinder extends Plugin {

    static options = {
        switchSelector:        'select.top-finder-switch',
        brandSelector:         'select.top-finder-brand',
        seriesSelector:        'select.top-finder-series',
        typeSelector:          'select.top-finder-types',
        deviceSelector:        'select.top-finder-devices',
        deviceHistorySelector: 'select.top-finder-device-history',

        seriesContainer: '.top-finder-series-container',
        typeContainer:   '.top-finder-types-container',
    };

    private _client: HttpClient;
    // these are not IDs, they are the "codes" of the entities, eg "canon"
    private BrandId: string | null;
    private SeriesId: string | null;
    private TypeId: string | null;
    private modal: PseudoModalUtil;
    private ajaxSpinnerMode: string | null;
    private spinnerOverlay: HTMLElement | null;
    private _isUpdatingProgrammatically: boolean = false;


    /**
     * Initialize the plugin
     * Sets up the HTTP client and initializes the brand, series, and type IDs
     */
    init() {
        console.log('TopdataTopFinderProFinder::init()');
        this._client = new HttpClient();

        // Read spinner configuration
        this.ajaxSpinnerMode = DomAccess.getAttribute(this.el, 'data-ajax-spinner-mode', false) ?? null;
        this.spinnerOverlay = DomAccess.querySelector(this.el, '.top-finder-ajax-overlay', false) ?? null;

        this.BrandId = (DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.brandSelector, false) as HTMLSelectElement)?.value ?? null;
        this.SeriesId = (DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.seriesSelector, false) as HTMLSelectElement)?.value ?? null;
        this.TypeId = (DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.typeSelector, false) as HTMLSelectElement)?.value ?? null;

        console.log({BrandId: this.BrandId, SeriesId: this.SeriesId, TypeId: this.TypeId});

        // Initialize SlimSelect for brand selector
        const brandSelect = this.el.querySelector('.top-finder-brand') as HTMLSelectElement;
        if (brandSelect) {
            this._initializeSlimSelect(brandSelect);
        }

        // Initialize SlimSelect for series selector
        const seriesSelect = this.el.querySelector('.top-finder-series') as HTMLSelectElement;
        if (seriesSelect) {
            this._initializeSlimSelect(seriesSelect);
        }

        // Initialize SlimSelect for types selector
        const typesSelect = this.el.querySelector('.top-finder-types') as HTMLSelectElement;
        if (typesSelect) {
            this._initializeSlimSelect(typesSelect);
        }

        // Initialize SlimSelect for devices selector
        const devicesSelect = this.el.querySelector('.top-finder-devices') as HTMLSelectElement;
        if (devicesSelect) {
            this._initializeSlimSelect(devicesSelect);
        }

        // Initialize SlimSelect for switch selector
        const switchSelect = this.el.querySelector('.top-finder-switch') as HTMLSelectElement;
        if (switchSelect) {
            this._initializeSlimSelect(switchSelect);
        }

        // Initialize SlimSelect for device history selector
        const deviceHistorySelect = this.el.querySelector('.top-finder-device-history') as HTMLSelectElement;
        if (deviceHistorySelect) {
            this._initializeSlimSelect(deviceHistorySelect);
        }

        // Handle dynamic content updates
        this._handleDynamicContent();

        // Add event listener for dynamic content updates
        const finderResultContainer = document.querySelector('.finder-result-container');
        if (finderResultContainer) {
            finderResultContainer.addEventListener('DOMContentLoaded', this._handleDynamicContent.bind(this));
        }

        this._registerEvents();

        // --- Trigger initial AJAX load if needed ---
        // Search globally for the first tab that needs AJAX loading
        const firstAjaxTab = document.querySelector('.top-finder-brand-devices-load-tab');
        if (firstAjaxTab) {
            // Use setTimeout to ensure the click happens after the current execution context
            setTimeout(() => {
                (firstAjaxTab as HTMLElement).click();
            }, 50); // Small delay might help ensure everything is ready
        }
        // --- End Trigger ---
    }

    /**
     * Register events
     * @private
     */
    /**
     * Register all event listeners for the plugin
     * @private
     */
    _registerEvents() {
        if (this.el.querySelector(TopdataTopFinderProFinder.options.switchSelector) !== null) {
            console.log("SwitchSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.switchSelector).addEventListener('change', this.onChangeSwitchSelection.bind(this));
        }

        if (this.el.querySelector(TopdataTopFinderProFinder.options.brandSelector) !== null) {
            console.log("BrandSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.brandSelector).addEventListener('change', this.onChangeBrandSelection.bind(this));
        }

        if (this.el.querySelector(TopdataTopFinderProFinder.options.typeSelector) !== null) {
            console.log("TypeSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.typeSelector).addEventListener('change', this.onChangeTypeSelection.bind(this));
        }

        if (this.el.querySelector(TopdataTopFinderProFinder.options.seriesSelector) !== null) {
            console.log("SeriesSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.seriesSelector).addEventListener('change', this.onChangeSeriesSelection.bind(this));
        }

        if (this.el.querySelector(TopdataTopFinderProFinder.options.deviceSelector) !== null) {
            console.log("DeviceSelector found");
            this.el.querySelector(TopdataTopFinderProFinder.options.deviceSelector).addEventListener('change', this.onChangeDeviceSelection.bind(this));
        }

        // ---- device history selector
        let deviceHistorySelector = this.el.querySelector(TopdataTopFinderProFinder.options.deviceHistorySelector);
        if (deviceHistorySelector) {
            console.log("DeviceHistorySelector found");
            deviceHistorySelector.addEventListener('change', this.onChangeDeviceSelection.bind(this));
        }

        let element

        element = DomAccess.querySelector(this.el, '.top-finder-selectboxes-close.top-finder-selectboxes-hide', false)
        if (element) {
            element.addEventListener('click', this.closeSelectbox.bind(this));
        }

        element = DomAccess.querySelector(this.el, '.top-finder-selectboxes-close.top-finder-selectboxes-show', false)
        if (element) {
            element.addEventListener('click', this.openSelectbox.bind(this));
        }

        this._registerAjaxTabEvents(); // Add call to register AJAX tab listeners
    }

    /**
     * Register listeners for tabs that load content via AJAX
     * @private
     */
    _registerAjaxTabEvents() {
        // Search globally instead of within this.el
        const ajaxTabs = document.querySelectorAll('.top-finder-brand-devices-load-tab');
        // *** ADD LOGGING HERE ***
        console.log('[TopFinder] Found AJAX tabs globally:', ajaxTabs.length, ajaxTabs);
        ajaxTabs.forEach(tab => {
            // *** ADD LOGGING HERE ***
            console.log('[TopFinder] Processing tab for listener:', tab);
            // Ensure listener isn't added multiple times if init runs again
            if (!(tab as HTMLElement).dataset.ajaxListenerAttached) {
                // *** ADD LOGGING HERE ***
                console.log('[TopFinder] Attaching listener to tab:', tab);
                tab.addEventListener('click', this.onAjaxTabClick.bind(this));
                (tab as HTMLElement).dataset.ajaxListenerAttached = 'true';
            } else {
                // *** ADD LOGGING HERE ***
                console.log('[TopFinder] Listener already attached to tab:', tab);
            }
        });
    }

    /**
     * Initializes SlimSelect for a given select element with consistent configuration.
     * This method is idempotent and can be safely called multiple times.
     * @param {HTMLSelectElement} element The select element to initialize SlimSelect on.
     * @private
     */
    private _initializeSlimSelect(element: HTMLSelectElement): void {
        // Destroy existing SlimSelect instance if it exists to prevent duplicates
        if ((element as any).slim) {
            (element as any).slim.destroy();
        }

        new SlimSelect({
            select: element,
            settings: {
                searchText: 'Search...',
                searchPlaceholder: 'Search options',
                searchHighlight: true,
                closeOnSelect: true,
                showSearch: true,
                searchFilter: (option, search) => {
                    return option.text.toLowerCase().includes(search.toLowerCase());
                }
            }
        });
    }

    /**
     * Handles dynamic content updates by reinitializing SlimSelect on new elements.
     * This method ensures that SlimSelect is properly initialized on all elements
     * with the top-finder-slimselect class, including those added dynamically.
     * @private
     */
    private _handleDynamicContent(): void {
        // First, select all elements with the top-finder-slimselect class
        const slimSelectElements = this.el.querySelectorAll('.top-finder-slimselect') as NodeListOf<HTMLSelectElement>;

        // Process each element
        slimSelectElements.forEach(element => {
            // Destroy existing SlimSelect instance if it exists to prevent duplicates
            if ((element as any).slim) {
                (element as any).slim.destroy();
            }
            // Initialize SlimSelect with our standard configuration
            this._initializeSlimSelect(element);
        });
    }

    /**
     * Handle clicks on tabs that load content via AJAX
     * @param {Event} event
     * @private
     */
    onAjaxTabClick(event) {
        const tab = event.currentTarget as HTMLElement;
        const className = 'top-finder-brand-devices-load-tab';

        // Prevent multiple clicks if already processing
        if (!tab.classList.contains(className)) {
            return;
        }
        // Prevent default anchor behavior if needed, although Bootstrap tabs might handle this
        event.preventDefault();

        tab.classList.remove(className); // Remove class immediately

        const path = DomAccess.getAttribute(tab, 'data-path');
        const tabPaneSelector = DomAccess.getAttribute(tab, 'href'); // e.g., "#brandXXX-tab-pane"

        if (!path || !tabPaneSelector) {
            console.error('[TopFinder] Missing data-path or href attribute on AJAX tab', tab);
            return;
        }

        // Find the container within the corresponding tab pane
        // Search the whole document as the pane might not be inside this.el
        const tabPane = document.querySelector(tabPaneSelector);
        if (!tabPane) {
            console.error('[TopFinder] Could not find tab pane for selector:', tabPaneSelector);
            return;
        }

        const container = DomAccess.querySelector(tabPane, '.topfinder-devices-compact', false);
        if (!container) {
            console.error('[TopFinder] Could not find .topfinder-devices-compact container in tab pane:', tabPaneSelector);
            // Optionally, show an error message in the tab pane
            tabPane.innerHTML = '<p>Error: Content container not found.</p>';
            return;
        }

        // Ensure loading indicator is visible (it should be there from Twig)
        // container.innerHTML = 'Loading...'; // Or manage spinner visibility

        // *** ADD LOGGING HERE ***
        console.log('[TopFinder] Attempting AJAX load:', { path: path, targetPane: tabPaneSelector });

        this._showSpinner();
        this._client.get(path, (responseText, request) => {
            try {
                if (request.status >= 400) {
                    console.error('[TopFinder] Error loading AJAX tab content:', request.status, responseText);
                    container.innerHTML = `<p>Error loading content (${request.status}). Please try again later.</p>`;
                    // Optionally re-add the class so the user can try again?
                    // tab.classList.add(className);
                    return;
                }
                const response = JSON.parse(responseText);
                if (response.success === true && typeof response.html !== 'undefined') {
                    container.innerHTML = response.html;
                    // Reinitialize Shopware plugins within the newly loaded content
                    // Cast window to any to satisfy TypeScript about PluginManager
                    (window as any).PluginManager.initializePlugins();
                } else {
                    console.error('[TopFinder] AJAX tab response unsuccessful or missing HTML:', response);
                    container.innerHTML = '<p>Error: Invalid response from server.</p>';
                }
            } catch (error) {
                console.error('[TopFinder] Error parsing AJAX tab response:', error, responseText);
                container.innerHTML = '<p>Error: Could not process server response.</p>';
            } finally {
                this._hideSpinner();
            }
        });
    }

    /**
     * Close the selectbox and set a cookie to remember this state
     * @param {Event} event - The event object
     */
    closeSelectbox(event) {
        let element = DomAccess.querySelector(this.el, '.top-finder-selectboxes')
        element.classList.add('top-finder-selectboxes-hidden')

        CookieStorage.setItem('topdata-device-selectboxes', 'hidden', 30);
        return false;
    }

    /**
     * Open the selectbox and remove the cookie that remembers the closed state
     * @param {Event} event - The event object
     */
    openSelectbox(event) {
        let element = DomAccess.querySelector(this.el, '.top-finder-selectboxes')
        element.classList.remove('top-finder-selectboxes-hidden')

        CookieStorage.setItem('topdata-device-selectboxes', '', 30);
        return false;
    }


    /**
     * @param {Event} event
     */
    /**
     * Handle the change event for the switch selection
     * @param {Event} event - The change event object
     */
    onChangeSwitchSelection(event) {
        event.preventDefault();
        console.log((event.currentTarget as HTMLSelectElement).value);

        CookieStorage.setItem('topdata-switch', (event.currentTarget as HTMLSelectElement).value, 30);

        if ((event.currentTarget as HTMLSelectElement).value === 'types') {
            (DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.seriesContainer) as HTMLElement).style.display = 'none';
            (DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.typeContainer) as HTMLElement).style.display = 'block';
        } else {
            (DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.typeContainer) as HTMLElement).style.display = 'none';
            (DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.seriesContainer) as HTMLElement).style.display = 'block';
        }
    }


    /**
     * Open a modal with the given content
     * @param {string} content - The content to display in the modal
     */
    openModal(content) {
        this.modal = new PseudoModalUtil(content);

        // open the modal window and make it visible
        this.modal.open();
    }

    /**
     * Handle the change event for brand selection
     * @param {Event} event - The change event object
     */
    async onChangeBrandSelection(event) {
        event.preventDefault();
        this._isUpdatingProgrammatically = true; // Set flag to prevent chain reactions

        let path;
        this.BrandId = (event.currentTarget as HTMLSelectElement).value;

        const seriesSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.seriesSelector, false) as HTMLSelectElement;
        const typeSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.typeSelector, false) as HTMLSelectElement;
        const deviceSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.deviceSelector) as HTMLSelectElement;

        const promises = [];

        if (typeSelector) {
            path = (event.currentTarget as HTMLElement).dataset.pathloadtypes;
            path = path
                .replace('brandcode', this.BrandId || '0')
                .replace('typecode', this.TypeId || '0')
                .replace('seriescode', this.SeriesId || '0');
            this.resetSelectOptions(typeSelector);
            promises.push(this.loadNewSelectOptions(path, typeSelector));
        }

        if (seriesSelector) {
            path = (event.currentTarget as HTMLElement).dataset.pathloadseries;
            path = path
                .replace('brandcode', this.BrandId || '0')
                .replace('typecode', this.TypeId || '0')
                .replace('seriescode', this.SeriesId || '0');
            this.resetSelectOptions(seriesSelector);
            promises.push(this.loadNewSelectOptions(path, seriesSelector));
        }

        if (deviceSelector) {
            path = (event.currentTarget as HTMLElement).dataset.pathloaddevices;
            path = path
                .replace('brandcode', this.BrandId || '0')
                .replace('typecode', this.TypeId || '0')
                .replace('seriescode', this.SeriesId || '0');
            this.resetSelectOptions(deviceSelector);
            promises.push(this.loadNewSelectOptions(path, deviceSelector));
        }

        try {
            await Promise.all(promises);
        } catch (error) {
            console.error("An error occurred while updating dropdowns:", error);
        } finally {
            this._isUpdatingProgrammatically = false; // Reset flag after all updates are done
        }
    }

    /**
     * Handle the change event for type selection
     * @param {Event} event - The change event object
     */
    onChangeTypeSelection(event) {
        if (this._isUpdatingProgrammatically) {
            return; // Prevent chain reaction from programmatic update
        }

        event.preventDefault();
        let path;
        this.TypeId = (event.currentTarget as HTMLSelectElement).value;
        const deviceSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.deviceSelector) as HTMLSelectElement;

        if (deviceSelector) {
            path = (event.currentTarget as HTMLElement).dataset.pathloaddevices;
            path = path
                .replace('brandcode', this.BrandId || '0')
                .replace('typecode', this.TypeId || '0')
                .replace('seriescode', this.SeriesId || '0');

            this.resetSelectOptions(deviceSelector);
            this.loadNewSelectOptions(path, deviceSelector);
        }
    }

    /**
     * Handle the change event for series selection
     * @param {Event} event - The change event object
     */
    onChangeSeriesSelection(event) {
        if (this._isUpdatingProgrammatically) {
            return; // Prevent chain reaction from programmatic update
        }

        event.preventDefault();
        let path;
        this.SeriesId = (event.currentTarget as HTMLSelectElement).value;
        const deviceSelector = DomAccess.querySelector(this.el, TopdataTopFinderProFinder.options.deviceSelector) as HTMLSelectElement;

        if (deviceSelector) {
            path = (event.currentTarget as HTMLElement).dataset.pathloaddevices;
            path = path
                .replace('brandcode', this.BrandId || '0')
                .replace('typecode', this.TypeId || '0')
                .replace('seriescode', this.SeriesId || '0');

            this.resetSelectOptions(deviceSelector);
            this.loadNewSelectOptions(path, deviceSelector);
        }
    }


    /**
     * Handle the change event for device selection
     * @param {Event} event - The change event object
     */
    onChangeDeviceSelection(event) {
        event.preventDefault();
        let path;
        let DeviceId = (event.currentTarget as HTMLSelectElement).value;

        if (DeviceId !== '') {
            path = (event.currentTarget as HTMLElement).dataset.pathgotodevice;
            path = path.replace('devicecode', DeviceId);
            window.location.href = path;
        }
        return false;
    }


    /**
     * Reset the options of a select element, keeping only the first option
     * @param {HTMLSelectElement} selectBox - The select element to reset
     * @returns {HTMLSelectElement} The reset select element
     */
    resetSelectOptions(selectBox: HTMLSelectElement) {
        if (selectBox && selectBox.options.length > 1) {
            while (selectBox.options.length > 1) {
                selectBox.remove(1);
            }
        }
        return selectBox;
    }

    /**
     * Show the spinner overlay if mode is 'overlay'
     * @private
     */
    _showSpinner() {
        console.log('TopFinder: Showing spinner overlay');
        console.log('Spinner mode:', this.ajaxSpinnerMode, 'Overlay element:', !!this.spinnerOverlay);
        if (this.ajaxSpinnerMode === 'overlay' && this.spinnerOverlay) {
            console.log('TopFinder: Setting overlay display to flex');
            this.spinnerOverlay.style.display = 'flex';
        }
    }

    /**
     * Hide the spinner overlay if mode is 'overlay'
     * @private
     */
    _hideSpinner() {
        console.log('TopFinder: Hiding spinner overlay');
        console.log('Spinner mode:', this.ajaxSpinnerMode, 'Overlay element:', !!this.spinnerOverlay);
        if (this.ajaxSpinnerMode === 'overlay' && this.spinnerOverlay) {
            console.log('TopFinder: Setting overlay display to none');
            this.spinnerOverlay.style.display = 'none';
        }
    }


    /**
     * Load new options for a select element from a given URL
     * @param {string} url - The URL to fetch new options from
     * @param {HTMLSelectElement} selectElement - The select element to populate with new options
     */
    loadNewSelectOptions(url: string, selectElement: HTMLSelectElement): Promise<void> {
        return new Promise((resolve, reject) => {
            this._showSpinner();
            this._client.get(url, (responseText, request) => {
                try {
                    if (request.status >= 400) {
                        console.error(`HTTP Error ${request.status}:`, responseText);
                        resolve(); // Resolve anyway to not break Promise.all
                        return;
                    }
                    const response = JSON.parse(responseText);
                    if (response.found === true) {
                        Object.entries(response.items).forEach(entry => {
                            const [key, value] = entry;
                            let newOption = new Option(String(value), key);
                            selectElement.add(newOption, undefined);
                        });
                    }
                    resolve();
                } catch (error) {
                    console.error("Failed to parse response or update select options:", error);
                    reject(error); // Reject on actual errors
                } finally {
                    this._hideSpinner();
                }
            });
        });
    }

}