<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder;

use Doctrine\DBAL\Connection;
use Shopware\Core\Content\Category\Exception\CategoryNotFoundException;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\ContainsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataConnectorSW6\Core\Content\Device\Agregate\DeviceCustomer\DeviceCustomerEntity;
use Topdata\TopdataConnectorSW6\Core\Content\Device\DeviceCollection;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\DeviceList\DeviceListPage;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\DeviceList\DeviceListPageLoadedEvent;

/**
 * Class DeviceListPageLoader
 *
 * This class is responsible for loading and managing device list pages,
 * including functionality for searching, adding, editing, and removing devices.
 */
class DeviceListPageLoader
{
    /**
     * This property holds the list of devices for the current customer.
     * It is initialized to null and populated when the device list is fetched.
     * it is actually a dict/map [with device ID as key], not a list
     */
    private ?array $deviceList = null;

    /**
     * This property holds the list of devices filtered by a keyword for the current customer.
     * It is initialized to null and populated when the device list is fetched with a keyword.
     * it is actually a dict/map [with device ID as key], not a list
     */
    private ?array $deviceListKeyworded = null;

    public function __construct(
        private readonly GenericPageLoader $genericLoader,
        private readonly EventDispatcherInterface $eventDispatcher,
        private readonly EntityRepository $topdataDeviceRepository,
        private readonly Connection $connection,
        private readonly SettingsService $settingsService,
        private readonly Translator $translator,
        private readonly UrlGeneratorInterface $router,
        private readonly EntityRepository $topdataDeviceToCustomerRepository
    ) {
    }

    /**
     * Load the device list page with optional keyword search
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return DeviceListPage
     * @throws CategoryNotFoundException
     * @throws InconsistentCriteriaIdsException
     */
    public function loadPage(Request $request, SalesChannelContext $salesChannelContext): DeviceListPage
    {
        $keyword = trim($request->get('keyword', ''));

        $page = DeviceListPage::createFrom($this->genericLoader->load($request, $salesChannelContext));
        $page->setDeviceList($this->getDeviceList($salesChannelContext, $keyword));
        $page->setDevices($this->getDevices($salesChannelContext, $keyword));

        $page->devicesCount = $this->countDevices($salesChannelContext, $page->getDevices(), $keyword);

        //        $page->properies = $this->entitiesHelper->getPropertyGroupsOptionsArray();

        if ($this->settingsService->getInt('deviceListCompactLimit', true)) {
            $page->setCompactModeLimit($this->settingsService->getInt('deviceListCompactLimit'));
        }

        $this->eventDispatcher->dispatch(
            new DeviceListPageLoadedEvent($page, $salesChannelContext, $request)
        );

        return $page;
    }

    /**
     * Load the device list page in JSON format
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return DeviceListPage
     */
    public function loadJson(Request $request, SalesChannelContext $salesChannelContext): DeviceListPage
    {
        $page        = $this->loadPage($request, $salesChannelContext);
        $page->title = $this->translator->trans('topdata-topfinder.popup.myDeviceList');

        $page->popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.myDeviceList'),
//            'path' => $this->router->generate('frontend.top-finder-api.popup-all-brands')
        ];

        return $page;
    }

    /**
     * Load a specific device's details in JSON format
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return DeviceListPage
     */
    public function loadDeviceJson(Request $request, SalesChannelContext $salesChannelContext): DeviceListPage
    {
        $page = $this->loadPage($request, $salesChannelContext);

        $id = (string)$request->get('id');

        $page->device = $page->getDevices()->get($id);

        if (!$page->device) {
            return $page;
        }

        $page->title = '<b>' . $page->device->brand->name . '</b> ' . $page->device->model;

        $page->popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.myDeviceList'),
            'path' => $this->router->generate('frontend.top-finder-api.popup-devicelist'),
        ];

        $page->popupPath[] = [
            'name' => $page->title,
//            'path' => $this->router->generate('frontend.top-finder-api.popup-all-brands')
        ];

        return $page;
    }

    /**
     * Load the page for adding a subdevice in JSON format
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return DeviceListPage
     */
    public function loadAddSubdeviceJson(Request $request, SalesChannelContext $salesChannelContext): DeviceListPage
    {
        $page = $this->loadPage($request, $salesChannelContext);

        $id = (string)$request->get('id');

        $page->device = $page->getDevices()->get($id);

        if (!$page->device) {
            return $page;
        }

        $page->title = $this->translator->trans('topdata-topfinder.popup.addSubdevice');

        $page->popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.myDeviceList'),
            'path' => $this->router->generate('frontend.top-finder-api.popup-devicelist'),
        ];

        $page->popupPath[] = [
            'name' => $page->device->brand->name . ' ' . $page->device->model,
            'path' => $this->router->generate('frontend.top-finder-api.popup-devicelist-device', ['id' => $id]),
        ];

        $page->popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.addSubdevice'),
        ];

        return $page;
    }

    /**
     * Add a subdevice and return the updated page in JSON format
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return DeviceListPage
     */
    public function addSubdeviceJson(Request $request, SalesChannelContext $salesChannelContext): DeviceListPage
    {
        $page = $this->loadPage($request, $salesChannelContext);

        $id = (string)$request->get('id');

        $page->device = $page->getDevices()->get($id);

        if (!$page->device) {
            return $page;
        }

        $userId = $salesChannelContext->getCustomer()->getId();

        $criteria = (new Criteria())
            ->addFilter(new EqualsFilter('isDealerManaged', false))
            ->addFilter(new EqualsFilter('customerId', $userId))
            ->addFilter(new EqualsFilter('deviceId', $id));

        $deviceCustomer = $this
            ->topdataDeviceToCustomerRepository
            ->search($criteria, $salesChannelContext->getContext())
            ->first();

        if (!$deviceCustomer) {
            return $page;
        }

        $extraInfo              = $deviceCustomer->getExtraInfo();
        $extraInfo['devices'][] = [
            DeviceCustomerEntity::DEVICE_NAME     => $request->get('name'),
            DeviceCustomerEntity::DEVICE_NUMBER   => $request->get('number'),
            DeviceCustomerEntity::DEVICE_LOCATION => $request->get('location'),
            DeviceCustomerEntity::USER            => $request->get('user'),
            DeviceCustomerEntity::DEVICE_NOTES    => $request->get('notes'),
            DeviceCustomerEntity::DEVICE_TIME     => date('Y-m-d H:i:s'),
        ];

        $this->topdataDeviceToCustomerRepository->update([
            [
                'id'        => $deviceCustomer->getId(),
                'extraInfo' => json_encode($extraInfo),
            ],
        ], $salesChannelContext->getContext());

        $this->deviceList = null;

        $page = $this->loadDeviceJson($request, $salesChannelContext);

        return $page;
    }

    /**
     * Get device IDs based on the given context and keyword
     *
     * @param SalesChannelContext $context
     * @param string $keyword
     * @param bool $keyworded
     * @return array
     */
    private function getDeviceIds(SalesChannelContext $context, string $keyword = '', bool $keyworded = false): array
    {
        return array_keys($this->getDeviceList($context, $keyword, $keyworded));
    }

    /**
     * Count the number of devices based on the given context, devices, and keyword
     *
     * @param SalesChannelContext $context
     * @param EntitySearchResult $devices
     * @param string $keyword
     * @return int
     */
    private function countDevices(
        SalesChannelContext $context,
        EntitySearchResult $devices,
        string $keyword = ''
    ): int {
        $return     = 0;
        $deviceList = $this->getDeviceList($context, $keyword);
        foreach ($devices as $device) {
            foreach ($deviceList as $id => $info) {
                if ($device->getId() === $id && isset($info['devices'])) {
                    $return += count($info['devices']);
                }
            }
        }

        return $return;
    }

    /**
     * Get the device list based on the given context, keyword, and keyworded flag
     *
     * @param SalesChannelContext $context
     * @param string $keyword
     * @param bool $keyworded
     * @return array
     */
    private function getDeviceList(SalesChannelContext $context, string $keyword = '', bool $keyworded = false): array
    {
        if (!$context->getCustomer() || $context->getCustomer()->getGuest()) {
            $this->deviceList = [];
        } elseif (null === $this->deviceList) {
            $this->deviceList          = [];
            $this->deviceListKeyworded = [];
            if ($keyword) {
                $rez = $this->connection
                    ->createQueryBuilder()
                    ->select('LOWER(HEX(device_id)) as device_id, extra_info, created_at, updated_at')
                    ->from('topdata_device_to_customer')
                    ->where('(customer_id = 0x' . $context->getCustomer()->getId() . ") AND (extra_info LIKE '%$keyword%' )")
                    ->executeQuery()
                    ->fetchAllAssociative();

                foreach ($rez as $val) {
                    $this->deviceListKeyworded[$val['device_id']]              = json_decode($val['extra_info'], true);
                    $this->deviceListKeyworded[$val['device_id']]['createdAt'] = date('d.m.Y H:i', strtotime($val['created_at']));
                    if ($val['updated_at']) {
                        $this->deviceListKeyworded[$val['device_id']]['updatedAt'] = date('d.m.Y H:i', strtotime($val['updated_at']));
                    }
                }
            }

            $rez = $this->connection
                ->createQueryBuilder()
                ->select('LOWER(HEX(device_id)) as device_id, extra_info, created_at, updated_at')
                ->from('topdata_device_to_customer')
                ->where('customer_id = 0x' . $context->getCustomer()->getId())
                ->executeQuery()
                ->fetchAllAssociative();
            foreach ($rez as $val) {
                $this->deviceList[$val['device_id']]              = $val['extra_info'] ? json_decode($val['extra_info'], true) : DeviceCustomerEntity::defaultExtraInfo();
                $this->deviceList[$val['device_id']]['createdAt'] = date('d.m.Y H:i', strtotime($val['created_at']));
                if ($val['updated_at']) {
                    $this->deviceList[$val['device_id']]['updatedAt'] = date('d.m.Y H:i', strtotime($val['updated_at']));
                }
            }
        }

        if ($keyword && $keyworded) {
            return $this->deviceListKeyworded;
        }

        return $this->deviceList;
    }

    /**
     * Get devices based on the given context and keyword
     *
     * @param SalesChannelContext $context
     * @param string $keyword
     * @return EntitySearchResult
     */
    private function getDevices(SalesChannelContext $context, string $keyword = ''): EntitySearchResult
    {
        if ($keyword && !$this->getDeviceIds($context, $keyword, true) && $this->getDeviceIds($context, $keyword)) {
            $criteria = (new Criteria($this->getDeviceIds($context, $keyword)))
                ->addFilter(new EqualsFilter('enabled', true))
                ->addFilter(new ContainsFilter('keywords', $keyword))
                ->addAssociations(['media', 'brand', 'series', 'type'])
                ->addSorting(new FieldSorting('brand.name', FieldSorting::ASCENDING))
                ->addSorting(new FieldSorting('model', FieldSorting::ASCENDING));

            $devices = $this->topdataDeviceRepository->search($criteria, $context->getContext());
            foreach ($devices as $device) {
                $device->setInDeviceList(true);
            }
        } elseif ($keyword && $this->getDeviceIds($context, $keyword)) {
            $criteria = (new Criteria($this->getDeviceIds($context, $keyword, true)))
                ->addFilter(new EqualsFilter('enabled', true))
                ->addAssociations(['media', 'brand', 'series', 'type'])
                ->addSorting(new FieldSorting('brand.name', FieldSorting::ASCENDING))
                ->addSorting(new FieldSorting('model', FieldSorting::ASCENDING));

            $devices = $this->topdataDeviceRepository->search($criteria, $context->getContext());
            foreach ($devices as $device) {
                $device->setInDeviceList(true);
            }
        } elseif ($this->getDeviceIds($context, $keyword)) {
            $criteria = (new Criteria($this->getDeviceIds($context, $keyword)))
                ->addFilter(new EqualsFilter('enabled', true))
                ->addAssociations(['media', 'brand', 'series', 'type'])
                ->addSorting(new FieldSorting('brand.name', FieldSorting::ASCENDING))
                ->addSorting(new FieldSorting('model', FieldSorting::ASCENDING));

            $devices = $this->topdataDeviceRepository->search($criteria, $context->getContext());
            foreach ($devices as $device) {
                $device->setInDeviceList(true);
            }
        } else {
            $devices = new EntitySearchResult('devicelist', 0, new DeviceCollection(), null, new Criteria(), $context->getContext());
        }

        return $devices;
    }

    /**
     * Load the page for editing a subdevice in JSON format
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return DeviceListPage
     */
    public function loadEditSubdeviceJson(Request $request, SalesChannelContext $salesChannelContext): DeviceListPage
    {
        $page = $this->loadPage($request, $salesChannelContext);

        $id    = (string)$request->get('id');
        $subid = (int)$request->get('subid');

        $page->device = $page->getDevices()->get($id);
        $deviceInfo   = isset($page->getDeviceList()[$id]) ? $page->getDeviceList()[$id] : null;
        if (!$page->device || is_null($deviceInfo) || !isset($deviceInfo['devices'][$subid])) {
            return $page;
        }

        $page->subdevice       = $deviceInfo['devices'][$subid];
        $page->subdevice['id'] = $subid;

        $page->title = (isset($page->subdevice['name']) && $page->subdevice['name'])
            ? $page->subdevice['name']
            : $this->translator->trans('topdata-topfinder.popup.device') . ' ' . ($subid + 1);

        $page->popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.myDeviceList'),
            'path' => $this->router->generate('frontend.top-finder-api.popup-devicelist'),
        ];

        $page->popupPath[] = [
            'name' => $page->device->brand->name . ' ' . $page->device->model,
            'path' => $this->router->generate('frontend.top-finder-api.popup-devicelist-device', ['id' => $id]),
        ];

        $page->popupPath[] = [
            'name' => $page->title,
        ];

        return $page;
    }

    /**
     * Edit a subdevice and return the updated page in JSON format
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return DeviceListPage
     */
    public function editSubdeviceJson(Request $request, SalesChannelContext $salesChannelContext): DeviceListPage
    {
        $page = $this->loadPage($request, $salesChannelContext);

        $id    = (string)$request->get('id');
        $subid = (int)$request->get('subid');

        $page->device = $page->getDevices()->get($id);
        $deviceInfo   = isset($page->getDeviceList()[$id]) ? $page->getDeviceList()[$id] : null;
        if (!$page->device || is_null($deviceInfo) || !isset($deviceInfo['devices'][$subid])) {
            return $page;
        }

        $page->subdevice = $deviceInfo['devices'][$subid];

        $userId = $salesChannelContext->getCustomer()->getId();

        $criteria = (new Criteria())
            ->addFilter(new EqualsFilter('isDealerManaged', false))
            ->addFilter(new EqualsFilter('customerId', $userId))
            ->addFilter(new EqualsFilter('deviceId', $id));

        $deviceCustomer = $this
            ->topdataDeviceToCustomerRepository
            ->search($criteria, $salesChannelContext->getContext())
            ->first();

        if (!$deviceCustomer) {
            return $page;
        }

        $extraInfo = $deviceCustomer->getExtraInfo();

        if (!isset($extraInfo['devices'][$subid])) {
            return $page;
        }

        $oldData = $extraInfo['devices'][$subid];
        if (isset($oldData[DeviceCustomerEntity::DEVICE_TIME])) {
            unset($oldData[DeviceCustomerEntity::DEVICE_TIME]);
        }

        $newData = [
            DeviceCustomerEntity::DEVICE_NAME     => $request->get('name'),
            DeviceCustomerEntity::DEVICE_NUMBER   => $request->get('number'),
            DeviceCustomerEntity::DEVICE_LOCATION => $request->get('location'),
            DeviceCustomerEntity::USER            => $request->get('user'),
            DeviceCustomerEntity::DEVICE_NOTES    => $request->get('notes'),
//            DeviceCustomerEntity::DEVICE_TIME => date('Y-m-d H:i:s')
        ];

        if (json_encode($oldData) === json_encode($newData)) {
            $this->deviceList = null;
            $page             = $this->loadEditSubdeviceJson($request, $salesChannelContext);

            return $page;
        }

        $newData[DeviceCustomerEntity::DEVICE_TIME] = date('Y-m-d H:i:s');

        $extraInfo['devices'][$subid] = $newData;

        $this->topdataDeviceToCustomerRepository->update([
            [
                'id'        => $deviceCustomer->getId(),
                'extraInfo' => json_encode($extraInfo),
            ],
        ], $salesChannelContext->getContext());

        $this->deviceList = null;

        $page = $this->loadEditSubdeviceJson($request, $salesChannelContext);

        return $page;
    }

    /**
     * Remove a subdevice and return the updated page in JSON format
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return DeviceListPage
     */
    public function removeSubdeviceJson(Request $request, SalesChannelContext $salesChannelContext): DeviceListPage
    {
        $page = $this->loadPage($request, $salesChannelContext);

        $id    = (string)$request->get('id');
        $subid = (int)$request->get('subid');

        $page->device = $page->getDevices()->get($id);
        $deviceInfo   = isset($page->getDeviceList()[$id]) ? $page->getDeviceList()[$id] : null;
        if (!$page->device || is_null($deviceInfo) || !isset($deviceInfo['devices'][$subid])) {
            return $page;
        }

        $page->subdevice = $deviceInfo['devices'][$subid];

        $userId = $salesChannelContext->getCustomer()->getId();

        $criteria = (new Criteria())
            ->addFilter(new EqualsFilter('isDealerManaged', false))
            ->addFilter(new EqualsFilter('customerId', $userId))
            ->addFilter(new EqualsFilter('deviceId', $id));

        $deviceCustomer = $this
            ->topdataDeviceToCustomerRepository
            ->search($criteria, $salesChannelContext->getContext())
            ->first();

        if (!$deviceCustomer) {
            return $page;
        }

        $extraInfo = $deviceCustomer->getExtraInfo();
        unset($extraInfo['devices'][$subid]);

        //        if(count($extraInfo['devices'])) {
        //remove only subdevice
        $extraInfo['devices'] = array_values($extraInfo['devices']);

        $this->topdataDeviceToCustomerRepository->update([
            [
                'id'        => $deviceCustomer->getId(),
                'extraInfo' => json_encode($extraInfo),
            ],
        ], $salesChannelContext->getContext());

        $this->deviceList = null;
        $page             = $this->loadDeviceJson($request, $salesChannelContext);
        //        }
        //        else {
        //            //remove device from devicelist
        //            $this->topdataDeviceToCustomerRepository->delete([['id'=>$deviceCustomer->getId()]], $salesChannelContext->getContext());
        //            $this->deviceList = null;
        //            $page = $this->loadJson($request, $salesChannelContext);
        //            $page->fallBack = true;
        //        }

        return $page;
    }
}
