<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Service;

use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Topdata\TopdataTopFinderProSW6\Core\Content\TopdataSeries\TopdataSeriesEntity;

/**
 * Class SeriesDataService
 *
 * This service is responsible for handling series data operations.
 * It provides methods for retrieving and processing series data.
 */
class SeriesDataService
{
    public function __construct(
        private readonly EntityRepository $topdataSeriesRepository,
        private readonly LoggerInterface $logger
    ) {
    }

    /**
     * Get a series by its code
     *
     * This method retrieves a specific series entity based on the provided code.
     * It returns the series entity if found, or null if not found.
     * The brand association is included in the returned entity.
     *
     * @param string $code The series code to search for
     * @param SalesChannelContext $context The sales channel context
     * @return TopdataSeriesEntity|null The series entity or null if not found
     */
    public function getSeriesByCode(string $code, SalesChannelContext $context): ?TopdataSeriesEntity
    {
        try {
            $criteria = new Criteria();
            $criteria->addFilter(new EqualsFilter('enabled', true));
            $criteria->addFilter(new EqualsFilter('code', $code));
            $criteria->addAssociation('brand');

            $series = $this->topdataSeriesRepository->search($criteria, $context->getContext())->first();

            return $series;
        } catch (\Exception $e) {
            $this->logger->error('Error fetching series by code: ' . $e->getMessage(), [
                'code' => $code
            ]);
            return null;
        }
    }

    /**
     * Get all series for a specific brand
     *
     * This method retrieves all series entities associated with the provided brand ID.
     * The results are sorted by the 'label' field in ascending order.
     *
     * @param string $brandId The brand ID to get series for
     * @param SalesChannelContext $context The sales channel context
     * @return EntitySearchResult The search result containing series entities
     */
    public function getSeriesForBrand(string $brandId, SalesChannelContext $context): EntitySearchResult
    {
        try {
            $criteria = new Criteria();
            $criteria->addFilter(new EqualsFilter('enabled', true));
            $criteria->addFilter(new EqualsFilter('brandId', $brandId));
            $criteria->addSorting(new FieldSorting('label', FieldSorting::ASCENDING));

            return $this->topdataSeriesRepository->search($criteria, $context->getContext());
        } catch (\Exception $e) {
            $this->logger->error('Error fetching series for brand: ' . $e->getMessage(), [
                'brandId' => $brandId
            ]);
            
            // Return an empty result in case of error
            return $this->topdataSeriesRepository->search(new Criteria(), $context->getContext());
        }
    }
}