<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Controller;

use Psr\Log\LoggerInterface;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Controller\StorefrontController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Throwable;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\OneBrandLetterPageLoader;
use Topdata\TopdataTopFinderProSW6\Util\ControllerUtil;

/**
 * Controller for handling TopFinder device-related API requests.
 *
 * This controller provides endpoints for retrieving devices, series, and types
 * for a specific brand.
 *
 * 04/2025 created (extracted from TopFinderApiController)
 */
#[Route(defaults: ['_routeScope' => ['storefront']])]
class ApiController_TopdataDevice extends StorefrontController
{
    private const VIEW_PATHS = [
        'devices' => '@Storefront/storefront/page/topfinder/one-brand-letter-devices.html.twig',
        'series'  => '@Storefront/storefront/page/topfinder/one-brand-letter-series.html.twig',
        'types'   => '@Storefront/storefront/page/topfinder/one-brand-letter-types.html.twig',
    ];

    private const CACHE_DURATION = 3600; // 1 hour

    public function __construct(
        private readonly OneBrandLetterPageLoader $oneBrandLetterPageLoader,
        private readonly LoggerInterface          $logger
    )
    {
    }

    /**
     * List devices for a specific brand.
     *
     * @param string $brandCode The code of the brand
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response containing the HTML for the brand devices list
     */
    #[Route(
        path: '/top-finder-api/brand/{brandCode}/devices',
        name: 'frontend.top-finder-api.brand-devices-list',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET', 'POST']
    )]
    public function listDevices(string $brandCode, Request $request, SalesChannelContext $context): JsonResponse
    {
        return $this->listEntities('devices', $brandCode, $request, $context);
    }

    /**
     * List series for a specific brand.
     *
     * @param string $brandCode The code of the brand
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response containing the HTML for the brand series list
     */
    #[Route(
        path: '/top-finder-api/brand/{brandCode}/series',
        name: 'frontend.top-finder-api.brand-series-list',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET', 'POST']
    )]
    public function listSeries(string $brandCode, Request $request, SalesChannelContext $context): JsonResponse
    {
        return $this->listEntities('series', $brandCode, $request, $context);
    }

    /**
     * List types for a specific brand.
     *
     * @param string $brandCode The code of the brand
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response containing the HTML for the brand types list
     */
    #[Route(
        path: '/top-finder-api/brand/{brandCode}/types',
        name: 'frontend.top-finder-api.brand-types-list',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET', 'POST']
    )]
    public function listTypes(string $brandCode, Request $request, SalesChannelContext $context): JsonResponse
    {
        return $this->listEntities('types', $brandCode, $request, $context);
    }

    /**
     * Generic method to list entities for a brand.
     *
     * @param string $entityType The type of entity (devices, series, types)
     * @param string $brandCode The code of the brand
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response containing the HTML for the entities list
     */
    private function listEntities(string $entityType, string $brandCode, Request $request, SalesChannelContext $context): JsonResponse
    {
        ControllerUtil::assertValidBrandcode($brandCode);

        // Get view path from the map
        $view = self::VIEW_PATHS[$entityType] ?? throw new \InvalidArgumentException("Invalid entity type: {$entityType}");

        // Load page data
        $page = $this->_loadPageData($brandCode, $request, $context);

        // Render view
        $html = $this->_renderEntityView($view, $page);

        // Prepare response
        return $this->_createCachedResponse(['success' => true, 'html' => $html]);
    }

    /**
     * Load page data using the page loader.
     *
     * @param string $brandCode The code of the brand
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return mixed The loaded page object
     * @throws Throwable If loading fails
     */
    private function _loadPageData(string $brandCode, Request $request, SalesChannelContext $context): mixed
    {
        try {
            return $this->oneBrandLetterPageLoader->load($request, $context, $brandCode);
        } catch (Throwable $e) {
            $this->logger->error('Failed to load page data', [
                'brandCode' => $brandCode,
                'error'     => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Render the entity view with the provided page data.
     *
     * Note: Named differently from parent class's protected renderView method
     *
     * @param string $view The view path
     * @param mixed $page The page data
     * @return string The rendered HTML
     * @throws Throwable If rendering fails
     */
    private function _renderEntityView(string $view, mixed $page): string
    {
        return $this->renderStorefront($view, ['page' => $page])->getContent();
    }

    /**
     * Create a cached JSON response.
     *
     * @param array $data The response data
     * @return JsonResponse The JSON response with cache headers
     */
    private function _createCachedResponse(array $data): JsonResponse
    {
        $response = new JsonResponse($data);
        $response->setPublic();
        $response->setMaxAge(self::CACHE_DURATION);
        $response->setSharedMaxAge(self::CACHE_DURATION);

        return $response;
    }
}