<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Subscriber;

use Shopware\Core\Content\Product\SalesChannel\SalesChannelProductEntity;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\Product\ProductPage;
use Shopware\Storefront\Pagelet\Header\HeaderPageletLoadedEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\ContainsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Topdata\TopdataTopFinderProSW6\Component\Collection;
use Topdata\TopdataTopFinderProSW6\Constants\CookieKeyNameConstants;
use Topdata\TopdataTopFinderProSW6\Service\DeviceHistoryService_Cookie;
use Topdata\TopdataTopFinderProSW6\Service\DeviceToCustomerService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Shopware\Storefront\Page\Suggest\SuggestPageLoadedEvent;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Doctrine\DBAL\Connection;
use Shopware\Storefront\Page\Product\ProductPageLoadedEvent;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\Content\Category\CategoryEntity;
use Topdata\TopdataTopFinderProSW6\Service\HelperService;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Shopware\Core\Content\Category\Service\CategoryBreadcrumbBuilder;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\AndFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\OrFilter;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Topdata\TopdataProductMenu\Storefront\Page\ProductMenu\ProductMenuLoadedEvent;
use Shopware\Storefront\Event\ThemeCompilerEnrichScssVariablesEvent;
use TopdataSoftwareGmbH\Util\UtilDebug;

/**
 * BaseSubscriber class for handling various events in the Topdata TopFinder Pro plugin.
 *
 * This class subscribes to multiple events and provides functionality for
 * adding extensions to header, product pages, and handling device suggestions.
 */
class BaseSubscriber implements EventSubscriberInterface
{


    public function __construct(
        private readonly Translator                  $translator,
        private readonly SettingsService             $settingsService,
        private readonly EntityRepository            $topdataDeviceRepository,
        private readonly Connection                  $connection,
        private readonly SalesChannelRepository      $salesChannelCategoryRepository,
        private readonly UrlGeneratorInterface       $router,
        protected readonly SystemConfigService       $systemConfigService,
        private readonly DeviceToCustomerService     $deviceToCustomerService,
        private readonly CategoryBreadcrumbBuilder   $categoryBreadcrumbBuilder,
        private readonly DeviceHistoryService_Cookie $deviceHistoryService_Cookie,
    )
    {
    }

    /**
     * Returns an array of events this subscriber wants to listen to.
     *
     * @return array The event names to listen to
     */
    public static function getSubscribedEvents(): array
    {
        return [
            HeaderPageletLoadedEvent::class              => 'addHeaderExtensions',
            SuggestPageLoadedEvent::class                => 'addDeviceSuggest',
            ProductPageLoadedEvent::class                => 'addProductPageExtensions',
            ProductMenuLoadedEvent::class                => 'addMenuItems',
            ThemeCompilerEnrichScssVariablesEvent::class => 'addScssVariables',
        ];
    }

    /**
     * Adds SCSS variables to the theme compiler.
     *
     * @param ThemeCompilerEnrichScssVariablesEvent $event The event object
     */
    public function addScssVariables(ThemeCompilerEnrichScssVariablesEvent $event): void
    {
        $scssVariables = [
            'topdata-bg-dark',
            'topdata-bg-light',
            'topdata-bg-white',
            'topdata-bg-grey',
            'topdata-bg-device',
            'topdata-border-color',
            'topdata-interactive-grey',

            'topdata-text',
            'topdata-text-invert',
            'topdata-text-invert-inactive',
            'topdata-link-invert-hover',
        ];

        $map = [];

        foreach ($scssVariables as $item) {
            $temp = explode('-', $item);
            $camelCaseKey = 'color';
            foreach ($temp as $t) {
                if ($t == 'topdata') {
                    continue;
                }
                $camelCaseKey .= ucfirst($t);
            }
            $map[$camelCaseKey] = $item;
        }

        $configPlugin = $this->systemConfigService->get('TopdataTopFinderProSW6.config', $event->getSalesChannelId());

        foreach ($map as $configVariable => $scssVariable) {
            if (isset($configPlugin[$configVariable]) && $configPlugin[$configVariable]) {
                $event->addVariable($scssVariable, $configPlugin[$configVariable]);
            }
        }
    }

    /**
     * Adds menu items to the product menu.
     *
     * @param ProductMenuLoadedEvent $event The event object
     */
    public function addMenuItems(ProductMenuLoadedEvent $event): void
    {
        $rez = $this->connection->executeQuery('
        SELECT device_id
         FROM topdata_device_to_product
         WHERE product_id = 0x' . $event->getPage()->getProductId() . '
         LIMIT 1')->fetchAllAssociative();

        if (!count($rez)) {
            $parentId = $this->connection->executeQuery('
                SELECT LOWER(HEX(parent_id)) as parentId FROM `product`
                 WHERE id = 0x' . $event->getPage()->getProductId() . '
                 LIMIT 1')->fetchOne();
            if ($parentId) {
                $rez = $this->connection->executeQuery('
                    SELECT device_id
                     FROM topdata_device_to_product
                     WHERE product_id = 0x' . $parentId . '
                     LIMIT 1')->fetchAllAssociative();
            }
        }

        if (count($rez)) {
            $event->getPage()->setMenuItem(
                $this->translator->trans('topdata-topfinder.productDevices'),
                '#',
                [
                    'path' => $this->router->generate('frontend.top-finder-api.compatible-devices', ['productid' => $event->getPage()->getProductId()]),
                ],
                ['top-finder-product-devices-load-popup']
            );
        }

        $event->getPage()->setMenuItem(
            $this->translator->trans('topdata-topfinder.showInProductList'),
            $this->router->generate('frontend.top-finder.list-products', ['ids' => $event->getPage()->getProductId()])
        );
    }

    /**
     * Adds extensions to the product page.
     *
     * @param ProductPageLoadedEvent $event The event object
     */
    public function addProductPageExtensions(ProductPageLoadedEvent $event): void
    {
        $productPage = $event->getPage();
        $this->_addBreadcrumbsToProductPage($productPage, $event->getSalesChannelContext());
        $this->_addDevicesToProductPage($productPage);
    }

    /**
     * Adds breadcrumbs to the product page.
     *
     * @param ProductPageLoadedEvent $event The event object
     */
    private function _addBreadcrumbsToProductPage(ProductPage $productPage, SalesChannelContext $salesChannelContext): void
    {
        if (!$this->settingsService->getBool('fixProductBreadcrumbs')) {
            return;
        }

        $categories = $productPage->getProduct()->getCategoryTree();

        if (!is_array($categories)) {
            return;
        }

        $categoryId = array_pop($categories);

        if (!is_string($categoryId)) {
            return;
        }

        /** @var CategoryEntity $category */
        $category = $this
            ->salesChannelCategoryRepository
            ->search(new Criteria([$categoryId]), $salesChannelContext)
            ->getEntities()
            ->first();

        if (!$category) {
            return;
        }

        $breadcrumbs = $this->categoryBreadcrumbBuilder->build(
            $category,
            $salesChannelContext->getSalesChannel(),
            $salesChannelContext->getSalesChannel()->getNavigationCategoryId()
        );

        if (is_array($breadcrumbs)) {
            $struct = new Collection();
            $struct->set('categories', $breadcrumbs);
            $productPage->addExtension('breadcrumbs', $struct);
        }
    }

    /**
     * Adds device information to the product page which is shown in a tab
     *
     * @param ProductPageLoadedEvent $event The event object
     */
    private function _addDevicesToProductPage(ProductPage $productPage): void
    {
        if ($this->settingsService->getBool('showDevicesTab') === false) {
            return;
        }

        $bHasDevices = $this->_hasDevices($productPage->getProduct());

        if ($bHasDevices) {
            $struct = $productPage->getExtension('product_devices');
            if ($struct === null) {
                $struct = new Collection();
            }
            $struct->set('devices', true);
            $productPage->addExtension('product_devices', $struct);
        }
    }

    /**
     * Adds various extensions to the header pagelet.
     *
     * @param HeaderPageletLoadedEvent $event The event object
     */
    public function addHeaderExtensions(HeaderPageletLoadedEvent $event): void
    {
        $this->addDevicesHistory($event);
        $this->addActiveBrandsList($event);
        $this->addDevicesSlider($event);
        $this->addDeviceList($event);
        $this->addDeviceFinderPopup($event);
    }

    /**
     * Adds the device finder popup to the header pagelet.
     *
     * @param HeaderPageletLoadedEvent $event The event object
     */
    private function addDeviceFinderPopup(HeaderPageletLoadedEvent $event): void
    {
        $struct = $event->getPagelet()->getExtension('finderPopup');
        if ($struct === null) {
            $struct = new Collection();
        }

        $struct->set('popupShowButton', $this->settingsService->getBool('popupShowButton'));
        $struct->set('popupSelectboxesMode', $this->settingsService->getString('popupSelectboxesMode'));

        $event->getPagelet()->addExtension('finderPopup', $struct);
    }

    /**
     * Adds the devices slider to the header pagelet.
     *
     * @param HeaderPageletLoadedEvent $event The event object
     */
    private function addDevicesSlider(HeaderPageletLoadedEvent $event): void
    {
        if (!$this->settingsService->getBool('showDeviceHistory')) {
            return;
        }

        if (!$this->settingsService->getBool('showDevicesSlider')) {
            return;
        }

        $struct = $event->getPagelet()->getExtension('deviceSlider');
        if ($struct === null) {
            $devicesHistory = $this->deviceHistoryService_Cookie->getDeviceEntities($event->getRequest(), $event->getContext());
            $struct = new Collection();

            $struct->set('userHiddable', $this->settingsService->getBool('deviceSliderUserHide'));
            $struct->set('hidden', $event->getRequest()->cookies->get(CookieKeyNameConstants::DEVICE_SLIDER) === 'hidden');

            if (count($devicesHistory)) {
                $deviceListIdsArray = array_keys($this->deviceToCustomerService->getDeviceListOfCustomerArray($event->getSalesChannelContext()));
                foreach ($devicesHistory as $device) {
                    if ($this->settingsService->getBool('showDevicelist')) {
                        $device->setInDeviceList(
                            in_array(
                                $device->getId(),
                                $deviceListIdsArray
                            )
                        );
                    } else {
                        $device->setInDeviceList(false);
                    }
                }

                $struct->set('devices', $devicesHistory);
            }
            $event->getPagelet()->addExtension('deviceSlider', $struct);
        }
    }

    /**
     * Adds the devices history to the header pagelet.
     *
     * @param HeaderPageletLoadedEvent $event The event object
     */
    private function addDevicesHistory(HeaderPageletLoadedEvent $event): void
    {
        if (!$this->settingsService->getBool('showDeviceHistory')) {
            return;
        }

        if (!$this->settingsService->getBool('showDeviceHistorySelect')) {
            return;
        }

        $struct = $event->getPagelet()->getExtension('deviceHistory');
        if ($struct === null) {
            $struct = new Collection();
            $cokie = $event->getRequest()->cookies->get(CookieKeyNameConstants::DEVICE_HISTORY);
            $devicesHistory = [];

            if ($cokie) {
                $devicesHistory = explode(',', $cokie);
            }

            if (is_array($devicesHistory) && count($devicesHistory)) {
                $criteria = (new Criteria(array_reverse($devicesHistory)))
                    ->addFilter(new EqualsFilter('enabled', true))
                    ->addAssociations(['brand']);
                $devicesHistory = $this->topdataDeviceRepository->search($criteria, $event->getContext())->getElements();
                $struct->set('devices', $devicesHistory);
            }

            $event->getPagelet()->addExtension('deviceHistory', $struct);
        }
    }

    /**
     * Adds the active brands list to the header pagelet.
     *
     * @param HeaderPageletLoadedEvent $event The event object
     */
    private function addActiveBrandsList(HeaderPageletLoadedEvent $event): void
    {
        $switch = $event->getRequest()->cookies->get(CookieKeyNameConstants::FINDER_SWITCH);
        $struct = $event->getPagelet()->getExtension('finder');
        if ($struct === null) {
            $struct = new Collection();
        }

        if ($switch) {
            $struct->set('switch', $switch);
        }

        $struct->set('search', $this->settingsService->getString('search'));

        $topbrands = $this->_getTopBrands();
        $brands = $this->_getNonTopBrands();

        $struct->set('loadPage', $this->settingsService->getBool('loadPage'));
        $struct->set('topbrands', $topbrands);
        $struct->set('brands', $brands);
        $struct->set('showSeries', $this->settingsService->getConfig('showSeries'));
        $struct->set('showTypes', $this->settingsService->getConfig('showTypes'));
        $struct->set('showDevicelist', $this->settingsService->getConfig('showDevicelist'));
        $struct->set('showMode', $this->settingsService->getString('selectboxesMode'));
        $struct->set('selectboxesShowMode', $this->settingsService->getString('selectboxesShowMode'));
        $struct->set('letters', range('a', 'z'));
        $struct->set('userHiddable', $this->settingsService->getBool('selectboxesUserHide'));
        $struct->set('hidden', $event->getRequest()->cookies->get(CookieKeyNameConstants::DEVICE_SELECTBOXES) === 'hidden');

        $event->getPagelet()->addExtension('finder', $struct); // this is used in topfinder-form.html.twig as `topfinder`
    }

    /**
     * Adds device suggestions to the suggest page.
     *
     * @param SuggestPageLoadedEvent $event The event object
     */
    public function addDeviceSuggest(SuggestPageLoadedEvent $event): void
    {
        if ($this->settingsService->getString('search') === 'combined') {
            $struct = $event->getPage()->getExtension('device_suggest');
            if ($struct === null) {
                $struct = new Collection();
            }

            $context = $event->getContext();
            $request = $event->getRequest();

            $term = $request->query->get('search');
            $term = mb_strtolower(trim($term));
            if (mb_strlen($term) > 2) {
                $devices = $this->_suggestDevices($term, $context);
                $total = $this->_totalSuggestDevices($term);
            } else {
                $devices = [];
                $total = 0;
            }
            $struct->set('devices', $devices);
            $struct->set('devices_total', $total);
            $event->getPage()->addExtension('device_suggest', $struct);
        }
    }

    /**
     * Calculates the total number of suggested devices for a given search term.
     * TODO: move to a service
     *
     * @param string $term The search term
     * @return int The total number of suggested devices
     */
    private function _totalSuggestDevices(string $term): int
    {
        $query = '';
        $terms = $this->_getTermsFromString($term);
        if (count($terms) > 1) {
            $andSQL = [];
            $i = 0;
            foreach ($terms as $word) {
                $i++;
                $andSQL[] = "(keywords LIKE '%$word%')";
            }
            $andSQL = implode('AND', $andSQL);
            $query = "SELECT COUNT(*) from topdata_device WHERE (is_enabled=1) AND ((keywords LIKE '%$term%') OR ($andSQL))";
        } else {
            $query = "SELECT COUNT(*) from topdata_device WHERE (is_enabled=1) AND (keywords LIKE '%$term%')";
        }
        $total = $this->connection->executeQuery($query)->fetchOne();

        return (int)$total;
    }

    /**
     * Suggests devices based on the given search term.
     *
     * @param string $term The search term
     * @param Context $context The context object
     * @return EntitySearchResult The search result containing suggested devices
     */
    private function _suggestDevices(string $term, Context $context): EntitySearchResult
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('enabled', true));
        $criteria->addAssociations(['media', 'brand']);
        $criteria->addSorting(new FieldSorting('model', FieldSorting::ASCENDING));

        $terms = $this->_getTermsFromString($term);
        if (count($terms) > 1) {
            $filters = [];
            foreach ($terms as $word) {
                $filters[] = new ContainsFilter('keywords', $word);
            }
            //            $criteria->addFilter(new AndFilter($filters));
            $criteria->addFilter(new OrFilter([
                new ContainsFilter('keywords', $term),
                new AndFilter($filters),
            ]));
        } else {
            $criteria->addFilter(new ContainsFilter('keywords', $term));
        }

        $criteria->setLimit(9);
        $devices = $this->topdataDeviceRepository->search($criteria, $context);

        return $devices;
    }

    /**
     * Adds the customer's device list to the header pagelet.
     *
     * @param HeaderPageletLoadedEvent $event The event object
     */
    private function addDeviceList(HeaderPageletLoadedEvent $event): void
    {
        $struct = new Collection();
        if ($this->settingsService->getBool('showDevicelist')) {
            $struct->set(
                'devices',
                $this->deviceToCustomerService->getDeviceListOfCustomerArray($event->getSalesChannelContext())
            );
            $struct->set('enabled', true);
        } else {
            $struct->set('enabled', false);
        }
        $event->getPagelet()->addExtension('deviceList', $struct);
    }

    /**
     * Extracts terms from a given string based on specified criteria.
     *
     * @param string $string The input string
     * @param int $min The minimum length of a term
     * @param int $max The maximum length of a term
     * @param int $maxCount The maximum number of terms to extract
     * @return array The extracted terms
     */
    private function _getTermsFromString(string $string, int $min = 3, int $max = 10, int $maxCount = 4): array
    {
        $rez = [];
        $string = str_replace(['-', '/', '+', '&', '.', ','], ' ', $string);
        $words = explode(' ', $string);
        foreach ($words as $word) {
            if (mb_strlen(trim($word)) >= $min) {
                $rez[] = mb_substr(trim($word), 0, $max);
                if (count($rez) >= $maxCount) {
                    break;
                }
            }
        }

        return $rez;
    }

    /**
     * 04/2025 created (extracted from addProductDevices)
     * TODO: move to a service
     */
    public function _hasDevices(SalesChannelProductEntity $product): bool
    {
        $productIds = [
            '0x' . $product->getId()
        ];
        if ($product->getParentId()) {
            $productIds[] = ', 0x' . $product->getParentId();
        }
        $SQL = '
                SELECT device_id
                 FROM topdata_device_to_product
                 WHERE product_id IN (' . implode(', ', $productIds) . ')
                 LIMIT 1
         ';
        $rez = $this->connection->executeQuery($SQL)->fetchAllAssociative();

        return count($rez) > 0;
    }

    /**
     * 04/2025 created (extracted)
     * TODO: move to BrandsDataService
     */
    public function _getTopBrands(): array
    {
        return $this->connection->createQueryBuilder()
            //->select ('LOWER(HEX(id)) as id, label as name')
            ->select('code, label as name')
            ->from('topdata_brand')
            ->where('(is_enabled = 1) AND (sort = 1)')
            ->orderBy('label')
            ->executeQuery()
            ->fetchAllAssociative();
    }

    /**
     * 04/2025 created (extracted)
     * TODO: move to BrandsDataService
     */
    public function _getNonTopBrands(): array
    {
        return $this->connection->createQueryBuilder()
            //->select ('LOWER(HEX(id)) as id, label as name')
            ->select('code, label as name')
            ->from('topdata_brand')
            ->where('(is_enabled = 1) AND (sort = 0)')
            ->orderBy('label')
            ->executeQuery()
            ->fetchAllAssociative();
    }
}
