<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\List;

use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataTopFinderProSW6\Component\BreadcrumbsTrait;
use Topdata\TopdataTopFinderProSW6\Constants\CookieKeyNameConstants;
use Topdata\TopdataTopFinderProSW6\Service\BrandDataService;
use Topdata\TopdataTopFinderProSW6\Service\TopdataDeviceDataService;
use Topdata\TopdataTopFinderProSW6\Service\SeriesDataService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Service\TopdataDeviceTypeService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\List\ListPage;

// Added
// Added
// Added
// Added
// Added

/**
 * Class BrandListPageLoader
 *
 * This class is responsible for loading a specific brand page.
 * It retrieves brand data, series, types, and devices based on settings and request parameters.
 */
class BrandListPageLoader
{
    use BreadcrumbsTrait;

    public function __construct(
        private readonly GenericPageLoader        $genericPageLoader,
        private readonly BrandDataService         $brandDataService,
        private readonly SeriesDataService        $seriesDataService,
        private readonly TopdataDeviceTypeService $topdataDeviceTypeService,
        private readonly TopdataDeviceDataService $topdataDeviceDataService,
        private readonly SettingsService          $settingsService,
        private readonly Translator               $translator,
        private readonly UrlGeneratorInterface    $router,
        private readonly LoggerInterface          $logger
    )
    {
    }

    /**
     * Load the brand page for a specific brand code
     *
     * This method loads the generic page, retrieves the brand data,
     * and configures the page with appropriate series, types, devices,
     * metadata, and breadcrumbs based on settings.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return ListPage The loaded brand page
     * @throws \Exception If an error occurs during loading
     */
    public function load(Request $request, SalesChannelContext $salesChannelContext): ListPage
    {
        try {
            // Get the brand code from the request
            $brandCode = $request->attributes->get('code');
            $startTime = microtime(true);
            $this->logger->info('BrandListPageLoader: Start loading', ['brandCode' => $brandCode, 'startTime' => $startTime]);

            // Load the generic page
            $page = $this->genericPageLoader->load($request, $salesChannelContext);

            // Create the list page from the generic page
            $listPage = new ListPage();
            // Copy all properties from the generic page to the list page
            foreach (get_object_vars($page) as $property => $value) {
                $listPage->$property = $value;
            }

            // Set the page type to brand
            $listPage->setPageType(TRASH____ListPageLoader::TYPE_BRAND);

            // Get the brand by code
            $brand = $this->brandDataService->getBrandByCode($brandCode, $salesChannelContext);

            // If the brand is not found, throw an exception
            if (!$brand) {
                // In a real implementation, we would throw a specific exception
                // For now, we'll log an error and throw a generic exception
                $this->logger->error('Brand not found', ['code' => $brandCode]);
                throw new \Exception('Brand not found: ' . $brandCode);
            }

            // Set the brand model to the page
            $listPage->model = $brand;

            $this->logger->info('BrandListPageLoader: Brand loaded', ['brandCode' => $brandCode, 'brandId' => $brand->getId(), 'duration_ms' => (microtime(true) - $startTime) * 1000]);
            $fetchStartTime = microtime(true);
            $this->logger->info('BrandListPageLoader: Starting data fetching block', ['brandCode' => $brandCode]);

            // Initialize the page show array
            $pageShow = [];

            // Determine which entities to show based on settings and request cookies
            if ($this->settingsService->getString('selectboxesMode') == 'both') {
                $switch = $request->cookies->get(CookieKeyNameConstants::FINDER_SWITCH);
                if ($switch == 'types') {
                    $pageShow['types'] = true;
                    $types = $this->topdataDeviceTypeService->getTypesForBrand($brand->getId(), $salesChannelContext);
                    $listPage->setTypes($types->getEntities());

                    $devicesTypeNull = $this->topdataDeviceDataService->getDevicesArray($brand->getCode(), '0', null);
                    if (count($devicesTypeNull)) {
                        $pageShow['devices'] = true;
                        $listPage->setDevices($devicesTypeNull);
                    }
                } else {
                    $pageShow['series'] = true;
                    $series = $this->seriesDataService->getSeriesForBrand($brand->getId(), $salesChannelContext);
                    $listPage->setSeries($series->getEntities());

                    $devicesSeriesNull = $this->topdataDeviceDataService->getDevicesArray($brand->getCode(), null);

                    if (count($devicesSeriesNull)) {
                        $pageShow['devices'] = true;
                        $listPage->setDevices($devicesSeriesNull);
                    }
                }
            } elseif ($this->settingsService->getBool('showSeries')) {
                $pageShow['series'] = true;
                $series = $this->seriesDataService->getSeriesForBrand($brand->getId(), $salesChannelContext);
                $listPage->setSeries($series->getEntities());

                $devicesSeriesNull = $this->topdataDeviceDataService->getDevicesArray($brand->getCode(), null);

                if (count($devicesSeriesNull)) {
                    $pageShow['devices'] = true;
                    $listPage->setDevices($devicesSeriesNull);
                }
            } elseif ($this->settingsService->getBool('showTypes')) {
                $pageShow['types'] = true;
                $types = $this->topdataDeviceTypeService->getTypesForBrand($brand->getId(), $salesChannelContext);
                $listPage->setTypes($types->getEntities());

                $devicesTypeNull = $this->topdataDeviceDataService->getDevicesArray($brand->getCode(), '0', null);
                if (count($devicesTypeNull)) {
                    $pageShow['devices'] = true;
                    $listPage->setDevices($devicesTypeNull);
                }
            }

            // If neither series nor types are shown, show all devices
            if (!$this->settingsService->getBool('showSeries') && !$this->settingsService->getBool('showTypes')) {
                $pageShow['devices'] = true;
                $devices = $this->topdataDeviceDataService->getDevicesArray($brand->getCode());
                $listPage->setDevices($devices);
            }

            $fetchEndTime = microtime(true);
            $devicesOnPage = $listPage->getDevices();
            $deviceCount = $devicesOnPage ? count($devicesOnPage) : 0;
            $this->logger->info('BrandListPageLoader: Finished data fetching block', ['brandCode' => $brandCode, 'deviceCount' => $deviceCount, 'duration_ms' => ($fetchEndTime - $fetchStartTime) * 1000]);

            // Set the page show array
            $listPage->setShow($pageShow);

            // Set the SEO title for the brand page
            $listPage->setTitle($this->translator->trans('topdata-topfinder.SEO.brandPageTitle', [
                '%brand%' => $brand->getName(),
            ]));

            // Set the SEO meta information
            // Note: We're logging a warning because we're not sure if the meta information methods
            // are available on the ListPage class. This needs to be verified in a real implementation.
            try {
                if (method_exists($listPage, 'getMetaInformation')) {
                    // Set the SEO meta title for the brand page
                    $listPage->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.brandMetaTitle', [
                        '%brand%' => $brand->getName(),
                    ]));

                    // Set the SEO meta description for the brand page
                    $listPage->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.brandMetaDescription', [
                        '%brand%' => $brand->getName(),
                    ]));

                    // Set the SEO robots meta tag for the brand page
                    $listPage->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.brandMetaRobots'));
                } else {
                    $this->logger->warning('Meta information methods not available on ListPage class');
                }
            } catch (\Exception $e) {
                $this->logger->warning('Error setting meta information: ' . $e->getMessage());
            }

            // Add breadcrumbs for the brand page
            $listPage->addBreadcrumb($this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'), $this->router->generate('frontend.top-finder.brands'));
            $listPage->addBreadcrumb($listPage->getTitle(), $this->router->generate('frontend.top-finder.brandq', ['code' => $brand->getCode()]));

            // Apply compact mode limit if configured
            if ($this->settingsService->getInt('searchCompactLimit', true)) {
                $listPage->setCompactModeLimit($this->settingsService->getInt('searchCompactLimit'));
            }

            // If devices are shown, load their content
            if (isset($pageShow['devices']) && $pageShow['devices'] && $listPage->getDevices() && count($listPage->getDevices()) > 0) { // Use safer check
                $devices = $listPage->getDevices();
                $deviceCount = count($devices);
                $loadContentStartTime = microtime(true);
                $this->logger->info('BrandListPageLoader: Starting loadDeviceContent', ['brandCode' => $brandCode, 'deviceCount' => $deviceCount]);

                $this->topdataDeviceDataService->loadDeviceContent($devices, $salesChannelContext, 'brand');
                // Re-set devices in case loadDeviceContent modifies them or returns a new collection/array
                $listPage->setDevices($devices);

                $loadContentEndTime = microtime(true);
                $this->logger->info('BrandListPageLoader: Finished loadDeviceContent', ['brandCode' => $brandCode, 'deviceCount' => $deviceCount, 'duration_ms' => ($loadContentEndTime - $loadContentStartTime) * 1000]);
            } else {
                $this->logger->info('BrandListPageLoader: Skipping loadDeviceContent', [
                    'brandCode'       => $brandCode,
                    'pageShowDevices' => $pageShow['devices'] ?? false,
                    'devicesExist'    => $listPage->getDevices() !== null,
                    'deviceCount'     => $listPage->getDevices() ? count($listPage->getDevices()) : 0
                ]);
            }

            $totalDuration = microtime(true) - $startTime;
            $this->logger->info('BrandListPageLoader: Finished loading', ['brandCode' => $brandCode, 'totalDuration_ms' => $totalDuration * 1000]);

            return $listPage;
        } catch (\Exception $e) {
            $this->logger->error('Error loading brand page: ' . $e->getMessage(), [
                'exception' => $e,
                'brandCode' => $request->attributes->get('code')
            ]);
            throw $e;
        }
    }
}