<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Service;

use Doctrine\DBAL\Connection;
use Shopware\Core\Checkout\Customer\CustomerEntity;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Topdata\TopdataConnectorSW6\Core\Content\Device\Agregate\DeviceCustomer\DeviceCustomerEntity;

/**
 * device_to_customer related functionalities
 *
 * 10/2024 created (extracted from HelperService)
 */
class DeviceToCustomerService
{
    public function __construct(
        protected readonly Connection       $connection,
        protected readonly SettingsService  $settingsService,
        protected readonly EntityRepository $topdataDeviceRepository,
    )
    {
    }

    /**
     * IDs of devices associated with the current customer.
     */
    private ?array $customerDeviceIds = null;


    /**
     * TODO: rename this method to something more descriptive
     *
     * Retrieves the list of devices associated with the current customer.
     *
     * This method checks if the device list is already populated. If not, it initializes the list
     * and fetches the device IDs from the database for the current logged-in customer.
     * The device IDs are then converted to hexadecimal format and stored in the device list.
     *
     * 10/2024 moved from TopFinderController to DeviceToCustomerService
     * 04/2025 TODO: remove this, move the caching to $this->fetchDeviceIdsOfCustomer()
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context containing customer information.
     * @return array The list of device IDs associated with the current customer.
     */
    public function getCustomerDeviceIds(SalesChannelContext $salesChannelContext): array
    {
        if ($this->customerDeviceIds === null) {
            $this->customerDeviceIds = $this->fetchDeviceIdsOfCustomer($salesChannelContext);
        }

        return $this->customerDeviceIds;
    }

    /**
     * Fetches the device IDs for the current customer from the database.
     * 10/2024 extracted from getCustomerDeviceIds
     *
     * @param SalesChannelContext $context The sales channel context.
     * @return string[] The list of device IDs in hexadecimal format.
     */
    private function fetchDeviceIdsOfCustomer(SalesChannelContext $context): array
    {
        $rows = $this->findByCustomer($context->getCustomer());

        return array_map(fn($row) => bin2hex($row['device_id']), $rows);
    }

    /**
     * Get the device-to-customer relation ID
     *
     * @param string $deviceId The device ID
     * @param string $customerId The customer ID
     * @return string|null The relation ID if found, null otherwise
     */
    public function deviceToCustomerId(string $deviceId, string $customerId): ?string
    {
        if (!Uuid::isValid($deviceId)) {
            return null;
        }

        if (!Uuid::isValid($customerId)) {
            return null;
        }

        return (string)$this->connection->executeQuery('
            SELECT LOWER(HEX(id)) as id
                FROM topdata_device_to_customer
                WHERE (0x' . $deviceId . ' = device_id) AND (0x' . $customerId . ' = customer_id)
                LIMIT 1
            ')->fetchOne();
    }

    /**
     * Get the list of devices for a customer
     * 10/2024 renamed from getDeviceList to getDeviceListForCustomer
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return array An array of device entities
     */
    public function getDeviceListForCustomer(SalesChannelContext $salesChannelContext)
    {
        $return = [];
        $deviceIds = [];
        if ($salesChannelContext->getCustomer() && !($salesChannelContext->getCustomer()->getGuest())) {
            $deviceIdsAssoc = $this->connection->createQueryBuilder()
                ->select('LOWER(HEX(device_id)) as device_id')
                ->from('topdata_device_to_customer')
                ->where('customer_id = 0x' . $salesChannelContext->getCustomer()->getId())
                ->execute()
                ->fetchAllAssociative();
            foreach ($deviceIdsAssoc as $val) {
                $deviceIds[] = $val['device_id'];
            }
        }
        if ($deviceIds) {
            $criteria = (new Criteria($deviceIds))
                ->addFilter(new EqualsFilter('enabled', true))
                ->addAssociations(['media', 'brand', 'series', 'type'])
                ->addSorting(new FieldSorting('model', FieldSorting::ASCENDING));

            $devices = $this->topdataDeviceRepository->search($criteria, $salesChannelContext->getContext());
            if ($devices) {
                $return = $devices->getEntities();
            }
        }

        return $return;
    }

    /**
     * Get the list of devices for a customer as an array
     * 10/2024 renamed getDeviceListArray --> getDeviceListOfCustomerArray
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return array An array of device data
     */
    public function getDeviceListOfCustomerArray(SalesChannelContext $salesChannelContext): array
    {
        $return = [];
        $deviceIds = [];
        if ($salesChannelContext->getCustomer() && !($salesChannelContext->getCustomer()->getGuest())) {
            $deviceIdsAssoc = $this->connection->createQueryBuilder()
                ->select('LOWER(HEX(device_id)) as device_id')
                ->from('topdata_device_to_customer')
                ->where('customer_id = 0x' . $salesChannelContext->getCustomer()->getId())
                ->execute()
                ->fetchAllAssociative();
            foreach ($deviceIdsAssoc as $val) {
                $deviceIds[] = $val['device_id'];
            }
        }

        if ($deviceIds) {
            $criteria = (new Criteria($deviceIds))
                ->addFilter(new EqualsFilter('enabled', true))
                ->addAssociations(['media', 'brand', 'series', 'type'])
                ->addSorting(new FieldSorting('brand.name', FieldSorting::ASCENDING))
                ->addSorting(new FieldSorting('model', FieldSorting::ASCENDING));

            $devices = $this->topdataDeviceRepository->search($criteria, $salesChannelContext->getContext())->getEntities();

            foreach ($devices as $device) {
                $return[$device->getId()] = [
                    'id'    => $device->getId(),
                    'model' => $device->getModel(),
                    'code'  => $device->getCode(),
                    'brand' => [
                        'id'   => $device->getBrand()->getId(),
                        'code' => $device->getBrand()->getCode(),
                        'name' => $device->getBrand()->getName(),
                    ],
                ];
            }
        }

        return $return;
    }

    /**
     * returns a list of all devices associated with the current customer.
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return array An array of device information
     */
    public function getDevicesOfCustomer(?CustomerEntity $customerEntity): array
    {
        if (!$customerEntity || $customerEntity->getGuest()) {
            return [];
        }

        $ret = [];

        $rez = $this->connection
            ->createQueryBuilder()
            ->select('LOWER(HEX(device_id)) as device_id, extra_info')
            ->from('topdata_device_to_customer')
            ->where('customer_id = 0x' . $customerEntity->getId())
            ->execute()
            ->fetchAllAssociative();

        foreach ($rez as $val) {
            $ret[$val['device_id']] = $val['extra_info'] ? json_decode($val['extra_info'], true) : DeviceCustomerEntity::defaultExtraInfo();
        }

        return $ret;
    }

    /**
     * 04/2025 created (extracted from TopdataDeviceDataService)
     */
    public function findByCustomer(?CustomerEntity $customerEntity): array
    {
        if (!$customerEntity || $customerEntity->getGuest()) {
            return [];
        }

        return $this->connection->createQueryBuilder()
            ->select(['device_id'])
            ->from('topdata_device_to_customer')
            ->where('customer_id = :customerId')
            ->setParameter('customerId', hex2bin($customerEntity->getId()))
            ->execute()
            ->fetchAllAssociative();
    }


}
