<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader;

use Doctrine\DBAL\Connection;
use Shopware\Core\Content\Category\Exception\CategoryNotFoundException;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepositoryInterface;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataConnectorSW6\Core\Content\Device\DeviceEntity;
use Topdata\TopdataTopFinderProSW6\Constants\CookieKeyNameConstants;
use Topdata\TopdataTopFinderProSW6\Service\DeviceToCustomerService;
use Topdata\TopdataTopFinderProSW6\Service\HelperService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Service\TopdataDeviceDataService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\PopupDeviceWidget\PopupDeviceWidgetPage;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\PopupFormWidget\PopupFormWidgetPage;
use Topdata\TopdataTopFinderProSW6\Util\UtilSearchTerm;

class PopupDeviceWidgetPageLoader
{
    public function __construct(
        private readonly GenericPageLoader       $genericPageLoader,
        private readonly Connection              $connection,
        private readonly SettingsService         $settingsService,
        private readonly UrlGeneratorInterface   $router,
        private readonly DeviceToCustomerService $deviceToCustomerService,
        private readonly TopdataDeviceDataService $topdataDeviceDataService
    )
    {
    }

    /**
     * @throws CategoryNotFoundException
     * @throws InconsistentCriteriaIdsException
     */
    public function load(Request $request, SalesChannelContext $salesChannelContext): PopupDeviceWidgetPage
    {
        $page = $this->genericPageLoader->load($request, $salesChannelContext);
        /** @var PopupFormWidgetPage $page */
        $page = PopupDeviceWidgetPage::createFrom($page);

        $deviceCode = $request->get('deviceCode');
        $device = $this->topdataDeviceDataService->findOneByDeviceCode($deviceCode, $salesChannelContext);
        if ($device) {
            if ($salesChannelContext->getCustomer() && !($salesChannelContext->getCustomer()->getGuest())) {
                $rez = $this->connection->createQueryBuilder()
                    ->select(['device_id'])
                    ->from('topdata_device_to_customer')
                    ->where('(customer_id = 0x'
                        . $salesChannelContext->getCustomer()->getId()
                        . ') AND (device_id = 0x'
                        . $device->getId()
                        . ')')
                    ->execute()
                    ->fetchAllAssociative();

                $device->setInDeviceList(count($rez) > 0);
            } else {
                $device->setInDeviceList(false);
            }
            $page->showDevicelist = $this->settingsService->getBool('showDevicelist');

            $page->device = $device;
        }

        $page->other = [
            'deviceCode'  => $deviceCode,
            'deviceFound' => $device ? true : false,
        ];

        return $page;
    }

    public function loadNew(Request $request, SalesChannelContext $salesChannelContext): PopupDeviceWidgetPage
    {
        $page = $this->genericPageLoader->load($request, $salesChannelContext);
        /** @var PopupFormWidgetPage $page */
        $page = PopupDeviceWidgetPage::createFrom($page);

        $deviceCode = $request->get('deviceCode');
        $device = $this->topdataDeviceDataService->findOneByDeviceCode($deviceCode, $salesChannelContext);

        $page->other = [
            'deviceCode'  => $deviceCode,
            'deviceFound' => $device ? true : false,
            'listType'    => $request->get('listType', false),
        ];

        if ($device) {
            if ($salesChannelContext->getCustomer() && !($salesChannelContext->getCustomer()->getGuest())) {
                $page->other['userLogged'] = true;

                $page->other['pathListAdd'] = $this
                    ->router
                    ->generate('frontend.top-finder-api.add-device', ['deviceId' => $device->getId()]);;

                $page->other['pathListRemove'] = $this
                    ->router
                    ->generate('frontend.top-finder-api.remove-device', ['deviceId' => $device->getId()]);

                $rez = $this->connection->createQueryBuilder()
                    ->select(['device_id'])
                    ->from('topdata_device_to_customer')
                    ->where('(customer_id = 0x'
                        . $salesChannelContext->getCustomer()->getId()
                        . ') AND (device_id = 0x'
                        . $device->getId()
                        . ')')
                    ->execute()
                    ->fetchAllAssociative();

                $device->setInDeviceList(count($rez) > 0);
            } else {
                $page->other['userLogged'] = false;
                $device->setInDeviceList(false);
            }
            $page->showDevicelist = $this->settingsService->getBool('showDevicelist');

            $page->device = $device;

            $this->setDevicesNavigation($page, $request, $salesChannelContext);
        }

        $page->deviceUrl = $this
            ->router
            ->generate('frontend.top-finder.device', ['deviceCode' => $deviceCode]);

        return $page;
    }

    private function setDevicesNavigation($page, $request, $salesChannelContext): void
    {
        $page->other['prevDevicePath'] = '';
        $page->other['prevDeviceText'] = 'topdata-topfinder.popup.devicePrev';
        $page->other['nextDevicePath'] = '';
        $page->other['nextDeviceText'] = 'topdata-topfinder.popup.deviceNext';
        $page->other['currentDeviceNumber'] = 0;
        $page->other['listDevicesCount'] = 0;

        if (!$page->other['listType']) {
            return;
        }
        /*
         * search
         * brand
         * brand-series
         * brand-types
         * series
         * types
         * history
         * device-list
         */
        $devices = [];

        $pathParams = ['listType' => $page->other['listType']];

        if ($page->other['listType'] == 'search') {
            $pathParams['term'] = UtilSearchTerm::filterTerm($request->get('term'));
            $devices = HelperService::findDevices($pathParams['term'], $this->connection);
        } elseif ($page->other['listType'] == 'brand') {
            $devices = $this->connection->fetchAllAssociative(
                'SELECT LOWER(HEX(id)) as id, code'
                . ' FROM `topdata_device`'
                . ' WHERE (is_enabled=1)AND(brand_id=0x' . $page->device->getBrandId() . ')'
                . ' ORDER BY code ASC'
            );
        } elseif ($page->other['listType'] == 'series' && $page->device->getSeriesId()) {
            $devices = $this->connection->fetchAllAssociative(
                'SELECT LOWER(HEX(id)) as id, code'
                . ' FROM `topdata_device`'
                . ' WHERE (is_enabled=1)AND(series_id=0x' . $page->device->getSeriesId() . ')'
                . ' ORDER BY code ASC'
            );
        } elseif ($page->other['listType'] == 'types' && $page->device->getTypeId()) {
            $devices = $this->connection->fetchAllAssociative(
                'SELECT LOWER(HEX(id)) as id, code'
                . ' FROM `topdata_device`'
                . ' WHERE (is_enabled=1)AND(type_id=0x' . $page->device->getTypeId() . ')'
                . ' ORDER BY code ASC'
            );
        } elseif ($page->other['listType'] == 'history') {
            $cokie = $request->cookies->get(CookieKeyNameConstants::DEVICE_HISTORY);
            $cookieIds = [];
            $deviceIds = [];
            if ($cokie) {
                $cookieIds = explode(',', $cokie);
            }

            foreach ($cookieIds as $value) {
                if (Uuid::isValid($value)) {
                    $deviceIds[] = $value;
                }
            }
            if (count($deviceIds)) {
                $unsortedDevices = $this->connection->fetchAllAssociative('
                    SELECT LOWER(HEX(id)) as id, 
                           code as code
                      FROM topdata_device 
                      WHERE (is_enabled=1)AND(topdata_device.id IN (0x' . implode(',0x', $deviceIds) . '))
                ');
                $deviceIds = array_reverse($deviceIds);

                foreach ($deviceIds as $id) {
                    foreach ($unsortedDevices as $device) {
                        if ($id === $device['id']) {
                            $devices[] = $device;
                        }
                    }
                }
            }
        } elseif ($page->other['listType'] == 'device-list') {
            $devices = array_values($this->deviceToCustomerService->getDeviceListOfCustomerArray($salesChannelContext));
        }

        $page->other['listDevicesCount'] = count($devices);

        foreach ($devices as $key => $device) {
            if ($page->device->getId() == $device['id']) {
                $page->other['currentDeviceNumber'] = $key + 1;

                if ($key > 0) {
                    $pathParams['deviceCode'] = $devices[$key - 1]['code'];
                    $page->other['prevDevicePath'] = $this->router->generate(
                        'frontend.top-finder-api.popup-device-new',
                        $pathParams
                    );
                }

                if ($key < (count($devices) - 1)) {
                    $pathParams['deviceCode'] = $devices[$key + 1]['code'];
                    $page->other['nextDevicePath'] = $this->router->generate(
                        'frontend.top-finder-api.popup-device-new',
                        $pathParams
                    );
                }

                break;
            }
        }
    }
}
