<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\Popup;

use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataTopFinderProSW6\Service\DeviceDataService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Service\TopdataDeviceTypeService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\List\ListPage;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\List\TRASH____ListPageLoader;

/**
 * Class PopupTypeDevicesPageLoader
 *
 * This class is responsible for loading the popup type devices page.
 * It retrieves devices for a specific type and prepares the page for display in a popup.
 */
class PopupTypeDevicesPageLoader
{

    public function __construct(
        private readonly GenericPageLoader        $genericPageLoader,
        private readonly DeviceDataService        $deviceDataService,
        private readonly SettingsService          $settingsService,
        private readonly Translator               $translator,
        private readonly UrlGeneratorInterface    $router,
        private readonly LoggerInterface          $logger,
        private readonly TopdataDeviceTypeService $typeDataService,
    )
    {
    }

    /**
     * Load the popup type devices page
     *
     * This method loads the generic page, retrieves a type by code,
     * gets the devices for that type, and configures the page for display in a popup.
     *
     * @param mixed $request The current request
     * @param mixed $salesChannelContext The sales channel context
     * @return ListPage The loaded popup type devices page
     */
    public function load($request, $salesChannelContext): ListPage
    {
        try {
            // Get the type code from the request attributes
            $typeCode = $request->attributes->get('code');

            // Load the generic page
            $page = $this->genericPageLoader->load($request, $salesChannelContext);

            // Create the list page from the generic page
            $listPage = new ListPage();
            // Copy all properties from the generic page to the list page
            foreach (get_object_vars($page) as $property => $value) {
                $listPage->$property = $value;
            }

            // Get the type by code
            $type = $this->typeDataService->getDeviceTypeByCode($typeCode, $salesChannelContext);

            // If the type is not found, return the empty page
            if (!$type) {
                $this->logger->warning('Type not found with code: ' . $typeCode);
                return $listPage;
            }

            // Get the brand from the type
            $brand = $type->getBrand();

            // If the brand is not found, return the empty page
            if (!$brand) {
                $this->logger->warning('Brand not found for type with code: ' . $typeCode);
                return $listPage;
            }

            // Set the title for the popup type devices page
            $listPage->setTitle($this->translator->trans('topdata-topfinder.SEO.typePageTitle', [
                '%brand%' => '<b>' . $brand->getName() . '</b>',
                '%type%'  => $type->getLabel(),
            ]));

            // Get devices for the type
            $devices = $this->deviceDataService->getDevicesArray($brand->getCode(), null, $type->getId());

            // Load device content (enrich the device data)
            $this->deviceDataService->loadDeviceContent($devices, $salesChannelContext, 'types');

            // Set the devices to the page
            $listPage->setDevices($devices);

            // Set the page type to type
            $listPage->setPageType(TRASH____ListPageLoader::TYPE_TYPE);

            // Set the page to show devices
            $listPage->setShow(['devices' => true]);

            // Generate the popup path
            $letter = ($brand->getCode())[0];
            if (preg_match('/^[0-9]{1}$/', $letter)) {
                $letter = '0';
                $letterStr = '0-9';
            } else {
                $letterStr = strtoupper($letter);
            }

            $popupPath = [];

            $popupPath[] = [
                'name' => $letterStr,
                'path' => $this->router->generate('frontend.top-finder-api.popup-letter', ['letter' => $letter]),
            ];

            $popupPath[] = [
                'name' => $brand->getName(),
                'path' => $this->router->generate('frontend.top-finder.popup_brand', ['brandCode' => $brand->getCode()]),
            ];

            $popupPath[] = [
                'name' => $this->translator->trans('topdata-topfinder.popup.brandTypes'),
                'path' => $this->router->generate('frontend.top-finder.popup_brand_types', ['brandCode' => $brand->getCode()]),
            ];

            $popupPath[] = [
                'name' => $type->getLabel(),
            ];

            $listPage->popupPath = $popupPath;

            return $listPage;
        } catch (\Exception $e) {
            $this->logger->error('Error loading popup type devices page: ' . $e->getMessage());
            throw $e;
        }
    }
}