<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Controller;

use Shopware\Core\Checkout\Cart\Exception\CustomerNotLoggedInException;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\Framework\Validation\DataBag\RequestDataBag;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Controller\StorefrontController;
use Shopware\Storefront\Page\Page;
use Symfony\Component\HttpFoundation\Cookie;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\Routing\Annotation\Route;
use Topdata\TopdataConnectorSW6\Core\Content\Device\DeviceEntity;
use Topdata\TopdataFoundationSW6\Util\UtilUuid;
use Topdata\TopdataTopFinderProSW6\Component\Collection;
use Topdata\TopdataTopFinderProSW6\Constants\CookieKeyNameConstants;
use Topdata\TopdataTopFinderProSW6\Constants\TopfinderPluginConstants;
use Topdata\TopdataTopFinderProSW6\Service\DeviceDataService;
use Topdata\TopdataTopFinderProSW6\Service\DeviceHistoryService_Cookie;
use Topdata\TopdataTopFinderProSW6\Service\DeviceToCustomerService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Device\DevicePage;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\BrandLetterPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\DevicePageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\ListProductsPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\PopupDeviceInfoWidgetPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\PopupDeviceWidgetPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\PopupFormWidgetPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\DeviceListPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\DeviceSearchPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\DeviceSuggestPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\List\BrandsListPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\List\TypeListPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\Popup\PopupSearchPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\Popup\SeriesListPageLoader;
use Topdata\TopdataTopFinderProSW6\Util\ControllerUtil;

/**
 * Controller for handling TopFinder related actions in the storefront.
 *
 * This controller manages various functionalities related to device search,
 * listing, and management in the TopFinder Pro plugin for Shopware 6.
 *
 * TODO: this controller is too big and should be split into multiple controllers
 */
#[Route(defaults: ['_routeScope' => ['storefront']])]
class TopFinderController extends StorefrontController
{




    public function __construct(
        private readonly EntityRepository                $topdataBrandRepository,
        private readonly EntityRepository                $topdataDeviceRepository,
        private readonly EntityRepository                $topdataSeriesRepository,
        private readonly EntityRepository                $topdataDeviceTypeRepository,
        private readonly DeviceListPageLoader            $deviceListPageLoader,
        private readonly DeviceSuggestPageLoader         $deviceSuggestPageLoader,
        private readonly DeviceSearchPageLoader          $deviceSearchPageLoader,
        private readonly SettingsService                 $settingsService,
        private readonly BrandsListPageLoader            $brandsListPageLoader,
        private readonly SeriesListPageLoader            $seriesListPageLoader,
        private readonly TypeListPageLoader              $typeListPageLoader,
        private readonly PopupDeviceWidgetPageLoader     $popupDeviceWidgetPageLoader,
        private readonly PopupFormWidgetPageLoader       $popupFormWidgetPageLoader,
        private readonly BrandLetterPageLoader           $brandLetterPageLoader,
        private readonly PopupDeviceInfoWidgetPageLoader $popupDeviceInfoWidgetPageLoader,
        private readonly ListProductsPageLoader          $listProductsPageLoader,
        private readonly DevicePageLoader                $devicePageLoader,
        private readonly DeviceDataService               $deviceDataService,
        private readonly DeviceToCustomerService         $deviceToCustomerService,
        private readonly DeviceHistoryService_Cookie     $deviceHistoryService_Cookie,
        private readonly PopupSearchPageLoader           $popupSearchPageLoader,
    )
    {
    }



//    /**
//     * 10/2024 commented out (unused)
//     *
//     *
//     * Retrieves the product listing for a given device.
//     *
//     * This method creates a Criteria object using the product IDs associated with the given device,
//     * dispatches a DeviceProductsCriteriaEvent to allow for modifications to the criteria,
//     * and then searches the product repository with the modified criteria.
//     *
//     * @param DeviceEntity $device The device entity containing the product IDs.
//     * @param SalesChannelContext $context The sales channel context.
//     * @return ProductListingResult The resulting product listing.
//     */
//    private function _getProductListingByDevice(DeviceEntity $device, SalesChannelContext $context): ProductListingResult
//    {
//        $criteria = new Criteria($device->getProducts()->getIds());
//
//        $this->eventDispatcher->dispatch(
//            new DeviceProductsCriteriaEvent($criteria, $context)
//        );
//
//        $result = $this->productRepository->search($criteria, $context);
//        $result = ProductListingResult::createFrom($result);
//
//        return $result;
//    }


    /**
     * renders "account/device-list.html.twig"
     *
     * Handles the request to display the device list for the logged-in customer.
     *
     * This method checks if the customer is logged in and not a guest. If the customer is not logged in,
     * it throws a CustomerNotLoggedInException. If the customer is logged in, it loads the device list page
     * and sets the meta information to prevent indexing by search engines. Finally, it renders the device list page.
     *
     * @param Request $request The HTTP request object.
     * @param SalesChannelContext $context The sales channel context.
     * @return Response The HTTP response object containing the rendered device list page.
     * @throws CustomerNotLoggedInException If the customer is not logged in or is a guest.
     */
    #[Route('/account/devicelist', name: 'frontend.account.devicelist.page', methods: ['GET'], defaults: ['_loginRequired' => true])]
    public function deviceList(Request $request, SalesChannelContext $context): Response
    {
        if (!$context->getCustomer() || $context->getCustomer()->getGuest()) {
            throw new CustomerNotLoggedInException(
                Response::HTTP_FORBIDDEN,
                'Customer is not logged in.',
                'Customer is not logged in.'
            );
        }

        $page = $this->deviceListPageLoader->loadPage($request, $context);

        $page->getMetaInformation()->setRobots('noindex,nofollow');

        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/account/device-list.html.twig', [
            'page' => $page,
        ]);
    }

    /**
     * Handles the device suggestion request.
     *
     * This method loads the device suggestion page based on the provided request and context,
     * renders the corresponding Twig template, and sets the appropriate HTTP headers.
     *
     * @param Request $request The HTTP request object.
     * @param SalesChannelContext $context The sales channel context.
     * @return Response The HTTP response object containing the rendered device suggestion page.
     */
    #[Route('/device-suggest', name: 'frontend.top-finder.device_suuggest', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function deviceSuggest(Request $request, SalesChannelContext $context): Response
    {
        $page = $this->deviceSuggestPageLoader->load($request, $context);

        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/device-suggest/index.html.twig', ['page' => $page]);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }


    /**
     * Handles the device search request.
     *
     * This method loads the device search page based on the provided request and context,
     * renders the corresponding Twig template, and sets the appropriate HTTP headers.
     *
     * @param Request $request The HTTP request object.
     * @param SalesChannelContext $context The sales channel context.
     * @return Response The HTTP response object containing the rendered device search page.
     */
    #[Route('/device-search', name: 'frontend.top-finder.device_search.page', methods: ['GET'])]
    public function deviceSearch(Request $request, SalesChannelContext $context): Response
    {
        $page = $this->deviceSearchPageLoader->loadPage($request, $context);

        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/device-search.html.twig', ['page' => $page]);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }


    /**
     * Handles the request to list all brands.
     *
     * This method loads the brand listing page based on the provided request and context,
     * and renders the corresponding Twig template.
     *
     * @param Request $request The HTTP request object.
     * @param SalesChannelContext $salesChanelContext The sales channel context.
     * @return Response The HTTP response object containing the rendered brand listing page.
     */
    #[Route('/top-finder/brands', name: 'frontend.top-finder.brands', methods: ['GET'])]
    public function listBrands(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $page = $this->brandsListPageLoader->load($request, $salesChanelContext);

        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list.html.twig', ['page' => $page]);
    }


    /**
     * Handles the request to list all series for a given brand.
     *
     * This method loads the series listing page based on the provided request and context,
     * and renders the corresponding Twig template. It also sets the brand code and series code,
     * and retrieves series and types if configured to do so.
     *
     * @param Request $request The HTTP request object.
     * @param RequestDataBag $data The data bag containing request data.
     * @param SalesChannelContext $context The sales channel context.
     * @return Response The HTTP response object containing the rendered series listing page.
     */
    #[Route('/top-finder/series/{code}', name: 'frontend.top-finder.series', methods: ['GET'])]
    public function listSeries(Request $request, RequestDataBag $data, SalesChannelContext $context): Response
    {
        // Load the series listing page using the dedicated loader
        $page = $this->seriesListPageLoader->load($request, $context);

        // Get the finder structure from the page header.
        $finderStruct = $page->getHeader()->getExtension('finder');
        if ($finderStruct !== null) {
            // Set the brand code and series code in the finder structure.
            $finderStruct->set('brandCode', $page->model->getBrand()->getCode());
            $finderStruct->set('seriesCode', $page->model->getCode());

            // Retrieve and set the series if configured to do so.
            if ($this->settingsService->getConfig('showSeries')) {
                $series = ControllerUtil::getEntityCollection(
                    $this->topdataSeriesRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page->model->getBrand()->getId(),
                    ],
                    'label'
                );
                $finderStruct->set('series', $series);
            }

            // Retrieve and set the types if configured to do so.
            if ($this->settingsService->getConfig('showTypes')) {
                $types = ControllerUtil::getEntityCollection(
                    $this->topdataDeviceTypeRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page->model->getBrand()->getId(),
                    ],
                    'label'
                );
                $finderStruct->set('types', $types);
            }

            // Retrieve and set the devices array.
            $devicesArray = $this->deviceDataService->getFilteredDevices(
                $page->model->getBrand()->getId(),
                $page->model->getId(),
                0,
                false
            );
            $finderStruct->set('devices', $devicesArray);
        }

        // Render the series listing page.
        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list.html.twig', ['page' => $page]);
    }

    /**
     * Handles the request to list all types for a given brand.
     *
     * This method loads the type listing page based on the provided request and context,
     * and renders the corresponding Twig template. It also sets the brand code and type code,
     * and retrieves series and types if configured to do so.
     *
     * @param Request $request The HTTP request object.
     * @param RequestDataBag $data The data bag containing request data.
     * @param SalesChannelContext $context The sales channel context.
     * @return Response The HTTP response object containing the rendered type listing page.
     */
    #[Route('/top-finder/type/{code}', name: 'frontend.top-finder.type', methods: ['GET'])]
    public function listType(string $code, SalesChannelContext $context): Response
    {
        // Load the type listing page using the dedicated loader
        $page_deviceTypeList = $this->typeListPageLoader->load($code, $context);
        $finderStruct = $page_deviceTypeList->getHeader()->getExtension('finder');
        if ($finderStruct !== null) {
            // Set the brand code and type code in the finder structure.
            $finderStruct->set('brandCode', $page_deviceTypeList->model->getBrand()->getCode());
            $finderStruct->set('typeCode', $page_deviceTypeList->model->getCode());

            // Retrieve and set the series if configured to do so.
            if ($this->settingsService->getConfig('showSeries')) {
                $series = ControllerUtil::getEntityCollection(
                    $this->topdataSeriesRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page_deviceTypeList->model->getBrand()->getId(),
                    ],
                    'label'
                );
                $finderStruct->set('series', $series);
            }

            // Retrieve and set the types if configured to do so.
            if ($this->settingsService->getConfig('showTypes')) {
                $types = ControllerUtil::getEntityCollection(
                    $this->topdataDeviceTypeRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page_deviceTypeList->model->getBrand()->getId(),
                    ],
                    'label'
                );
                $finderStruct->set('types', $types);
            }

            // Retrieve and set the devices array.
            $devicesArray = $this->deviceDataService->getFilteredDevices(
                $page_deviceTypeList->model->getBrand()->getId(),
                0,
                $page_deviceTypeList->model->getId(),
                false
            );
            $finderStruct->set('devices', $devicesArray);
        }

        // Render the type listing page.
        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list.html.twig', ['page' => $page_deviceTypeList]);
    }

    #[Route('/top-finder/popupForm', name: 'frontend.top-finder.popup_form', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function popupForm(Request $request, SalesChannelContext $salesChannelContext): Response
    {
        $page = $this->popupFormWidgetPageLoader->load($request, $salesChannelContext);
        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/popup-form/index.html.twig', ['page' => $page]);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder/get-popup-device', name: 'frontend.top-finder.popup_device', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function popupDevice(Request $request, SalesChannelContext $salesChannelContext): Response
    {
        $page = $this->popupDeviceWidgetPageLoader->load($request, $salesChannelContext);
        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/popup-device/index.html.twig', ['page' => $page]);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder/letter/{letter}', name: 'frontend.top-finder.letter', methods: ['GET'])]
    public function listLetter(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $letter = $request->get('letter');
        if (!preg_match('/^[a-z0]{1}$/', $letter)) {
            throw new BadRequestHttpException('letter'/*, '/letter'*/);
        }

        $page = $this->brandLetterPageLoader->load($request, $salesChanelContext, $letter);

        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/brand-letter.html.twig', ['page' => $page]);
    }

    #[Route('/top-finder/history-remove/{deviceId}', name: 'frontend.top-finder.history-remove', methods: ['GET'])]
    public function historyRemove(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $deviceId = $request->get('deviceId');
        if (!UtilUuid::isValidUuid($deviceId)) {
            throw new BadRequestHttpException('deviceId');
        }

        $referer = $request->headers->get('referer');

        $cokie = $request->cookies->get(CookieKeyNameConstants::DEVICE_HISTORY);

        $devicesHistory = [];

        if ($cokie) {
            $devicesHistory = explode(',', $cokie);
        }

        if (!is_array($devicesHistory)) {
            $devicesHistory = [];
        }

        $historyKey = array_search($deviceId, $devicesHistory);

        if ($historyKey !== false) {
            unset($devicesHistory[$historyKey]);
        }

        if (!$referer) {
            $referer = '/';
        }

        $response = new RedirectResponse($referer);

        $response->headers->setCookie(
            Cookie::create(
                CookieKeyNameConstants::DEVICE_HISTORY,
                implode(',', $devicesHistory),
                time() + 3600 * 24 * 30
            )
        );

        return $response;
    }

    #[Route('/top-finder/popup-device-info/{deviceId}', name: 'frontend.top-finder.popup_device_info', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function popupDeviceInfo(Request $request, SalesChannelContext $salesChannelContext): Response
    {
        $deviceId = $request->get('deviceId');

        if (!UtilUuid::isValidUuid($deviceId)) {
            throw new BadRequestHttpException('deviceId');
        }

        if (!$salesChannelContext->getCustomer() || $salesChannelContext->getCustomer()->getGuest()) {
            throw new BadRequestHttpException('userId');
        }

        $userId = $salesChannelContext->getCustomer()->getId();

        $page = $this->popupDeviceInfoWidgetPageLoader->load($request, $salesChannelContext, $deviceId, $userId);
        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/popup-device-info/index.html.twig', ['page' => $page]);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    /**
     * @throws CustomerNotLoggedInException
     */
    #[Route('/top-finder/devicelist-find', name: 'frontend.top-finder.devicelist-find', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function deviceListFind(Request $request, SalesChannelContext $context): Response
    {
        $page = $this->deviceListPageLoader->loadPage($request, $context);

        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/account/device-list-find.html.twig', ['page' => $page]);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route(
        path: '/top-finder/list-products',
        name: 'frontend.top-finder.list-products',
        methods: ['GET']
    )]
    public function listProducts(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $productIds = explode('-', $request->get('ids'));

        foreach ($productIds as $key => $id) {
            if (Uuid::isValid((string)$id) === false) {
                unset($productIds[$key]);
            }
        }
        if (!count($productIds)) {
            throw new BadRequestHttpException('ids');
        }

        $page = $this->listProductsPageLoader->loadPage($request, $salesChanelContext, $productIds);

        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list-products.html.twig', ['page' => $page]);
    }

    #[Route(
        path: '/top-finder/popup-search',
        name: 'frontend.top-finder.popup_search',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function listPopupSearch(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $page = $this->popupSearchPageLoader->load($request, $salesChanelContext);
        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);
        $offset = $request->query->get('offset');
        $defaultOffset = TopfinderPluginConstants::POPUP_DEVICES_PRELOAD_LENGTH;

        if ($offset) {
            $response = $this->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/_listed-devices.html.twig',
                ['page' => $page]
            );
        } else {
            $response = $this->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devices.html.twig',
                ['page' => $page, 'listMode' => $listMode]
            );
            $response->headers->set('loadOffset', (string)$defaultOffset);
        }

        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');
        $response->headers->set('returnPath', ControllerUtil::getReturnPath($request));

        return $response;
    }

    #[Route('/top-finder/popup-search-suggest', name: 'frontend.top-finder.popup_search_suggest', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function suggestPopupSearch(Request $request, SalesChannelContext $salesChanelContext)
    {
        $page = $this->popupSearchPageLoader->load($request, $salesChanelContext);

        $response = $this->renderStorefront(
            '@TopdataTopFinderProSW6/storefront/widget/popup-form/suggest-devices.html.twig',
            ['page' => $page]
        );

        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

}
