<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Controller;

use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Controller\StorefrontController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\Routing\Annotation\Route;
use Topdata\TopdataTopFinderProSW6\Constants\CookieKeyNameConstants;
use Topdata\TopdataTopFinderProSW6\Constants\TopfinderPluginConstants;
use Topdata\TopdataTopFinderProSW6\Service\DeviceDataService;
use Topdata\TopdataTopFinderProSW6\Service\HelperService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\OneBrandLetterPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\List\BrandListPageLoader;
use Topdata\TopdataTopFinderProSW6\Util\ControllerUtil;

/**
 * 11/2024 created (extracted from TopFinderController)
 */
#[Route(defaults: ['_routeScope' => ['storefront']])]
class TopFinderBrandController extends StorefrontController
{


    public function __construct(
        private readonly EntityRepository         $topdataSeriesRepository,
        private readonly EntityRepository         $topdataDeviceTypeRepository,
        private readonly SettingsService          $settingsService,
        private readonly BrandListPageLoader      $brandListPageLoader,
        private readonly OneBrandLetterPageLoader $oneBrandLetterPageLoader,
        private readonly HelperService            $helperService,
        private readonly DeviceDataService        $deviceDataService
    )
    {
    }

    /**
     * renders "one-brand-letter.html.twig"
     *
     * Handles the request to list a specific brand
     *
     * This method loads the brand listing page based on the provided request and context,
     * and renders the corresponding Twig template. It also sets the brand code and retrieves
     * series and types if configured to do so.
     *
     * @param Request $request The HTTP request object.
     * @param SalesChannelContext $salesChanelContext The sales channel context.
     * @return Response The HTTP response object containing the rendered brand listing page.
     * @throws BadRequestHttpException If the brand code is invalid.
     */
    #[Route(
        path: '/top-finder/brand/{brandCode}',
        name: 'frontend.top-finder.show-brand',
        methods: ['GET'],
    )]
    public function showBrandAction(string $brandCode, Request $request, SalesChannelContext $salesChanelContext): Response
    {
        ControllerUtil::assertValidBrandcode($brandCode);
        $page = $this->oneBrandLetterPageLoader->load(
            $request,
            $salesChanelContext,
            $brandCode
        );

        $finderStruct = $page->getHeader()->getExtension('finder');
        if ($finderStruct !== null) {
            $finderStruct->set('brandCode', $page->brand['code']);

            if ($this->settingsService->getConfig('showSeries')) {
                $series = ControllerUtil::getEntityCollection(
                    $this->topdataSeriesRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page->brand['id'],
                    ],
                    'label'
                );
                $finderStruct->set('series', $series);
            }

            if ($this->settingsService->getConfig('showTypes')) {
                $types = ControllerUtil::getEntityCollection(
                    $this->topdataDeviceTypeRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page->brand['id'],
                    ],
                    'label'
                );
                $finderStruct->set('types', $types);
            }

            $devicesArray = $this->deviceDataService->getFilteredDevices($page->brand['id'], 0, 0, false);
            $finderStruct->set('devices', $devicesArray);
        }

        return $this->renderStorefront(
            '@TopdataTopFinderProSW6/storefront/page/topfinder/one-brand-letter.html.twig',
            ['page' => $page]
        );
    }

    /**
     * renders "list.html.twig"
     *
     * Handles the request to list a specific brand.
     *
     * This method loads the brand listing page based on the provided request and context,
     * and renders the corresponding Twig template. It also sets the brand code and retrieves
     * series and types if configured to do so.
     *
     * @param Request $request The HTTP request object.
     * @param SalesChannelContext $salesChanelContext The sales channel context.
     * @return Response The HTTP response object containing the rendered brand listing page.
     * @throws BadRequestHttpException If the brand code is invalid.
     */
    #[Route('/top-finder/brandq/{code}', name: 'frontend.top-finder.brandq', methods: ['GET'])]
    public function listBrandQ(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $brandCode = $request->get('code');
        ControllerUtil::assertValidBrandcode($brandCode);
        $page = $this->brandListPageLoader->load($request, $salesChanelContext);

        $finderStruct = $page->getHeader()->getExtension('finder');
        if ($finderStruct !== null) {
            $finderStruct->set('brandCode', $page->model->getCode());

            if ($this->settingsService->getConfig('showSeries')) {
                $series = ControllerUtil::getEntityCollection(
                    $this->topdataSeriesRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page->model->getId(),
                    ],
                    'label'
                );
                $finderStruct->set('series', $series);
            }

            if ($this->settingsService->getConfig('showTypes')) {
                $types = ControllerUtil::getEntityCollection(
                    $this->topdataDeviceTypeRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page->model->getId(),
                    ],
                    'label'
                );
                $finderStruct->set('types', $types);
            }

            $devicesArray = $this->deviceDataService->getFilteredDevices($page->model->getId(), 0, 0, false);
            //$devicesArray = $this->getDevicesArray($page->model->getId(), -1, -1, false);
            $finderStruct->set('devices', $devicesArray);
        }

        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list.html.twig', ['page' => $page]);
    }


    #[Route('/top-finder/letter-brand/{brandCode}', name: 'frontend.top-finder.letter-brand', methods: ['GET'])]
    public function listLetterBrand($brandCode, Request $request, SalesChannelContext $salesChanelContext): Response
    {
        ControllerUtil::assertValidBrandcode($brandCode);
        $page = $this->oneBrandLetterPageLoader->load($request, $salesChanelContext, $brandCode);

        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/one-brand-letter.html.twig', ['page' => $page]);
    }


    #[Route(
        path: '/top-finder/popup-brand/{brandCode}',
        name: 'frontend.top-finder.popup_brand',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function listPopupBrand($brandCode, Request $request, SalesChannelContext $salesChanelContext): Response
    {
        ControllerUtil::assertValidBrandcode($brandCode);
        $page = $this->oneBrandLetterPageLoader->loadJson(
            $request,
            $salesChanelContext,
            $brandCode,
            'all'
        );

        $offset = $request->get('offset');
        $defaultOffset = TopfinderPluginConstants::POPUP_DEVICES_PRELOAD_LENGTH;
        if ($offset) {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/_listed-devices.html.twig',
                    ['page' => $page, 'offset' => $offset]
                );
        } else {
            $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-devices.html.twig',
                    [
                        'page'     => $page,
                        'mode'     => 'all',
                        'listMode' => $listMode,
                        'offset'   => $defaultOffset,
                    ]
                );
            $response->headers->set('returnPath', ControllerUtil::getReturnPath($request));
            $response->headers->set('loadOffset', (string)$defaultOffset);
        }

        //        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }


    #[Route(
        path: '/top-finder/popup-brand-series/{brandCode}',
        name: 'frontend.top-finder.popup_brand_series',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET'],
    )]
    public function listPopupBrandSeries($brandCode, Request $request, SalesChannelContext $salesChanelContext): Response
    {
        ControllerUtil::assertValidBrandcode($brandCode);
        $page = $this->oneBrandLetterPageLoader->loadJson(
            $request,
            $salesChanelContext,
            $brandCode,
            'series'
        );

        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);
        $offset = $request->get('offset');
        $defaultOffset = TopfinderPluginConstants::POPUP_DEVICES_PRELOAD_LENGTH;
        if ($offset) {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/_listed-series-devices.html.twig',
                    ['page' => $page, 'offset' => $offset, 'hideMenuSeries' => true]
                );
        } else {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-devices.html.twig',
                    [
                        'page'           => $page,
                        'mode'           => 'series',
                        'listMode'       => $listMode,
                        'offset'         => $defaultOffset,
                        'hideMenuSeries' => true,
                    ]
                );
            $response->headers->set('returnPath', ControllerUtil::getReturnPath($request));
            $response->headers->set('loadOffset', (string)$defaultOffset);
        }

        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }


    #[Route('/top-finder/popup-brand-types/{brandCode}', name: 'frontend.top-finder.popup_brand_types', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function listPopupBrandTypes($brandCode, Request $request, SalesChannelContext $salesChanelContext)
    {
        ControllerUtil::assertValidBrandcode($brandCode);
        $page = $this->oneBrandLetterPageLoader->loadJson(
            $request,
            $salesChanelContext,
            $brandCode,
            'types'
        );

        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);
        $offset = $request->get('offset');
        $defaultOffset = TopfinderPluginConstants::POPUP_DEVICES_PRELOAD_LENGTH;
        if ($offset) {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/_listed-types-devices.html.twig',
                    ['page' => $page, 'offset' => $offset]
                );
        } else {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-devices.html.twig',
                    [
                        'page'     => $page,
                        'mode'     => 'types',
                        'listMode' => $listMode,
                        'offset'   => $defaultOffset,
                    ]
                );

            $response->headers->set('returnPath', ControllerUtil::getReturnPath($request));
            $response->headers->set('loadOffset', (string)$defaultOffset);
        }

        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }


}
