<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Service;

use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Symfony\Component\HttpFoundation\Cookie;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Topdata\TopdataTopFinderProSW6\Constants\CookieKeyNameConstants;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;

/**
 * Manages the device history USING COOKIES
 *
 * This service handles adding devices to the history, removing duplicates,
 * and ensuring the history does not exceed the configured limit.
 *
 * 05/2025 created (extracted from TopFinderController)
 */
class DeviceHistoryService_Cookie
{
    public function __construct(
        private readonly SettingsService $settingsService,
        private readonly EntityRepository $topdataDeviceRepository,
    )
    {
    }

    /**
     * Retrieves the current device history from the request cookies.
     *
     * @param Request $request The HTTP request containing the cookie.
     * @return array The current device history array.
     */
    public function getDeviceHistory(Request $request): array
    {
        $cookie = $request->cookies->get(CookieKeyNameConstants::DEVICE_HISTORY);

        if ($cookie) {
            $devicesHistory = explode(',', $cookie);
            if (!is_array($devicesHistory)) {
                $devicesHistory = [];
            }
        } else {
            $devicesHistory = [];
        }

        return $devicesHistory;
    }

    /**
     * Updates the device history by adding the specified device ID.
     * Ensures the history doesn't exceed the configured limit.
     *
     * @param array $deviceHistory The current device history array.
     * @param string $deviceId The device ID to add to the history.
     */
    private function _addToDeviceHistory(array &$deviceHistory, string $deviceId): void
    {
        // Check if the device is already in the history and remove it
        $historyKey = array_search($deviceId, $deviceHistory);
        if ($historyKey !== false) {
            unset($deviceHistory[$historyKey]);
        }

        // Add the device to the end of the history array
        $deviceHistory[] = $deviceId;

        // Ensure the history doesn't exceed the configured limit
        while (count($deviceHistory) > $this->settingsService->getInt('deviceHistoryElements')) {
            array_shift($deviceHistory);
        }
    }

    /**
     * Convenience method to get and update the device history in one call.
     *
     * @param Request $request The HTTP request containing the cookie.
     * @param string $deviceId The device ID to add to the history.
     * @return array The updated device history array.
     */
    public function getAndUpdateDeviceHistory(Request $request, string $deviceId): array
    {
        $deviceHistory = $this->getDeviceHistory($request);

        $this->_addToDeviceHistory($deviceHistory, $deviceId);

        return $deviceHistory;
    }
    /**
     * Sets the device history cookie in the response.
     *
     * @param Response $response The HTTP response object.
     * @param array $deviceHistory The device history array to store.
     */
    public function setDeviceHistoryCookie(Response $response, array $deviceHistory): void
    {
        $cookie = new Cookie(
            CookieKeyNameConstants::DEVICE_HISTORY,
            implode(',', $deviceHistory),
            time() + (86400 * 30), // 30 days
            '/',
            null,
            false,
            false,
            false,
            Cookie::SAMESITE_LAX
        );
        $response->headers->setCookie($cookie);
    }

    /**
     * Updates the device history, sets the cookie, and returns the updated history.
     *
     * @param Request $request The HTTP request object.
     * @param Response $response The HTTP response object.
     * @param string $deviceId The device ID to add to the history.
     * @return array The updated device history array.
     */
    public function updateDeviceHistoryAndSetCookie(Request $request, Response $response, string $deviceId): array
    {
        $deviceHistory = $this->getDeviceHistory($request);
        $this->_addToDeviceHistory($deviceHistory, $deviceId);
        $this->setDeviceHistoryCookie($response, $deviceHistory);

        return $deviceHistory;
    }

    /**
     * Get the device history entities from the request cookies
     *
     * @param Request $request The current request
     * @param Context $context The current context
     * @return array An array of device entities from the history
     */
    public function getDeviceEntities(Request $request, Context $context): array
    {
        $deviceIds = $this->getDeviceHistory($request);

        if (empty($deviceIds)) {
            return [];
        }

        $criteria = (new Criteria(array_reverse($deviceIds)))
            ->addFilter(new EqualsFilter('enabled', true))
            ->addAssociations(['brand', 'media']);

        return $this->topdataDeviceRepository->search($criteria, $context)->getElements();
    }
}
