<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Service;

use Doctrine\DBAL\Connection;
use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Contracts\Translation\TranslatorInterface;
use Topdata\TopdataConnectorSW6\Core\Content\Device\Agregate\DeviceCustomer\DeviceCustomerEntity;
use Topdata\TopdataTopFinderProSW6\Constants\CookieKeyNameConstants;
use Topdata\TopdataTopFinderProSW6\Util\UtilSearchTerm;
use Topdata\TopdataFoundationSW6\Util\UtilUuid;

/**
 * Class DeviceDataService
 *
 * This service is responsible for handling device data operations.
 * It provides methods for retrieving, searching, and processing device data.
 */
class DeviceDataService
{
    /**
     * Stores device ids of current user
     */
    private ?array $deviceList = null;

    public function __construct(
        private readonly Connection            $connection,
        private readonly EntityRepository      $mediaRepository,
        private readonly UrlGeneratorInterface $router,
        private readonly LoggerInterface       $logger
    )
    {
    }

    /**
     * Get the list of devices for the current user
     *
     * @param SalesChannelContext $context The sales channel context
     * @return array The list of devices
     */
    public function getCustomerDeviceList(SalesChannelContext $context): array
    {
        if (!$context->getCustomer() || $context->getCustomer()->getGuest()) {
            $this->deviceList = [];
        } elseif (null === $this->deviceList) {
            $this->deviceList = [];

            try {
                $result = $this->connection
                    ->createQueryBuilder()
                    ->select('LOWER(HEX(device_id)) as device_id, extra_info')
                    ->from('topdata_device_to_customer')
                    ->where('customer_id = 0x' . $context->getCustomer()->getId())
                    ->execute()
                    ->fetchAllAssociative();

                foreach ($result as $val) {
                    $this->deviceList[$val['device_id']] = $val['extra_info']
                        ? json_decode($val['extra_info'], true)
                        : DeviceCustomerEntity::defaultExtraInfo();
                }
            } catch (\Exception $e) {
                $this->logger->error('Error fetching customer device list: ' . $e->getMessage());
                $this->deviceList = [];
            }
        }

        return $this->deviceList;
    }

    /**
     * Get devices array filtered by brand, series, and type
     *
     * @param string $brandCode The brand code
     * @param string|null $seriesId The series ID (null for devices without series)
     * @param string|null $typeId The type ID (null for devices without type)
     * @return array The array of devices
     */
    public function getDevicesArray(string $brandCode, ?string $seriesId = null, ?string $typeId = null): array
    {
        $conditions = [];
        $conditions[] = '(topdata_device.is_enabled=1)';
        $conditions[] = '(topdata_brand.code = "' . $brandCode . '")';

        if ($seriesId === null) {
            $conditions[] = '(topdata_device.series_id IS NULL)';
        } elseif (Uuid::isValid((string)$seriesId)) {
            $conditions[] = '(topdata_device.series_id = 0x' . $seriesId . ')';
        }

        if ($typeId === null) {
            $conditions[] = '(topdata_device.type_id IS NULL)';
        } elseif (Uuid::isValid((string)$typeId)) {
            $conditions[] = '(topdata_device.type_id = 0x' . $typeId . ')';
        }

        try {
            $devices = $this->connection->fetchAllAssociative('
                SELECT topdata_device.model as name, 
                       LOWER(HEX(topdata_device.id)) as id, 
                       LOWER(HEX(topdata_device.brand_id)) as brand_id, 
                       LOWER(HEX(topdata_device.series_id)) as series_id, 
                       LOWER(HEX(topdata_device.type_id)) as type_id, 
                       topdata_device.code as code, 
                       topdata_brand.label as brand_name,
                       topdata_brand.code as brand_code,
                       topdata_series.code as series_code,
                       LOWER(HEX(topdata_device.media_id)) as media_id,
                       topdata_series.label as series_name,
                       topdata_device_type.label as type_name
                  FROM topdata_device 
                  LEFT JOIN topdata_brand ON (topdata_brand.id=topdata_device.brand_id)
                  LEFT JOIN topdata_series ON (topdata_series.id=topdata_device.series_id)
                  LEFT JOIN topdata_device_type ON (topdata_device_type.id=topdata_device.type_id)
                  WHERE ' . implode(' AND ', $conditions) . '
                  ORDER BY topdata_device.code
            ');

            return $devices;
        } catch (\Exception $e) {
            $this->logger->error('Error fetching devices array: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Load device content with media and customer device list information
     *
     * @param array $devices The devices array to load content for (passed by reference)
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @param string $listType The type of list (brand, series, types, search, history)
     * @param string $searchTerm The search term (used for search list type)
     */
    public function loadDeviceContent(
        array               &$devices,
        SalesChannelContext $salesChannelContext,
        string              $listType = 'brand',
        string              $searchTerm = ''
    ): void
    {
        $mediaIds = [];
        $devicelist = $this->getCustomerDeviceList($salesChannelContext);

        foreach ($devices as $key => $device) {
            if ($listType == 'search' && $searchTerm) {
                $devices[$key]['path'] = $this->router->generate('frontend.top-finder-api.popup-device-new', [
                    'deviceCode' => $device['code'],
                    'listType'   => $listType,
                    'term'       => $searchTerm,
                ]);
            } else {
                $devices[$key]['path'] = $this->router->generate('frontend.top-finder-api.popup-device-new', [
                    'deviceCode' => $device['code'],
                    'listType'   => $listType,
                ]);
            }

            if ($device['media_id']) {
                $mediaIds[] = $device['media_id'];
            }

            if (isset($devicelist[$device['id']]) && isset($devicelist[$device['id']]['devices'])) {
                $devices[$key]['devicelist'] = count($devicelist[$device['id']]['devices']);
            } else {
                $devices[$key]['devicelist'] = -1;
            }
        }

        if (count($mediaIds)) {
            try {
                $medias = $this->mediaRepository->search(new Criteria($mediaIds), $salesChannelContext->getContext());

                foreach ($devices as $key => $device) {
                    if ($device['media_id'] && $medias->get($device['media_id'])) {
                        $devices[$key]['media_url'] = $medias->get($device['media_id'])->getUrl();
                    }
                }
            } catch (\Exception $e) {
                $this->logger->error('Error loading media for devices: ' . $e->getMessage());
            }
        }
    }

    /**
     * Find devices based on a search term
     *
     * @param string $term The search term
     * @param int $limit Maximum number of results to return (0 for no limit)
     * @param int $offset Offset for pagination
     * @return array An array of matching devices
     */
    public function findDevices(string $term, int $limit = 0, int $offset = 0): array
    {
        $filteredTerm = UtilSearchTerm::filterTerm($term);
        if ($filteredTerm !== $term) {
            $term = $filteredTerm;
        }

        if (!$term) {
            return [];
        }

        $SQLadd = '';
        if ($limit > 0) {
            $SQLadd = " LIMIT $limit";
            if ($offset > 0) {
                $SQLadd .= " OFFSET $offset";
            }
        }

        $terms = UtilSearchTerm::getTermsFromString($term);

        $WhereAdd = '';
        if (count($terms)) {
            $WhereAdd = 'OR(';
            foreach ($terms as $key => $value) {
                $terms[$key] = '(topdata_device.keywords LIKE "%' . $value . '%")';
            }
            $WhereAdd .= implode('AND', $terms);
            $WhereAdd .= ')';
        }

        try {
            $devices = $this->connection->fetchAllAssociative('
                SELECT topdata_device.model as name, 
                       LOWER(HEX(topdata_device.id)) as id, 
                       LOWER(HEX(topdata_device.brand_id)) as brand_id, 
                       LOWER(HEX(topdata_device.series_id)) as series_id, 
                       LOWER(HEX(topdata_device.type_id)) as type_id, 
                       topdata_device.code as code, 
                       topdata_brand.label as brand_name,
                       topdata_brand.code as brand_code,
                       topdata_series.code as series_code,
                       LOWER(HEX(topdata_device.media_id)) as media_id,
                       topdata_series.label as series_name,
                       topdata_device_type.label as type_name
                  FROM topdata_device 
                  LEFT JOIN topdata_brand ON (topdata_brand.id=topdata_device.brand_id)
                  LEFT JOIN topdata_series ON (topdata_series.id=topdata_device.series_id)
                  LEFT JOIN topdata_device_type ON (topdata_device_type.id=topdata_device.type_id)
                  WHERE (topdata_device.is_enabled=1)AND((topdata_device.keywords LIKE "%' . $term . '%")' . $WhereAdd . ')
                  ORDER BY `topdata_device`.`code` ASC ' . $SQLadd);

            return count($devices) ? $devices : [];
        } catch (\Exception $e) {
            $this->logger->error('Error finding devices: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Count the number of devices matching a search term
     *
     * @param string $term The search term
     * @return int The number of matching devices
     */
    public function countFindDevices(string $term): int
    {
        $filteredTerm = UtilSearchTerm::filterTerm($term);
        if ($filteredTerm !== $term) {
            $term = $filteredTerm;
        }

        if (!$term) {
            return 0;
        }

        $terms = UtilSearchTerm::getTermsFromString($term);

        $WhereAdd = '';
        if (count($terms)) {
            $WhereAdd = 'OR(';
            foreach ($terms as $key => $value) {
                $terms[$key] = '(topdata_device.keywords LIKE "%' . $value . '%")';
            }
            $WhereAdd .= implode('AND', $terms);
            $WhereAdd .= ')';
        }

        try {
            $devicesCount = $this->connection->fetchAllAssociative('
                SELECT COUNT(*) as cnt
                  FROM topdata_device 
                  WHERE (topdata_device.is_enabled=1)AND((topdata_device.keywords LIKE "%' . $term . '%")' . $WhereAdd . ')
            ');

            return isset($devicesCount[0]['cnt']) ? (int)$devicesCount[0]['cnt'] : 0;
        } catch (\Exception $e) {
            $this->logger->error('Error counting devices: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get device history from request or provided array
     *
     * @param Request $request The request object
     * @param array|null $deviceHistory Optional device history array
     * @return array Array of valid device IDs
     */
    public function getDeviceHistory(Request $request, ?array $deviceHistory = null): array
    {
        // Use passed device history if provided, otherwise fall back to cookie
        $cookieIds = [];
        if ($deviceHistory !== null) {
            $cookieIds = $deviceHistory;
        } else {
            // Fallback: Read cookie directly if history not passed
            $cookie = $request->cookies->get(CookieKeyNameConstants::DEVICE_HISTORY);
            if ($cookie) {
                $cookieIds = explode(',', $cookie);
            }
        }

        $deviceIds = [];
        foreach ($cookieIds as $value) {
            if (Uuid::isValid($value)) {
                $deviceIds[] = $value;
            }
        }

        return $deviceIds;
    }

    /**
     * Get devices by IDs
     *
     * @param array $deviceIds Array of device IDs
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return array Array of devices sorted according to the input deviceIds order
     */
    public function getDevicesByIds(array $deviceIds, SalesChannelContext $salesChannelContext): array
    {
        if (empty($deviceIds)) {
            return [];
        }

        try {
            $devices = $this->connection->fetchAllAssociative('
                SELECT topdata_device.model as name, 
                       LOWER(HEX(topdata_device.id)) as id, 
                       LOWER(HEX(topdata_device.brand_id)) as brand_id, 
                       LOWER(HEX(topdata_device.series_id)) as series_id, 
                       LOWER(HEX(topdata_device.type_id)) as type_id, 
                       topdata_device.code as code, 
                       topdata_brand.label as brand_name,
                       topdata_brand.code as brand_code,
                       topdata_series.code as series_code,
                       LOWER(HEX(topdata_device.media_id)) as media_id,
                       topdata_series.label as series_name,
                       topdata_device_type.label as type_name
                  FROM topdata_device 
                  LEFT JOIN topdata_brand ON (topdata_brand.id=topdata_device.brand_id)
                  LEFT JOIN topdata_series ON (topdata_series.id=topdata_device.series_id)
                  LEFT JOIN topdata_device_type ON (topdata_device_type.id=topdata_device.type_id)
                  WHERE topdata_device.id IN (0x' . implode(',0x', $deviceIds) . ')
            ');

            // Sort devices according to the input deviceIds order
            $deviceIds = array_reverse($deviceIds);
            $sortedDevices = [];
            foreach ($deviceIds as $id) {
                foreach ($devices as $device) {
                    if ($id === $device['id']) {
                        $sortedDevices[] = $device;
                    }
                }
            }

            return $sortedDevices;
        } catch (\Exception $e) {
            $this->logger->error('Error getting devices by IDs: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get topdata devices filtered by brand, series and type
     *
     * @param string $brandId The brand ID
     * @param string|int $seriesId The series ID or -1 for devices without series
     * @param string|int $typeId The type ID or -1 for devices without type
     * @param bool $codeAsKey Whether to use device code as array key
     * @return array The filtered devices
     */
    public function getFilteredDevices($brandId, $seriesId, $typeId, bool $codeAsKey = true): array
    {
        $startTime = microtime(true);
        $this->logger->info('DeviceDataService::getFilteredDevices - Start', [
            'brandId'   => $brandId,
            'seriesId'  => $seriesId,
            'typeId'    => $typeId,
            'codeAsKey' => $codeAsKey
        ]);

        $return = [];
        $params = [];

        $params[] = 'is_enabled = 1';

        if (UtilUuid::isValidUuid($brandId)) {
            $params[] = 'brand_id = 0x' . $brandId;
        } else {
            return $return;
        }

        if ($seriesId == -1) {
            $params[] = 'series_id is null';
        } elseif (UtilUuid::isValidUuid($seriesId)) {
            $params[] = 'series_id = 0x' . $seriesId;
        }

        if ($typeId == -1) {
            $params[] = 'type_id is null';
        } elseif (UtilUuid::isValidUuid($typeId)) {
            $params[] = 'type_id = 0x' . $typeId;
        }

        $queryStartTime = microtime(true);
        $this->logger->info('DeviceDataService::getFilteredDevices - Executing query', ['params' => $params]);

        try {
            $devices = $this->connection->fetchAllAssociative(
                'SELECT code, model FROM `topdata_device`'
                . ' WHERE (' . implode(') AND (', $params) . ')'
                . ' ORDER BY model ASC'
            );

            $queryEndTime = microtime(true);
            $deviceCount = count($devices);
            $this->logger->info('DeviceDataService::getFilteredDevices - Query finished', [
                'deviceCount' => $deviceCount,
                'duration_ms' => ($queryEndTime - $queryStartTime) * 1000
            ]);

            if ($codeAsKey) {
                foreach ($devices as $device) {
                    $return[$device['code']] = $device['model'];
                }
            } else {
                $return = $devices;
            }

            $endTime = microtime(true);
            $this->logger->info('DeviceDataService::getFilteredDevices - End', [
                'resultCount'       => count($return),
                'total_duration_ms' => ($endTime - $startTime) * 1000
            ]);

            return $return;
        } catch (\Exception $e) {
            $this->logger->error('Error in getFilteredDevices: ' . $e->getMessage(), [
                'brandId'   => $brandId,
                'seriesId'  => $seriesId,
                'typeId'    => $typeId
            ]);
            return [];
        }
    }
}