<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\Popup;

use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataTopFinderProSW6\Service\DeviceDataService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\List\ListPage;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\List\TRASH____ListPageLoader;

/**
 * Class PopupSeriesDevicesPageLoader
 *
 * This class is responsible for loading the popup series devices page.
 * It retrieves devices for a specific series and prepares the page for display in a popup.
 */
class PopupSeriesDevicesPageLoader
{

    public function __construct(
        private readonly GenericPageLoader     $genericPageLoader,
        private readonly DeviceDataService     $deviceDataService,
        private readonly SettingsService       $settingsService,
        private readonly Translator            $translator,
        private readonly UrlGeneratorInterface $router,
        private readonly LoggerInterface       $logger,
    )
    {
    }


    /**
     * Load the popup series devices page
     *
     * This method loads the generic page, retrieves a series by code,
     * gets the devices for that series, and configures the page for display in a popup.
     *
     * @param mixed $request The current request
     * @param mixed $salesChannelContext The sales channel context
     * @return ListPage The loaded popup series devices page
     */
    public function load($request, $salesChannelContext): ListPage
    {
        try {
            // Get the series code from the request attributes
            $seriesCode = $request->attributes->get('code');
            
            // Load the generic page
            $page = $this->genericPageLoader->load($request, $salesChannelContext);
            
            // Create the list page from the generic page
            $listPage = new ListPage();
            // Copy all properties from the generic page to the list page
            foreach (get_object_vars($page) as $property => $value) {
                $listPage->$property = $value;
            }
            
            // Get the series by code
            $series = $this->seriesDataService->getSeriesByCode($seriesCode, $salesChannelContext);
            
            // If the series is not found, return the empty page
            if (!$series) {
                $this->logger->warning('Series not found with code: ' . $seriesCode);
                return $listPage;
            }
            
            // Get the brand from the series
            $brand = $series->getBrand();
            
            // If the brand is not found, return the empty page
            if (!$brand) {
                $this->logger->warning('Brand not found for series with code: ' . $seriesCode);
                return $listPage;
            }
            
            // Set the title for the popup series devices page
            $listPage->setTitle($this->translator->trans('topdata-topfinder.SEO.seriesPageTitle', [
                '%brand%' => '<b>' . $brand->getName() . '</b>',
                '%series%' => $series->getLabel(),
            ]));
            
            // Get devices for the series
            $devices = $this->deviceDataService->getDevicesArray($brand->getCode(), $series->getId());
            
            // Load device content (enrich the device data)
            $this->deviceDataService->loadDeviceContent($devices, $salesChannelContext, 'series');
            
            // Set the devices to the page
            $listPage->setDevices($devices);
            
            // Set the page type to series
            $listPage->setPageType(TRASH____ListPageLoader::TYPE_SERIES);
            
            // Set the page to show devices
            $listPage->setShow(['devices' => true]);
            
            // Generate the popup path
            $letter = ($brand->getCode())[0];
            if (preg_match('/^[0-9]{1}$/', $letter)) {
                $letter = '0';
                $letterStr = '0-9';
            } else {
                $letterStr = strtoupper($letter);
            }
            
            $popupPath = [];
            
            $popupPath[] = [
                'name' => $letterStr,
                'path' => $this->router->generate('frontend.top-finder-api.popup-letter', ['letter' => $letter]),
            ];
            
            $popupPath[] = [
                'name' => $brand->getName(),
                'path' => $this->router->generate('frontend.top-finder.popup_brand', ['brandCode' => $brand->getCode()]),
            ];
            
            $popupPath[] = [
                'name' => $this->translator->trans('topdata-topfinder.popup.brandSeries'),
                'path' => $this->router->generate('frontend.top-finder.popup_brand_series', ['brandCode' => $brand->getCode()]),
            ];
            
            $popupPath[] = [
                'name' => $series->getLabel(),
            ];
            
            $listPage->popupPath = $popupPath;
            
            return $listPage;
        } catch (\Exception $e) {
            $this->logger->error('Error loading popup series devices page: ' . $e->getMessage());
            throw $e;
        }
    }
}