<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Controller;

use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Controller\StorefrontController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\EventListener\AbstractSessionListener;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\Routing\Annotation\Route;
use Topdata\TopdataFoundationSW6\Util\UtilJsonResponse;
use Topdata\TopdataTopFinderProSW6\Constants\CookieKeyNameConstants;
use Topdata\TopdataTopFinderProSW6\Service\DeviceDataService;
use Topdata\TopdataTopFinderProSW6\Service\DeviceHistoryService_Cookie;
use Topdata\TopdataTopFinderProSW6\Service\DeviceToCustomerService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Device\DevicePage;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\BrandLetterPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\CompatibleDevicesWidgetPageV2Loader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\DevicePageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\OneBrandLetterPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\PopupDeviceWidgetPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\DeviceListPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\DeviceSearchPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\Popup\PopupBrandsPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\Popup\PopupHistoryPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\Popup\PopupSeriesDevicesPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\Popup\PopupTypeDevicesPageLoader;
use Topdata\TopdataTopFinderProSW6\Util\ControllerUtil;

/**
 * Controller for handling TopFinder API requests.
 *
 * This controller provides various endpoints for the TopFinder functionality,
 * including retrieving series, types, devices, and handling device-related actions.
 * It was created in 10/2024, extracted from TopFinderController.
 *
 * TODO: this controller is too big and should be refactored into smaller, more focused classes
 */
#[Route(defaults: ['_routeScope' => ['storefront']])]
class TopFinderApiController extends StorefrontController
{


    public function __construct(
        private readonly EntityRepository                    $topdataBrandRepository,
        private readonly EntityRepository                    $topdataSeriesRepository,
        private readonly EntityRepository                    $topdataDeviceTypeRepository,
        private readonly DeviceListPageLoader                $deviceListPageLoader,
        private readonly DeviceSearchPageLoader              $deviceSearchPageLoader,
        private readonly CompatibleDevicesWidgetPageV2Loader $compatibleDevicesWidgetPageV2Loader,
        private readonly PopupDeviceWidgetPageLoader         $popupDeviceWidgetPageLoader,
        private readonly BrandLetterPageLoader               $brandLetterPageLoader,
        private readonly OneBrandLetterPageLoader            $oneBrandLetterPageLoader,
        private readonly DevicePageLoader                    $devicePageLoader,
        private readonly DeviceToCustomerService             $deviceToCustomerService,
        private readonly DeviceHistoryService_Cookie         $deviceHistoryService_Cookie,
        private readonly PopupBrandsPageLoader               $popupBrandsPageLoader,
        private readonly PopupSeriesDevicesPageLoader        $popupSeriesDevicesPageLoader,
        private readonly PopupTypeDevicesPageLoader          $popupTypeDevicesPageLoader,
        private readonly PopupHistoryPageLoader              $popupHistoryPageLoader,
        private readonly DeviceDataService                 $deviceDataService
    )
    {
    }

    /**
     * Retrieves series for a given brand code.
     *
     * This method is called via AJAX to populate a dropdown with the series of a chosen brand.
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return JsonResponse JSON response containing series data or not found status
     */
    #[Route(
        path: '/top-finder-api/get-series/{brandCode}',
        name: 'frontend.top-finder-api.get-series',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET', 'POST']
    )]
    public function getSeriesAction(string $brandCode, SalesChannelContext $salesChannelContext): JsonResponse
    {
        if (!trim($brandCode)) {
            return new JsonResponse(['found' => false]);
        }

        $brand = $this
            ->topdataBrandRepository
            ->search(
                (new Criteria())
                    ->addFilter(new EqualsFilter('enabled', true))
                    ->addFilter(new EqualsFilter('code', $brandCode)),
                $salesChannelContext->getContext()
            )
            ->getEntities()
            ->first();

        if (!$brand) {
            return new JsonResponse(['found' => false]);
        }

        $brandId = $brand->getId();
        $seriesMap = $this->_getSeriesMap($brandId);

        if (!count($seriesMap)) {
            return new JsonResponse(['found' => false]);
        }

        return new JsonResponse([
            'found' => true,
            'items' => $seriesMap,
        ]);
    }

    /**
     * Retrieves types for a given brand code.
     *
     * This method is called via AJAX to populate a dropdown with the types of a chosen brand.
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return JsonResponse JSON response containing types data or not found status
     */
    #[Route(
        path: '/top-finder-api/get-types/{brandCode}',
        name: 'frontend.top-finder-api.get-types',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET', 'POST'],
    )]
    public function getTypes(string $brandCode, SalesChannelContext $salesChannelContext): JsonResponse
    {
        if (!trim($brandCode)) {
            return new JsonResponse(['found' => false]);
        }

        $brand = $this
            ->topdataBrandRepository
            ->search(
                (new Criteria())
                    ->addFilter(new EqualsFilter('enabled', true))
                    ->addFilter(new EqualsFilter('code', $brandCode)),
                $salesChannelContext->getContext()
            )
            ->getEntities()
            ->first();

        if (!$brand) {
            return new JsonResponse(['found' => false]);
        }

        $brandId = $brand->getId();

        $typesArray = $this->_getTypesArray($brandId);

        if (!count($typesArray)) {
            return new JsonResponse(['found' => false]);
        }

        return new JsonResponse([
            'found' => true,
            'items' => $typesArray,
        ]);
    }

    /**
     * Retrieves devices for a given brand, type, and series.
     *
     * This loads the devices of a brand (eg "canon") and optionally a series (eg "pixma") and a type (eg "printer")
     * it is called via ajax to populate a dropdown with the devices of a chosen brand
     *
     * This method is called via AJAX to populate a dropdown with the devices of a chosen brand, type, and series.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return JsonResponse JSON response containing devices data or not found status
     */
    #[Route(
        path: '/top-finder-api/get-devices/{brandCode}/{typeCode}/{seriesCode}',
        name: 'frontend.top-finder-api.get-devices',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET', 'POST']
    )]
    public function getDevices(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $brandCode = $this->_resolveCode((string)$request->get('brandCode'), $this->topdataBrandRepository, $salesChannelContext->getContext());
        $seriesCode = $this->_resolveCode((string)$request->get('seriesCode'), $this->topdataSeriesRepository, $salesChannelContext->getContext());
        $typeCode = $this->_resolveCode((string)$request->get('typeCode'), $this->topdataDeviceTypeRepository, $salesChannelContext->getContext());

        if (($brandCode === false) || ($seriesCode === false) || ($typeCode === false)) {
            return new JsonResponse(['found' => false]);
        }

        $devicesArray = $this->deviceDataService->getFilteredDevices($brandCode, $seriesCode, $typeCode);

        if (!count($devicesArray)) {
            return new JsonResponse(['found' => false]);
        }

        return new JsonResponse([
            'found' => true,
            'items' => $devicesArray,
        ]);
    }

    /**
     * Load more devices in the device search results.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response containing the HTML for additional search results
     */
    #[Route(
        path: '/top-finder-api/device-search-more',
        name: 'frontend.top-finder-api.device-search-more',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function deviceSearchMore(Request $request, SalesChannelContext $context): JsonResponse
    {
        $page = $this->deviceSearchPageLoader->loadPage($request, $context);

        $result = ['success' => true];
        $result['html'] = $this->renderStorefront(
            '@Storefront/storefront/page/topfinder/device-search-more.html.twig',
            ['page' => $page]
        )->getContent();

        return new JsonResponse($result);
    }

    /**
     * List devices for a specific brand.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChanelContext The sales channel context
     * @return JsonResponse JSON response containing the HTML for the brand devices list
     */
    // listBrandDevices action has been refactored into separate actions in TopdataDeviceApiController

    /**
     * Get compatible devices for a specific product.
     *
     * @param string $productid The ID of the product
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return JsonResponse JSON response containing the HTML for compatible devices
     */
    #[Route(
        path: '/top-finder-api/compatible-devices/{productid}',
        name: 'frontend.top-finder-api.compatible-devices',
        defaults: ['productid' => '', 'XmlHttpRequest' => true],
        methods: ['GET', 'POST']
    )]
    public function compatibleDevices(string $productid, Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        if (!$productid) {
            return UtilJsonResponse::error('Something went wrong');
        }

        $result = ['success' => true];
        $page = $this->compatibleDevicesWidgetPageV2Loader->load($request, $salesChannelContext, $productid);
        $result['html'] = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/compatible-devices/index.html.twig', [
            'page' => $page,
        ])->getContent();
        $result['title'] = $page->product->getName() . ' ' . $this->trans('topdata-topfinder.productDevices');

        if ($result) {
            return new JsonResponse($result);
        }
    }

    /**
     * Get information for a new device popup.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return JsonResponse JSON response containing the HTML and data for the new device popup
     */
    #[Route(
        path: '/top-finder-api/get-popup-device-new',
        name: 'frontend.top-finder-api.popup-device-new',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET', 'POST']
    )]
    public function popupDeviceNew(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $page = $this
            ->popupDeviceWidgetPageLoader
            ->loadNew($request, $salesChannelContext);

        $result = ['success' => true];

        if (!$page->device) {
            return new JsonResponse(['status' => 'false', 'message' => 'no Device Found'], 500);
        }

        $result['deviceId'] = $page->device->getId();
        $result['deviceUrl'] = $page->deviceUrl;
        $result['inDeviceList'] = $page->device->getInDeviceList();
        $result['userLogged'] = $page->other['userLogged'];
        if ($result['userLogged']) {
            $result['pathListAdd'] = $page->other['pathListAdd'];
            $result['pathListRemove'] = $page->other['pathListRemove'];
        }

        //        $devices = $this->getDevicesArray(
        //            $page->device->getBrand()->getId(),
        //            $page->device->getSeries() ? $page->device->getSeries()->getId() : 0,
        //            0
        //            //$page->device->getType() ? $page->device->getType()->getId() : false
        //        );
        //
        //        $series = $this->getSeriesArray($page->device->getBrand()->getId());
        //        $types = $this->getTypesArray($page->device->getBrand()->getId());

        //        $popupHome = [
        //            'brandskey' => $page->device->getBrand()->getCode(),
        //            'series' => $series,
        //            'serieskey' => $page->device->getSeries() ? $page->device->getSeries()->getCode() : null,
        //            'types' => $types,
        //            'typeskey' => $page->device->getType() ? $page->device->getType()->getCode() : null,
        //            'devices' => $devices,
        //            'deviceskey' => $page->device->getCode()
        //        ];
        //
        //        $result['popupHome'] = $popupHome;

        $devicelist = $this->deviceToCustomerService->getDeviceListInfo($salesChannelContext);

        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/box-device.html.twig',
                [
                    'page'       => $page,
                    'devicelist' => $devicelist,
                ]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    /**
     * List devices for a specific letter in a popup.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChanelContext The sales channel context
     * @return JsonResponse JSON response containing the HTML for the letter-specific device list
     */
    #[Route(
        path: '/top-finder-api/popup-letter/{letter}',
        name: 'frontend.top-finder-api.popup-letter',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function listPopupLetter(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $letter = $request->get('letter');
        if (!preg_match('/^[a-z0]{1}$/', $letter)) {
            throw new BadRequestHttpException('letter'/*, '/letter'*/);
        }

        $page = $this->brandLetterPageLoader->loadJson($request, $salesChanelContext, $letter);
        $result = [];
        $result['success'] = true;
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-letter.html.twig',
                ['page' => $page]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route(
        path: '/top-finder-api/popup-all-brands',
        name: 'frontend.top-finder-api.popup-all-brands',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function listPopupAllBrands(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $page = $this->popupBrandsPageLoader->load($request, $salesChanelContext);

        $result = [];
        $result['success'] = true;
        $result['html'] = $this->renderStorefront(
            '@TopdataTopFinderProSW6/storefront/widget/popup-form/all-brands.html.twig',
            ['page' => $page]
        )->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    /**
     * FIXME? UNUSED?
     */
    #[Route(
        path: '/top-finder-api/popup-brandq/{code}',
        name: 'frontend.top-finder.popup_brandq',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function listPopupBrandDevices(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $code = $request->get('code');
        if (!preg_match('/^[a-z0-9_-]+$/', $code)) {
            throw new BadRequestHttpException('code'/*, '/code'*/);
        }

        //        $page = $this->listPageLoader->loadBrandDevicesJson($request, $salesChanelContext);

        $page = $this->oneBrandLetterPageLoader->loadJson(
            $request,
            $salesChanelContext,
            $code,
            'all'
        );

        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);

        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);

        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-devices.html.twig',
                ['page' => $page, 'mode' => 'all', 'listMode' => $listMode]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route(
        path: '/top-finder-api/popup-series/{code}',
        name: 'frontend.top-finder-api.popup-series',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function listPopupSerieDevices($code, Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        if (!preg_match('/^[a-z0-9_-]+$/', $code)) {
            throw new BadRequestHttpException('code'/*, '/code'*/);
        }
        $page = $this->popupSeriesDevicesPageLoader->load($request, $salesChanelContext);

        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);

        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devices.html.twig',
                ['page' => $page, 'listMode' => $listMode, 'hideMenuSeries' => true]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route(
        path: '/top-finder-api/popup-type/{code}',
        name: 'frontend.top-finder-api.popup-type',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function listPopupTypeDevices($code, Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        if (!preg_match('/^[a-z0-9_-]+$/', $code)) {
            throw new BadRequestHttpException('code'/*, '/code'*/);
        }
        $page = $this->popupTypeDevicesPageLoader->load($request, $salesChanelContext);

        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);

        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devices.html.twig',
                ['page' => $page, 'listMode' => $listMode]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route(
        path: '/top-finder-api/popup-devicelist',
        name: 'frontend.top-finder-api.popup-devicelist',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET'],
    )]
    public function popupDeviceList(Request $request, SalesChannelContext $context): JsonResponse
    {
        $page = $this->deviceListPageLoader->loadJson($request, $context);
        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);
        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/index.html.twig',
                ['page' => $page, 'listMode' => $listMode]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route(
        path: '/top-finder-api/popup-devicelist-device',
        name: 'frontend.top-finder-api.popup-devicelist-device',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function popupDeviceListDevice(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');

        if (false === Uuid::isValid($id)) {
            return UtilJsonResponse::error('Invalid device ID.');
        }

        $page = $this->deviceListPageLoader->loadDeviceJson($request, $context);

        if (is_null($page->device)) {
            return UtilJsonResponse::error('Device not found.');
        }

        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/device.html.twig',
                ['page' => $page]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route(
        path: '/top-finder-api/popup-devicelist-subdevice-add',
        name: 'frontend.top-finder-api.popup-devicelist-subdevice-add',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function popupDeviceListDeviceSubdeviceAdd(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');

        if (false === Uuid::isValid($id)) {
            return UtilJsonResponse::error('Something went wrong.');
        }

        $page = $this->deviceListPageLoader->loadAddSubdeviceJson($request, $context);

        if (is_null($page->device)) {
            return UtilJsonResponse::error('Something went wrong.');
        }

        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/subdevice-add.html.twig',
                ['page' => $page]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route(
        path: '/top-finder-api/popup-devicelist-subdevice-add-confirmed',
        name: 'frontend.top-finder-api.popup-devicelist-subdevice-add-confirmed',
        defaults: ['XmlHttpRequest' => true],
        methods: ['POST']
    )]
    public function popupDeviceListDeviceSubdeviceAddConfirmed(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');

        if (false === Uuid::isValid($id)) {
            return UtilJsonResponse::error('Something went wrong.');
        }

        $page = $this->deviceListPageLoader->addSubdeviceJson($request, $context);

        if (is_null($page->device)) {
            return UtilJsonResponse::error('Something went wrong.');
        }

        $result = [];
        $result['success'] = true;
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/device.html.twig',
                ['page' => $page]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route(
        path: '/top-finder-api/popup-devicelist-subdevice-edit',
        name: 'frontend.top-finder-api.popup-devicelist-subdevice-edit',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function popupDeviceListDeviceSubdeviceEdit(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');

        if (false === Uuid::isValid($id)) {
            return UtilJsonResponse::error('Something went wrong.');
        }

        $page = $this->deviceListPageLoader->loadEditSubdeviceJson($request, $context);

        if (is_null($page->device)) {
            return UtilJsonResponse::error('Something went wrong.');
        }

        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/subdevice-edit.html.twig',
                ['page' => $page]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route(
        path: '/top-finder-api/popup-devicelist-subdevice-edit-confirmed',
        name: 'frontend.top-finder-api.popup-devicelist-subdevice-edit-confirmed',
        defaults: ['XmlHttpRequest' => true],
        methods: ['POST']
    )]
    public function popupDeviceListDeviceSubdeviceEditConfirmed(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');

        if (false === Uuid::isValid($id)) {
            return UtilJsonResponse::error('Something went wrong.');
        }

        $page = $this->deviceListPageLoader->editSubdeviceJson($request, $context);

        if (is_null($page->device)) {
            return UtilJsonResponse::error('Something went wrong.');
        }

        $result = [];
        $result['success'] = true;
        $result['html'] = $this
            ->renderStorefront(
            //                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/device.html.twig',
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/subdevice-edit.html.twig',
                ['page' => $page]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route(
        path: '/top-finder-api/popup-devicelist-subdevice-remove',
        name: 'frontend.top-finder-api.popup-devicelist-subdevice-remove',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function popupDeviceListDeviceSubdeviceRemove(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');

        if (false === Uuid::isValid($id)) {
            return UtilJsonResponse::error('Something went wrong.');
        }

        $page = $this->deviceListPageLoader->removeSubdeviceJson($request, $context);

        //        if(is_null($page->device)) {
        //            return new JsonResponse(['success'=>false]);
        //        }

        $result = [];
        $result['success'] = true;
        if ($page->fallBack) {
            $result['html'] = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/index.html.twig',
                    ['page' => $page]
                )
                ->getContent();
        } else {
            $result['html'] = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/device.html.twig',
                    ['page' => $page]
                )
                ->getContent();
        }

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route(
        path: '/top-finder-api/popup-history',
        name: 'frontend.top-finder-api.popup-history',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET']
    )]
    public function listPopupHistory(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        // Use service to get history
        $deviceHistory = $this->deviceHistoryService_Cookie->getDeviceHistory($request);
        // Pass history to loader
        $page = $this->popupHistoryPageLoader->load($request, $salesChanelContext, $deviceHistory);

        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);

        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devices.html.twig',
                ['page' => $page, 'listMode' => $listMode]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route(
        path: '/top-finder-api/device-filters/{deviceCode}/{propertyGroupOptionId}',
        name: 'frontend.top-finder-api.device-tab-filters',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET', 'POST']
    )]
    public function getDeviceTabFilters(Request $request, SalesChannelContext $context): JsonResponse
    {
        // Allows to fetch only aggregations over the gateway.
        $request->request->set('only-aggregations', true);

        // Allows to convert all post-filters to filters. This leads to the fact that only aggregation values are returned, which are combinable with the previous applied filters.
        $request->request->set('reduce-aggregations', true);

        /** @var DevicePage $page */
        $page = $this->devicePageLoader->loadDevicePageData($request, $context, true);
        //$listing = $page->listing->getResult();
        $listing = $page->listing;

        $mapped = [];
        foreach ($listing->getAggregations() as $aggregation) {
            $mapped[$aggregation->getName()] = $aggregation;
        }

        $response = new JsonResponse($mapped);
        $response->headers->set(AbstractSessionListener::NO_AUTO_CACHE_CONTROL_HEADER, '1');

        return $response;
    }

    #[Route(
        path: '/top-finder-api/device-filters/{deviceCode}',
        name: 'frontend.top-finder-api.device-filters',
        defaults: ['XmlHttpRequest' => true],
        methods: ['GET', 'POST']
    )]
    public function getDeviceFilters(Request $request, SalesChannelContext $context): JsonResponse
    {
        // Allows to fetch only aggregations over the gateway.
        $request->request->set('only-aggregations', true);

        // Allows to convert all post-filters to filters. This leads to the fact that only aggregation values are returned, which are combinable with the previous applied filters.
        $request->request->set('reduce-aggregations', true);

        /** @var DevicePage $page */
        $page = $this->devicePageLoader->loadDevicePageData($request, $context, true);
        //$listing = $page->listing->getResult();
        $listing = $page->listing;

        $mapped = [];
        foreach ($listing->getAggregations() as $aggregation) {
            $mapped[$aggregation->getName()] = $aggregation;
        }

        $response = new JsonResponse($mapped);
        $response->headers->set(AbstractSessionListener::NO_AUTO_CACHE_CONTROL_HEADER, '1');

        return $response;
    }

    /**
     * private helper
     * Retrieves an array of types for a given brand ID.
     *
     * @param string $brandId The ID of the brand
     * @return array An array of types, with type code as key and label as value
     */
    private function _getTypesArray(string $brandId): array
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('enabled', true));
        $criteria->addFilter(new EqualsFilter('brandId', $brandId));
        $criteria->addSorting(new FieldSorting('label', FieldSorting::ASCENDING));
        $types = $this->topdataDeviceTypeRepository->search($criteria, Context::createDefaultContext())->getEntities();

        if (!count($types)) {
            return [];
        }

        $typesArray = [];
        foreach ($types as $type) {
            $typesArray[$type->getCode()] = $type->getLabel();
        }

        return $typesArray;
    }

    /**
     * Private helper function to resolve a code to an id.
     *
     * @param string $code The code to resolve
     * @param EntityRepository $repository The repository to search in
     * @param Context $context The context for the search
     * @return string|bool The resolved ID or false if not found
     */
    private function _resolveCode($code, $repository, $context)
    {
        if ($code == '0') {
            return '0';
        }

        if ($code == '-1') {
            return '-1';
        }

        if (!trim($code)) {
            return false;
        }

        $entity = $repository
            ->search(
                (new Criteria())
                    ->addFilter(new EqualsFilter('enabled', true))
                    ->addFilter(new EqualsFilter('code', $code)),
                $context
            )
            ->getEntities()
            ->first();

        if (!$entity) {
            return false;
        }

        return $entity->getId();
    }

    /**
     * private helper
     * Retrieves an array of series for a given brand ID.
     *
     * @param string $brandId The ID of the brand
     * @return array An array of series, with series code as key and label as value
     */
    private function _getSeriesMap(string $brandId): array
    {
        $seriesArray = [];

        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('enabled', true));
        $criteria->addFilter(new EqualsFilter('brandId', $brandId));
        $criteria->addSorting(new FieldSorting('label', FieldSorting::ASCENDING));
        $series = $this->topdataSeriesRepository->search($criteria, Context::createDefaultContext())->getEntities();

        if (!count($series)) {
            return $seriesArray;
        }

        foreach ($series as $serie) {
            $seriesArray[$serie->getCode()] = $serie->getLabel();
        }

        return $seriesArray;
    }
}
