<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\Popup;

use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataTopFinderProSW6\Service\DeviceDataService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\List\ListPage;

/**
 * Class PopupHistoryPageLoader
 *
 * This class is responsible for loading the popup history page.
 * It retrieves devices from the user's browsing history and prepares the page for display in a popup.
 */
class PopupHistoryPageLoader
{
    public function __construct(
        private readonly GenericPageLoader     $genericPageLoader,
        private readonly DeviceDataService     $deviceDataService,
        private readonly SettingsService       $settingsService,
        private readonly Translator            $translator,
        private readonly UrlGeneratorInterface $router,
        private readonly LoggerInterface       $logger,
    )
    {
    }


    /**
     * Load the popup history page
     *
     * This method loads the generic page, retrieves devices from the user's browsing history,
     * and configures the page for display in a popup.
     *
     * @param mixed $request The current request
     * @param mixed $salesChannelContext The sales channel context
     * @param array|null $deviceHistory Optional device history array
     * @return ListPage The loaded popup history page
     */
    public function load($request, $salesChannelContext, ?array $deviceHistory = null): ListPage
    {
        try {
            // Load the generic page
            $page = $this->genericPageLoader->load($request, $salesChannelContext);

            // Create the list page from the generic page
            $listPage = new ListPage();
            // Copy all properties from the generic page to the list page
            foreach (get_object_vars($page) as $property => $value) {
                $listPage->$property = $value;
            }

            // Set the title for the popup history page
            $listPage->setTitle($this->translator->trans('topdata-topfinder.popup.deviceHistoryTitle'));

            // Set the popup path
            $popupPath = [
                [
                    'name' => $this->translator->trans('topdata-topfinder.popup.deviceHistoryTitle'),
                ]
            ];
            $listPage->popupPath = $popupPath;

            // Check if device history is enabled in settings
            if (!$this->settingsService->getBool('showDeviceHistory')) {
                return $listPage;
            }

            // Get device IDs from history
            $deviceIds = $this->deviceDataService->getDeviceHistory($request, $deviceHistory);

            // If no devices in history, return the empty page
            if (empty($deviceIds)) {
                return $listPage;
            }

            // Get devices by IDs
            $devices = $this->deviceDataService->getDevicesByIds($deviceIds, $salesChannelContext);

            // Load device content (enrich the device data)
            $this->deviceDataService->loadDeviceContent($devices, $salesChannelContext, 'history');

            // Set the devices to the page
            $listPage->setDevices($devices);

            // Set the page type to history
            $listPage->setPageType('history');

            // Set the page to show devices
            $listPage->setShow(['devices' => true]);

            return $listPage;
        } catch (\Exception $e) {
            $this->logger->error('Error loading popup history page: ' . $e->getMessage());
            throw $e;
        }
    }
}