<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Sitemap;

use Shopware\Core\Content\Sitemap\Provider\AbstractUrlProvider;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\Plugin\Exception\DecorationPatternException;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataConnectorSW6\Core\Content\Series\SeriesCollection;
use Topdata\TopdataConnectorSW6\Core\Content\Series\SeriesEntity;
use Shopware\Core\Content\Sitemap\Struct\Url;
use Shopware\Core\Content\Sitemap\Struct\UrlResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\Content\Sitemap\Service\ConfigHandler;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\NotFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsAnyFilter;
use Shopware\Core\Content\Seo\SeoUrlPlaceholderHandlerInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;

/**
 * Provides URLs for series to be included in the sitemap.
 */
class SeriesUrlProvider extends AbstractUrlProvider
{
    public const CHANGE_FREQ = 'daily';

    public function __construct(
        private readonly EntityRepository $topdataSeriesRepository,
        private readonly SettingsService $settingsService,
        private readonly SeoUrlPlaceholderHandlerInterface $seoUrlPlaceholderHandler,
        private readonly ConfigHandler $configHandler
    ) {
    }

    /**
     * @throws DecorationPatternException
     */
    public function getDecorated(): AbstractUrlProvider
    {
        throw new DecorationPatternException(self::class);
    }

    /**
     * Returns the name of this URL provider.
     */
    public function getName(): string
    {
        return 'device_series';
    }

    /**
     * Retrieves URLs for series to be included in the sitemap.
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @param int $limit The maximum number of URLs to return
     * @param int|null $offset The offset for pagination
     * @return UrlResult The result containing the URLs and the next offset
     */
    public function getUrls(SalesChannelContext $salesChannelContext, int $limit, ?int $offset = null): UrlResult
    {
        if ($this->settingsService->getBool('showSeries')) {
            return new UrlResult([], null);
        }
        $series = $this->_getSeries($salesChannelContext, $limit, $offset);
        $urls   = [];
        $url    = new Url();
        foreach ($series as $serie) {
            /** @var \DateTimeInterface $lastmod */
            $lastmod = $serie->getUpdatedAt() ?: $serie->getCreatedAt();

            $newUrl = clone $url;
            $newUrl->setLoc($this->seoUrlPlaceholderHandler->generate('frontend.top-finder.series', ['code' => $serie->getCode()]));
            $newUrl->setLastmod($lastmod);
            $newUrl->setChangefreq(self::CHANGE_FREQ);
            $newUrl->setResource(SeriesEntity::class);
            $newUrl->setIdentifier($serie->getId());

            $urls[] = $newUrl;
        }

        if (\count($urls) < $limit) { // last run
            $nextOffset = null;
        } elseif ($offset === null) { // first run
            $nextOffset = $limit;
        } else { // 1+n run
            $nextOffset = $offset + $limit;
        }

        return new UrlResult($urls, $nextOffset);
    }

    /**
     * Retrieves a collection of series based on the given criteria.
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @param int $limit The maximum number of series to return
     * @param int|null $offset The offset for pagination
     * @return SeriesCollection The collection of series
     */
    private function _getSeries(SalesChannelContext $salesChannelContext, int $limit, ?int $offset): SeriesCollection
    {
        $criteria = new Criteria();
        $criteria->setLimit($limit);

        if ($offset !== null) {
            $criteria->setOffset($offset);
        }

        $criteria->addFilter(new EqualsFilter('enabled', true));
        $criteria->addSorting(new FieldSorting('wsId'));

        $excludedSeriesIds = $this->_getExcludedSeriesIds($salesChannelContext);
        if (!empty($excludedSeriesIds)) {
            $criteria->addFilter(new NotFilter(NotFilter::CONNECTION_AND, [new EqualsAnyFilter('id', $excludedSeriesIds)]));
        }

        return $this->topdataSeriesRepository->search($criteria, $salesChannelContext->getContext())->getEntities();
    }

    /**
     * Retrieves the IDs of series that should be excluded from the sitemap.
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return array An array of excluded series IDs
     */
    private function _getExcludedSeriesIds(SalesChannelContext $salesChannelContext): array
    {
        $salesChannelId = $salesChannelContext->getSalesChannel()->getId();

        $excludedUrls = $this->configHandler->get(ConfigHandler::EXCLUDED_URLS_KEY);
        if (empty($excludedUrls)) {
            return [];
        }

        $excludedUrls = array_filter($excludedUrls, static function (array $excludedUrl) use ($salesChannelId) {
            if ($excludedUrl['resource'] !== SeriesEntity::class) {
                return false;
            }

            if ($excludedUrl['salesChannelId'] !== $salesChannelId) {
                return false;
            }

            return true;
        });

        return array_column($excludedUrls, 'identifier');
    }
}
