<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder;

use Doctrine\DBAL\Connection;
use Shopware\Core\Content\Category\Exception\CategoryNotFoundException;
use Shopware\Core\Content\Product\SalesChannel\Search\ProductSearchGatewayInterface;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Core\Framework\DataAbstractionLayer\EntityCollection;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataConnectorSW6\Core\Content\Device\Agregate\DeviceCustomer\DeviceCustomerEntity;
use Topdata\TopdataTopFinderProSW6\Constants\CookieKeyNameConstants;
use Topdata\TopdataTopFinderProSW6\Controller\TopFinderController;
use Topdata\TopdataTopFinderProSW6\Service\HelperService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Util\UtilSearchTerm;

/**
 * Class ListPageLoader
 *
 * This class is responsible for loading and managing list pages for the TopFinder functionality.
 * It handles various types of lists including brands, series, and device types.
 */
class ListPageLoader
{
    public const TYPE_BRANDS = 'brands';
    public const TYPE_BRAND  = 'brand';
    public const TYPE_TYPE   = 'type';
    public const TYPE_SERIES = 'series';

    private Request $request;
    private ListPage $listPage;
    private SalesChannelContext $salesChannelContext;

    /**
     * Stores device ids of current user
     */
    private ?array $deviceList = null;

    public function __construct(
        private readonly GenericPageLoader     $genericPageLoader,
        private readonly EntityRepository      $topdataDeviceRepository,
        private readonly EntityRepository      $topdataBrandRepository,
        private readonly EntityRepository      $topdataDeviceTypeRepository,
        private readonly EntityRepository      $topdataSeriesRepository,
        private readonly Connection            $connection,
        private readonly SettingsService       $settingsService,
        private readonly UrlGeneratorInterface $router,
        private readonly Translator            $translator,
        private readonly EntityRepository      $mediaRepository
    )
    {
    }

    /**
     * Load the list page based on the given type
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @param string $type The type of list to load (brands, brand, type, or series)
     * @return ListPage The loaded list page
     *
     * @throws CategoryNotFoundException
     * @throws InconsistentCriteriaIdsException
     */
    public function loadPage(Request $request, SalesChannelContext $salesChannelContext, string $type): ListPage
    {
        $this->listPage = ListPage::createFrom($this->genericPageLoader->load($request, $salesChannelContext));
        $this->listPage->setPageType($type);

        $this->request = $request;
        $this->salesChannelContext = $salesChannelContext;

        switch ($type) {
            case self::TYPE_SERIES:
                $this->_loadSeries();
                break;
            case self::TYPE_TYPE:
                $this->_loadType();
                break;
            case self::TYPE_BRAND:
                $this->_loadBrand($request);
                break;
            case self::TYPE_BRANDS:
                $this->_loadBrands();
                break;
            default:
                /**
                 * @todo throw unknown list
                 */
                break;
        }

        if ($this->settingsService->getInt('searchCompactLimit', true)) {
            $this->listPage->setCompactModeLimit($this->settingsService->getInt('searchCompactLimit'));
        }

        return $this->listPage;
    }

    /**
     * FIXME? is it the same as ControllerUtil::getEntityCollection?
     * Get an entity collection based on given parameters
     *
     * @param EntityRepository $repository The repository to search in
     * @param array $params The search parameters
     * @param array $sortingFields The fields to sort by
     * @param array $associations The associations to load
     * @param bool $deleteNullParams Whether to delete null parameters
     * @return EntityCollection The resulting entity collection
     */
    private function _getEntityCollection(
        EntityRepository $repository,
        array            $params = [],
        array            $sortingFields = [],
        array            $associations = [],
        bool             $deleteNullParams = true
    ): EntityCollection
    {
        $criteria = new Criteria();
        foreach ($params as $key => $value) {
            if (($value === null) && $deleteNullParams) {
                continue;
            }
            $criteria->addFilter(new EqualsFilter($key, $value));
        }
        if ($associations) {
            $criteria->addAssociations($associations);
        }
        foreach ($sortingFields as $sortingField) {
            $criteria->addSorting(new FieldSorting($sortingField, FieldSorting::ASCENDING));
        }

        return $repository->search($criteria, $this->salesChannelContext->getContext())->getEntities();
    }

    /**
     * Load series data and populate the list page with the series information and associated devices.
     *
     * This method retrieves a series entity based on the request parameters, sets the series model to the page,
     * and populates the page with the associated devices. It also sets the SEO metadata and breadcrumbs for the series page.
     *
     * @throws InconsistentCriteriaIdsException If the criteria IDs are inconsistent.
     * @throws CategoryNotFoundException If the series is not found.
     */
    private function _loadSeries(): void
    {
        // Retrieve the series entity based on the request parameters
        $series = $this->_getEntityCollection(
            $this->topdataSeriesRepository,
            [
                'enabled' => true,
                'code'    => $this->request->get('code'),
            ],
            [],
            ['brand']
        )->first();

        // If the series is not found, handle the error (to be implemented)
        if (!$series) {
            /**
             * @todo throw not found series
             */
        }

        // Set the series model to the page
        $this->listPage->model = $series;

        // Set the page to show devices
        $this->listPage->setShow(['devices' => true]);

        // Retrieve and set the devices associated with the series
        $devices = $this->_getDevicesArray($series->getBrand()->getCode(), $series->getId());
        $this->listPage->setDevices($devices);

        // Set the SEO title for the series page
        $this->listPage->setTitle($this->translator->trans('topdata-topfinder.SEO.seriesPageTitle', [
            '%brand%'  => $series->getBrand()->getName(),
            '%series%' => $series->getLabel(),
        ]));

        // Set the SEO meta title for the series page
        $this->listPage->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.seriesMetaTitle', [
            '%brand%'  => $series->getBrand()->getName(),
            '%series%' => $series->getLabel(),
        ]));

        // Set the SEO meta description for the series page
        $this->listPage->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.seriesMetaDescription', [
            '%brand%'  => $series->getBrand()->getName(),
            '%series%' => $series->getLabel(),
        ]));

        // Set the SEO robots meta tag for the series page
        $this->listPage->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.seriesMetaRobots'));

        // Add breadcrumbs for the series page
        $this->listPage->addBreadcrumb($this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'), $this->router->generate('frontend.top-finder.brands'));
        $this->listPage->addBreadcrumb(
            $this->translator->trans('topdata-topfinder.SEO.brandPageTitle', ['%brand%' => $series->getBrand()->getName()]),
            $this->router->generate('frontend.top-finder.brandq', ['code' => $series->getBrand()->getCode()])
        );
        $this->listPage->addBreadcrumb($this->listPage->getTitle(), $this->router->generate('frontend.top-finder.series', ['code' => $series->getCode()]));
    }

    /**
     * Load type data and populate the list page with the type information and associated devices.
     *
     * This method retrieves a type entity based on the request parameters, sets the type model to the page,
     * and populates the page with the associated devices. It also sets the SEO metadata and breadcrumbs for the type page.
     *
     * @throws InconsistentCriteriaIdsException If the criteria IDs are inconsistent.
     * @throws CategoryNotFoundException If the type is not found.
     */
    private function _loadType(): void
    {
        // Retrieve the type entity based on the request parameters
        $type = \Topdata\TopdataTopFinderProSW6\Util\ControllerUtil::getEntityCollection(
            $this->topdataDeviceTypeRepository,
            [
                'enabled' => true,
                'code'    => $this->request->get('code'),
            ],
            [],
            ['brand']
        )->first();

        // If the type is not found, handle the error (to be implemented)
        if (!$type) {
            /**
             * @todo throw not found
             */
        }

        // Set the type model to the page
        $this->listPage->model = $type;

        // Set the page to show devices
        $this->listPage->setShow(['devices' => true]);

        // Retrieve and set the devices associated with the type
        $devices = $this->_getDevicesArray($type->getBrand()->getCode(), 0, $type->getId());
        $this->listPage->setDevices($devices);

        // Set the SEO title for the type page
        $this->listPage->setTitle($this->translator->trans('topdata-topfinder.SEO.typePageTitle', [
            '%brand%' => $type->getBrand()->getName(),
            '%type%'  => $type->getLabel(),
        ]));

        // Set the SEO meta title for the type page
        $this->listPage->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.typeMetaTitle', [
            '%brand%' => $type->getBrand()->getName(),
            '%type%'  => $type->getLabel(),
        ]));

        // Set the SEO meta description for the type page
        $this->listPage->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.typeMetaDescription', [
            '%brand%' => $type->getBrand()->getName(),
            '%type%'  => $type->getLabel(),
        ]));

        // Set the SEO robots meta tag for the type page
        $this->listPage->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.typeMetaRobots'));

        // Add breadcrumbs for the type page
        $this->listPage->addBreadcrumb($this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'), $this->router->generate('frontend.top-finder.brands'));
        $this->listPage->addBreadcrumb(
            $this->translator->trans('topdata-topfinder.SEO.brandPageTitle', ['%brand%' => $type->getBrand()->getName()]),
            $this->router->generate('frontend.top-finder.brandq', ['code' => $type->getBrand()->getCode()])
        );
        $this->listPage->addBreadcrumb($this->listPage->getTitle(), $this->router->generate('frontend.top-finder.type', ['code' => $type->getCode()]));
    }

    /**
     * Load brand data and populate the list page with the brand information and associated devices.
     *
     * This method retrieves a brand entity based on the request parameters, sets the brand model to the page,
     * and populates the page with the associated devices. It also sets the SEO metadata and breadcrumbs for the brand page.
     *
     * @param Request $request The current request
     * @throws InconsistentCriteriaIdsException If the criteria IDs are inconsistent.
     * @throws CategoryNotFoundException If the brand is not found.
     */
    private function _loadBrand(Request $request): void
    {
        // Retrieve the brand entity based on the request parameters
        $brand = \Topdata\TopdataTopFinderProSW6\Util\ControllerUtil::getEntityCollection(
            $this->topdataBrandRepository,
            [
                'enabled' => true,
                'code'    => $this->request->get('code'),
            ]
        )->first();

        // If the brand is not found, handle the error (to be implemented)
        if (!$brand) {
            /**
             * @todo throw not found
             */
        }

        // Set the brand model to the page
        $this->listPage->model = $brand;

        $pageShow = [];

        $pageTitle = $brand->getName();

        // Determine which entities to show based on settings and request cookies
        if ($this->settingsService->getString('selectboxesMode') == 'both') {
            $switch = $request->cookies->get(CookieKeyNameConstants::FINDER_SWITCH);
            if ($switch == 'types') {
                $pageShow['types'] = true;
                $types = \Topdata\TopdataTopFinderProSW6\Util\ControllerUtil::getEntityCollection(
                    $this->topdataDeviceTypeRepository,
                    ['enabled' => true, 'brandId' => $brand->getId()],
                    ['label']
                );
                $this->listPage->setTypes($types);

                $devicesTypeNull = $this->_getDevicesArray($brand->getCode(), 0, null);
                if (count($devicesTypeNull)) {
                    $pageShow['devices'] = true;
                    $this->listPage->setDevices($devicesTypeNull);
                }
            } else {
                $pageShow['series'] = true;
                $series = \Topdata\TopdataTopFinderProSW6\Util\ControllerUtil::getEntityCollection(
                    $this->topdataSeriesRepository,
                    ['enabled' => true, 'brandId' => $brand->getId()],
                    ['label']
                );
                $this->listPage->setSeries($series);

                $devicesSeriesNull = $this->_getDevicesArray($brand->getCode(), null);

                if (count($devicesSeriesNull)) {
                    $pageShow['devices'] = true;
                    $this->listPage->setDevices($devicesSeriesNull);
                }
            }
        } elseif ($this->settingsService->getBool('showSeries')) {
            $pageShow['series'] = true;
            $series = \Topdata\TopdataTopFinderProSW6\Util\ControllerUtil::getEntityCollection(
                $this->topdataSeriesRepository,
                ['enabled' => true, 'brandId' => $brand->getId()],
                ['label']
            );
            $this->listPage->setSeries($series);

            $devicesSeriesNull = $this->_getDevicesArray($brand->getCode(), null);

            if (count($devicesSeriesNull)) {
                $pageShow['devices'] = true;
                $this->listPage->setDevices($devicesSeriesNull);
            }
        } elseif ($this->settingsService->getBool('showTypes')) {
            $pageShow['types'] = true;
            $types = \Topdata\TopdataTopFinderProSW6\Util\ControllerUtil::getEntityCollection(
                $this->topdataDeviceTypeRepository,
                ['enabled' => true, 'brandId' => $brand->getId()],
                ['label']
            );
            $this->listPage->setTypes($types);

            $devicesTypeNull = $this->_getDevicesArray($brand->getCode(), 0, null);
            if (count($devicesTypeNull)) {
                $pageShow['devices'] = true;
                $this->listPage->setDevices($devicesTypeNull);
            }
        }

        if (!$this->settingsService->getBool('showSeries') && !$this->settingsService->getBool('showTypes')) {
            $pageShow['devices'] = true;
            $devices = $this->_getDevicesArray($brand->getCode());
            $this->listPage->setDevices($devices);
        }

        $this->listPage->setShow($pageShow);

        // Set the SEO title for the brand page
        $this->listPage->setTitle($this->translator->trans('topdata-topfinder.SEO.brandPageTitle', [
            '%brand%' => $brand->getName(),
        ]));

        // Set the SEO meta title for the brand page
        $this->listPage->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.brandMetaTitle', [
            '%brand%' => $brand->getName(),
        ]));

        // Set the SEO meta description for the brand page
        $this->listPage->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.brandMetaDescription', [
            '%brand%' => $brand->getName(),
        ]));

        // Set the SEO robots meta tag for the brand page
        $this->listPage->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.brandMetaRobots'));

        // Add breadcrumbs for the brand page
        $this->listPage->addBreadcrumb($this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'), $this->router->generate('frontend.top-finder.brands'));
        $this->listPage->addBreadcrumb($this->listPage->getTitle(), $this->router->generate('frontend.top-finder.brandq', ['code' => $brand->getCode()]));
    }

    /**
     * Load brand data and populate the list page with the brand information.
     *
     * This method retrieves all enabled brands from the database, sets the brand data to the page,
     * and configures the SEO metadata and breadcrumbs for the brands page.
     */
    private function _loadBrands(): void
    {
        // Set the page to show brands
        $this->listPage->setShow(['brands' => true]);

        // Retrieve all enabled brands from the database
        $brands = $this->connection->fetchAllAssociative(
            'SELECT code, label as name, sort FROM `topdata_brand`'
            . ' WHERE is_enabled = 1'
            . ' ORDER BY sort DESC, name ASC'
        );

        // Set the retrieved brands to the page
        $this->listPage->setBrands($brands);

        // Set the SEO title for the brands page
        $this->listPage->setTitle($this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'));

        // Set the SEO meta title for the brands page
        $this->listPage->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.brandsMetaTitle'));

        // Set the SEO meta description for the brands page
        $this->listPage->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.brandsMetaDescription'));

        // Set the SEO robots meta tag for the brands page
        $this->listPage->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.brandsMetaRobots'));

        // Add breadcrumbs for the brands page
        $this->listPage->addBreadcrumb(
            $this->listPage->getTitle(),
            $this->router->generate('frontend.top-finder.brands')
        );
    }

    private function _getDevices(array $params, bool $deleteNullParams = true): EntityCollection
    {
        $devices = \Topdata\TopdataTopFinderProSW6\Util\ControllerUtil::getEntityCollection(
            $this->topdataDeviceRepository,
            array_merge(['enabled' => true], $params),
            ['brand.name', 'series.label', 'model'],
            ['media', 'brand', 'type', 'series'],
            $deleteNullParams
        );

        foreach ($devices as $device) {
            $device->setInDeviceList(in_array($device->getId(), $this->_getDeviceList()));
        }

        return $devices;
    }

    private function _getDevicesArray($brandCode, $seriesId = 0, $typeId = 0)
    {
        $conditions = [];
        $conditions[] = '(topdata_device.is_enabled=1)';
        $conditions[] = '(topdata_brand.code = "' . $brandCode . '")';

        if ($seriesId === null) {
            $conditions[] = '(topdata_device.series_id IS NULL)';
        } elseif (Uuid::isValid((string)$seriesId)) {
            $conditions[] = '(topdata_device.series_id = 0x' . $seriesId . ')';
        }

        if ($typeId === null) {
            $conditions[] = '(topdata_device.type_id IS NULL)';
        } elseif (Uuid::isValid((string)$typeId)) {
            $conditions[] = '(topdata_device.type_id = 0x' . $typeId . ')';
        }

        $devices = $this->connection->fetchAllAssociative('
SELECT topdata_device.model as name, 
       LOWER(HEX(topdata_device.id)) as id, 
       LOWER(HEX(topdata_device.brand_id)) as brand_id, 
       LOWER(HEX(topdata_device.series_id)) as series_id, 
       LOWER(HEX(topdata_device.type_id)) as type_id, 
       topdata_device.code as code, 
       topdata_brand.label as brand_name,
       topdata_brand.code as brand_code,
       topdata_series.code as series_code,
       LOWER(HEX(topdata_device.media_id)) as media_id,
       topdata_series.label as series_name,
       topdata_device_type.label as type_name
  FROM topdata_device 
  LEFT JOIN topdata_brand ON (topdata_brand.id=topdata_device.brand_id)
  LEFT JOIN topdata_series ON (topdata_series.id=topdata_device.series_id)
  LEFT JOIN topdata_device_type ON (topdata_device_type.id=topdata_device.type_id)
  WHERE ' . implode('AND', $conditions) . '
  ORDER BY topdata_device.code
            ');

        return $devices;
    }

    public function loadJsonBrands(Request $request, SalesChannelContext $salesChannelContext): ListPage
    {
        $this->salesChannelContext = $salesChannelContext;
        $page = $this->genericPageLoader->load($request, $salesChannelContext);
        $this->listPage = ListPage::createFrom($page);
        $this->listPage->setPageType(self::TYPE_BRANDS);
        $this->request = $request;

        $this->listPage->setShow(['brands' => true]);

        $brands = $this->connection->fetchAllAssociative(
            'SELECT code, label as name, sort FROM `topdata_brand`'
            . ' WHERE is_enabled = 1'
            . ' ORDER BY sort DESC, name ASC'
        );

        $this->listPage->setBrands($brands);

        $this->listPage->setTitle($this->translator->trans('topdata-topfinder.popup.allBrands'));

        $this->listPage->popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.allBrands'),
//            'path' => $this->router->generate('frontend.top-finder-api.popup-all-brands')
        ];

        return $this->listPage;
    }

    public function loadBrandDevicesJson(Request $request, SalesChannelContext $salesChannelContext): ListPage
    {
        $this->salesChannelContext = $salesChannelContext;
        $page = $this->genericPageLoader->load($request, $salesChannelContext);
        $this->listPage = ListPage::createFrom($page);
        $this->request = $request;

        $brand = \Topdata\TopdataTopFinderProSW6\Util\ControllerUtil::getEntityCollection(
            $this->topdataBrandRepository,
            [
                'enabled' => true,
                'code'    => $this->request->get('code'),
            ]
        )->first();

        if (!$brand) {
            return $this->listPage;
        }

        $this->listPage->setTitle($this->translator->trans('topdata-topfinder.SEO.brandPageTitle', [
            '%brand%' => '<b>' . $brand->getName() . '</b>',
        ]));

        $devices = $this->_getDevicesArray($brand->getCode());
        $this->_loadDevicesContent($devices, $salesChannelContext, 'brand');
        $this->listPage->setDevices($devices);

        return $this->listPage;
    }

    public function loadSerieDevicesJson(Request $request, SalesChannelContext $salesChannelContext): ListPage
    {
        $this->salesChannelContext = $salesChannelContext;
        $page = $this->genericPageLoader->load($request, $salesChannelContext);
        $this->listPage = ListPage::createFrom($page);
        $this->request = $request;

        $series = \Topdata\TopdataTopFinderProSW6\Util\ControllerUtil::getEntityCollection(
            $this->topdataSeriesRepository,
            [
                'enabled' => true,
                'code'    => $this->request->get('code'),
            ],
            [],
            ['brand']
        )->first();

        if (!$series) {
            return $this->listPage;
        }

        $this->listPage->setTitle($this->translator->trans('topdata-topfinder.SEO.seriesPageTitle', [
            '%brand%'  => '<b>' . $series->getBrand()->getName() . '</b>',
            '%series%' => $series->getLabel(),
        ]));

        $devices = $this->_getDevicesArray($series->getBrand()->getCode(), $series->getId());
        $this->_loadDevicesContent($devices, $salesChannelContext, 'series');
        $this->listPage->setDevices($devices);

        $letter = ($series->getBrand()->getCode())[0];
        if (preg_match('/^[0-9]{1}$/', $letter)) {
            $letter = '0';
            $letterStr = '0-9';
        } else {
            $letterStr = strtoupper($letter);
        }

        $popupPath = [];

        $popupPath[] = [
            'name' => $letterStr,
            'path' => $this->router->generate('frontend.top-finder-api.popup-letter', ['letter' => $letter]),
        ];

        $popupPath[] = [
            'name' => $series->getBrand()->getName(),
            'path' => $this->router->generate('frontend.top-finder.popup_brand', ['brandCode' => $series->getBrand()->getCode()]),
        ];

        $popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.brandSeries'),
            'path' => $this->router->generate('frontend.top-finder.popup_brand_series', ['brandCode' => $series->getBrand()->getCode()]),
        ];

        $popupPath[] = [
            'name' => $series->getLabel(),
        ];

        $this->listPage->popupPath = $popupPath;

        return $this->listPage;
    }

    public function loadTypeDevicesJson(Request $request, SalesChannelContext $salesChannelContext): ListPage
    {
        $this->salesChannelContext = $salesChannelContext;
        $page = $this->genericPageLoader->load($request, $salesChannelContext);
        $this->listPage = ListPage::createFrom($page);
        $this->request = $request;

        $type = \Topdata\TopdataTopFinderProSW6\Util\ControllerUtil::getEntityCollection(
            $this->topdataDeviceTypeRepository,
            [
                'enabled' => true,
                'code'    => $this->request->get('code'),
            ],
            [],
            ['brand']
        )->first();

        if (!$type) {
            return $this->listPage;
        }

        $this->listPage->setTitle($this->translator->trans('topdata-topfinder.SEO.typePageTitle', [
            '%brand%' => '<b>' . $type->getBrand()->getName() . '</b>',
            '%type%'  => $type->getLabel(),
        ]));

        $devices = $this->_getDevicesArray($type->getBrand()->getCode(), 0, $type->getId());
        $this->_loadDevicesContent($devices, $salesChannelContext, 'types');
        $this->listPage->setDevices($devices);

        $letter = ($type->getBrand()->getCode())[0];
        if (preg_match('/^[0-9]{1}$/', $letter)) {
            $letter = '0';
            $letterStr = '0-9';
        } else {
            $letterStr = strtoupper($letter);
        }

        $popupPath = [];

        $popupPath[] = [
            'name' => $letterStr,
            'path' => $this->router->generate('frontend.top-finder-api.popup-letter', ['letter' => $letter]),
        ];

        $popupPath[] = [
            'name' => $type->getBrand()->getName(),
            'path' => $this->router->generate('frontend.top-finder.popup_brand', ['brandCode' => $type->getBrand()->getCode()]),
        ];

        $popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.brandTypes'),
            'path' => $this->router->generate('frontend.top-finder.popup_brand_types', ['brandCode' => $type->getBrand()->getCode()]),
        ];

        $popupPath[] = [
            'name' => $type->getLabel(),
        ];

        $this->listPage->popupPath = $popupPath;

        return $this->listPage;
    }

    /**
     * Get the list of devices for the current user
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return array The list of devices
     */
    private function _getDeviceList(SalesChannelContext $salesChannelContext): array
    {
        if (!$salesChannelContext->getCustomer() || $salesChannelContext->getCustomer()->getGuest()) {
            $this->deviceList = [];
        } elseif (null === $this->deviceList) {
            $this->deviceList = [];

            $rez = $this->connection
                ->createQueryBuilder()
                ->select('LOWER(HEX(device_id)) as device_id, extra_info')
                ->from('topdata_device_to_customer')
                ->where('customer_id = 0x' . $salesChannelContext->getCustomer()->getId())
                //')AND(is_dealer_managed=0)'
                ->execute()
                ->fetchAllAssociative();
            foreach ($rez as $val) {
                $this->deviceList[$val['device_id']] = $val['extra_info'] ? json_decode($val['extra_info'], true) : DeviceCustomerEntity::defaultExtraInfo();
            }
        }

        return $this->deviceList;
    }

    private function _loadDevicesContent(&$devices, SalesChannelContext $salesChannelContext, string $listType = 'brand', string $searchTerm = '')
    {
        $mediaIds = [];
        $devicelist = $this->_getDeviceList($salesChannelContext);
        foreach ($devices as $key => $device) {
            if ($listType == 'search' && $searchTerm) {
                $devices[$key]['path'] = $this->router->generate('frontend.top-finder-api.popup-device-new', [
                    'deviceCode' => $device['code'],
                    'listType'   => $listType,
                    'term'       => $searchTerm,
                ]);
            } else {
                $devices[$key]['path'] = $this->router->generate('frontend.top-finder-api.popup-device-new', [
                    'deviceCode' => $device['code'],
                    'listType'   => $listType,
                ]);
            }
            if ($device['media_id']) {
                $mediaIds[] = $device['media_id'];
            }

            if (isset($devicelist[$device['id']]) && isset($devicelist[$device['id']]['devices'])) {
                $devices[$key]['devicelist'] = count($devicelist[$device['id']]['devices']);
            } else {
                $devices[$key]['devicelist'] = -1;
            }
        }

        if (count($mediaIds)) {
            $medias = $this->mediaRepository->search(new Criteria($mediaIds), $salesChannelContext->getContext());

            foreach ($devices as $key => $device) {
                if ($device['media_id'] && $medias->get($device['media_id'])) {
                    $devices[$key]['media_url'] = $medias->get($device['media_id'])->getUrl();
                }
            }
        }
    }

    public function loadHistoryDevicesJson(Request $request, SalesChannelContext $salesChannelContext): ListPage
    {
        $this->salesChannelContext = $salesChannelContext;
        $page = $this->genericPageLoader->load($request, $salesChannelContext);
        $this->listPage = ListPage::createFrom($page);
        $this->request = $request;

        $this->listPage->setTitle($this->translator->trans('topdata-topfinder.popup.deviceHistoryTitle'));

        $popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.deviceHistoryTitle'),
        ];

        $this->listPage->popupPath = $popupPath;

        if (!$this->settingsService->getBool('showDeviceHistory')) {
            return $this->listPage;
        }

        $cokie = $request->cookies->get(CookieKeyNameConstants::DEVICE_HISTORY);
        $cookieIds = [];
        $deviceIds = [];
        if ($cokie) {
            $cookieIds = explode(',', $cokie);
        }

        foreach ($cookieIds as $value) {
            if (Uuid::isValid($value)) {
                $deviceIds[] = $value;
            }
        }

        if (!$deviceIds) {
            return $this->listPage;
        }

        $devices = $this->connection->fetchAllAssociative('
SELECT topdata_device.model as name, 
       LOWER(HEX(topdata_device.id)) as id, 
       LOWER(HEX(topdata_device.brand_id)) as brand_id, 
       LOWER(HEX(topdata_device.series_id)) as series_id, 
       LOWER(HEX(topdata_device.type_id)) as type_id, 
       topdata_device.code as code, 
       topdata_brand.label as brand_name,
       topdata_brand.code as brand_code,
       topdata_series.code as series_code,
       LOWER(HEX(topdata_device.media_id)) as media_id,
       topdata_series.label as series_name,
       topdata_device_type.label as type_name
  FROM topdata_device 
  LEFT JOIN topdata_brand ON (topdata_brand.id=topdata_device.brand_id)
  LEFT JOIN topdata_series ON (topdata_series.id=topdata_device.series_id)
  LEFT JOIN topdata_device_type ON (topdata_device_type.id=topdata_device.type_id)
  WHERE topdata_device.id IN (0x' . implode(',0x', $deviceIds) . ')
            ');
        $deviceIds = array_reverse($deviceIds);

        $sortedDevices = [];
        foreach ($deviceIds as $id) {
            foreach ($devices as $device) {
                if ($id === $device['id']) {
                    $sortedDevices[] = $device;
                }
            }
        }

        $this->_loadDevicesContent($sortedDevices, $salesChannelContext, 'history');
        $this->listPage->setDevices($sortedDevices);

        return $this->listPage;
    }

    public function loadSearchDevicesJson(Request $request, SalesChannelContext $salesChannelContext): ListPage
    {
        $defaultOffset = TopFinderController::POPUP_DEVICES_PRELOAD_LENGTH;

        $this->salesChannelContext = $salesChannelContext;
        $page = $this->genericPageLoader->load($request, $salesChannelContext);
        $this->listPage = ListPage::createFrom($page);
        $this->request = $request;

        $offset = 1 * $request->query->get('offset');

        $term = $request->query->get('term');

        $term = UtilSearchTerm::filterTerm($term);

        $this->listPage->setTitle($this->translator->trans('topdata-topfinder.popup.deviceSearchTitle', ['%term%' => $term]));
        $this->listPage->searchTerm = $term;

        $popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.deviceSearchTitle', ['%term%' => $term]),
        ];

        $this->listPage->popupPath = $popupPath;

        if (!$term) {
            return $this->listPage;
        }

        $this->listPage->devicesTotal = HelperService::countFindDevices($term, $this->connection);

        if ($this->listPage->devicesTotal) {
            if ($offset == $defaultOffset) {
                $devices = HelperService::findDevices($term, $this->connection);
                $devices = array_slice($devices, $offset);
            } else {
                $devices = HelperService::findDevices($term, $this->connection, $defaultOffset);
            }
        }

        $this->_loadDevicesContent($devices, $salesChannelContext, 'search', $term);
        $this->listPage->setDevices($devices);

        return $this->listPage;
    }
}
