<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\Page\Device;

use Doctrine\DBAL\Connection;
use Shopware\Core\Content\Product\Aggregate\ProductVisibility\ProductVisibilityDefinition;
use Shopware\Core\Content\Product\Events\ProductListingCriteriaEvent;
use Shopware\Core\Content\Product\Events\ProductListingResultEvent;
use Shopware\Core\Content\Product\SalesChannel\Listing\ProductListingResult;
use Shopware\Core\Content\Product\SalesChannel\ProductAvailableFilter;
use Shopware\Core\Content\Product\SalesChannel\Sorting\ProductSortingCollection;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Struct\ArrayStruct;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataConnectorSW6\Core\Content\Device\DeviceEntity;
use Topdata\TopdataConnectorSW6\Service\EntitiesHelperService;
use Topdata\TopdataTopFinderProSW6\Content\Product\SalesChannel\DeviceListing\ProductDeviceListingLoader;
use Topdata\TopdataTopFinderProSW6\Service\ProductSortingHelperService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;

/**
 * Class DevicePageLoader
 *
 * This class is responsible for loading and preparing the device page data.
 * It handles the loading of device information, product listings, and other
 * related data for displaying a device page in the storefront.
 */
class DevicePageLoader
{
    public const PRODUCTS_PER_PAGE = 10; // TODO: this should be a setting for the topfinder
    public const REQUEST_KEY_PAGE  = 'p'; // ?p=2 for page 2
    const        DEFAULT_SORT_KEY  = 'topseller';


    public function __construct(
        private readonly GenericPageLoader           $genericPageLoader,
        private readonly EntityRepository            $topdataDeviceRepository,
        private readonly ProductDeviceListingLoader  $productDeviceListingLoader,
        private readonly EventDispatcherInterface    $eventDispatcher,
        private readonly EntitiesHelperService       $entitiesHelperService,
        private readonly Translator                  $translator,
        private readonly UrlGeneratorInterface       $router,
        private readonly SettingsService             $settingsService,
        private readonly Connection                  $connection,
        private readonly EntityRepository            $propertyGroupOptionRepository,
        private readonly ProductSortingHelperService $productSortingHelperService,
    )
    {
    }

    /**
     * Get the child product IDs for the given product IDs.
     *
     * @param array $productIds An array of parent product IDs
     * @return array An array of child product IDs
     */
    private function _getProductsChildIds(array $productIds): array
    {
        $ids = [];
        if (!count($productIds)) {
            return $ids;
        }

        $productIds = '0x' . implode(',0x', $productIds);

        $rez = $this->connection->fetchAllAssociative("
             SELECT LOWER(HEX(id)) as id
              FROM `product`
              WHERE parent_id IN ($productIds)
        ");

        foreach ($rez as $item) {
            $ids[] = $item['id'];
        }

        return $ids;
    }


    /**
     * ==== MAIN ====
     *
     * Load the device page data.
     *
     *
     * This method is responsible for loading the device page data, including device information,
     * product listings, and other related data for displaying a device page in the storefront.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @param bool $partialLoad Whether to perform a partial load -- ignored
     * @return DevicePage The loaded device page
     * @throws NotFoundHttpException If the device is not found
     */
    public function loadDevicePageData(Request $request, SalesChannelContext $salesChannelContext, bool $partialLoad = false): DevicePage
    {
        $devicePage = DevicePage::createFrom($this->genericPageLoader->load($request, $salesChannelContext));

        $device = $this->_loadDevice($request->get('deviceCode'), $salesChannelContext);
        $devicePage->setDevice($device);

        if (!$device) {
            // this could happen eg if a device is disabled
            throw new NotFoundHttpException('Device not found: ' . $request->get('deviceCode'));
        }

        $this->_loadTabs($devicePage, $request, $salesChannelContext);

        $this->_setActiveTab($devicePage, $request, $salesChannelContext);

        if ($request->isXmlHttpRequest() === false) {
            $this->_loadDeviceSynonyms($devicePage, $salesChannelContext);
        }

        $productIds = $device->getProducts()->getIds();
        $productWithChildsIds = [];
        foreach ($device->getProducts() as $prod) {
            if ($prod->getParentId()) {
                $productIds[] = $prod->getParentId();
            } elseif ($prod->getChildCount()) {
                $productWithChildsIds[] = $prod->getId();
            }
        }

        $productIds = array_merge(
            $productIds,
            $this->_getProductsChildIds($productWithChildsIds)
        );

        $productIds = array_unique($productIds);
        if (empty($productIds)) {
            // zero products found, but we need a criteria object
            $criteria = new Criteria();
            $criteria->addFilter(new EqualsFilter('id', '00000000000000000000000000000000'));
        } else {
            $criteria = new Criteria($productIds);
        }

        $criteria->setTitle('cms::product-listing');
        $this->eventDispatcher->dispatch(
            new ProductListingCriteriaEvent($request, $criteria, $salesChannelContext)
        );

        $criteria->addFilter(
            new ProductAvailableFilter($salesChannelContext->getSalesChannel()->getId(), ProductVisibilityDefinition::VISIBILITY_ALL)
        );

        if ($devicePage->activeTabId) {
            $criteria->addFilter(new EqualsFilter('properties.id', $devicePage->activeTabId));
        }

        // ---- pagination support (10/2024 added as it was missing somehow)
        // $criteria->setLimit($request->get('limit', self::PRODUCTS_PER_PAGE));
        $criteria->setLimit(self::PRODUCTS_PER_PAGE);
        $criteria->setOffset(($request->get(self::REQUEST_KEY_PAGE, 1) - 1) * $criteria->getLimit());
        $criteria->setTotalCountMode(Criteria::TOTAL_COUNT_MODE_EXACT);
        $sortKey = $request->get('order', self::DEFAULT_SORT_KEY);
        $this->productSortingHelperService->addSortingFromUrlKey($criteria, $sortKey);

        // NOTE: page.listing is used as "searchResult" in the twig template
        $devicePage->listing = ProductListingResult::createFrom(
            $this->productDeviceListingLoader->load($criteria, $salesChannelContext)
        );

        // ---- add availableSortings to search result (10/2024 added as it was missing somehow)
        $availableSortings = $this->productSortingHelperService->getAvailableProductSortings($salesChannelContext->getContext());
        $devicePage->listing->setAvailableSortings($availableSortings);
        $devicePage->listing->setSorting($sortKey); // 11/2024 this was also missing in the original code

        $this->eventDispatcher->dispatch(
            new ProductListingResultEvent($request, $devicePage->listing, $salesChannelContext)
        );

        if ($request->isXmlHttpRequest() === false) {
            $this->_setSeoInformation($devicePage);
        }

        return $devicePage;
    }


    /**
     * 06/2024 changed signature: expecting $deviceCode instead of $request
     */
    /**
     * Load a device entity by its code.
     *
     * @param string|null $deviceCode The device code
     * @param SalesChannelContext $context The sales channel context
     * @return DeviceEntity|null The loaded device entity or null if not found
     */
    private function _loadDevice(?string $deviceCode, SalesChannelContext $context): ?DeviceEntity
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('enabled', true));
        $criteria->addFilter(new EqualsFilter('code', $deviceCode));
        $criteria->addAssociations(['media', 'products', 'brand', 'type', 'series']);

        return $this->topdataDeviceRepository->search($criteria, $context->getContext())->getEntities()->first();
    }

    /**
     * Set the active tab for the device page as in request parameter `propertyGroupOptionId`
     *
     * @param DevicePage $page The device page
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     */
    private function _setActiveTab(DevicePage $page, Request $request, SalesChannelContext $context): void
    {
        //        if($page->tabs && $page->tabs->first()) {
        //            $page->activeTabId = $page->tabs->first()->getId();
        //        }
        //        else {
        //            return;
        //        }

        $propertyGroupOptionId = $request->get('propertyGroupOptionId', '');
        if (!Uuid::isValid($propertyGroupOptionId)) {
            return;
        }

        if ($page->tabs->get($propertyGroupOptionId)) {
            $page->activeTabId = $propertyGroupOptionId;
        }
    }

    /**
     * Load the tabs for the device page.
     *
     * @param DevicePage $devicePage The device page
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     */
    private function _loadTabs(DevicePage $devicePage, Request $request, SalesChannelContext $context): void
    {
        if (!$devicePage->getDevice()) {
            return;
        }
        $deviceId = $devicePage->getDeviceId();
        $propertyGroupId = $this->settingsService->getString('groupingTabPropery');

        if (!Uuid::isValid($propertyGroupId)) {
            return;
        }

        $rez = $this->connection->fetchAllAssociative("
 SELECT LOWER(HEX(pgo.id)) as pgo_id, 
        COUNT(pp.product_id) as cnt
  FROM 
  `product_property` as pp,
  `property_group_option` as pgo,
  `topdata_device_to_product` as tdp
  WHERE pp.property_group_option_id=pgo.id 
        AND tdp.product_id = pp.product_id 
	AND tdp.device_id = 0x$deviceId
	AND pgo.property_group_id = 0x$propertyGroupId
  GROUP BY pgo.id
            ");

        $devicePage->allProductsCount = 0;
        $propertyGroupOptionIds = [];
        foreach ($rez as $row) {
            $propertyGroupOptionIds[$row['pgo_id']] = $row['cnt'];
            $devicePage->allProductsCount += $row['cnt'];
        }

        if (!count($propertyGroupOptionIds)) {
            return;
        }

        $criteria = new Criteria(array_keys($propertyGroupOptionIds));
        $criteria->addSorting(new FieldSorting('name'));
        $devicePage->tabs = $this->propertyGroupOptionRepository->search(
            $criteria,
            $context->getContext()
        )->getEntities();
        foreach ($devicePage->tabs as $entity) {
            $entity->addExtension('productCount', new ArrayStruct([$propertyGroupOptionIds[$entity->getId()]]));
        }
    }

    /**
     * Load device synonyms for the device page.
     *
     * @param DevicePage $page The device page
     * @param SalesChannelContext $context The sales channel context
     */
    private function _loadDeviceSynonyms(DevicePage $page, SalesChannelContext $context): void
    {
        $deviceSynonymIds = $this->entitiesHelperService->getDeviceSynonymsIds($page->getDeviceId());
        if ($deviceSynonymIds) {
            // load synonyms
            $synonyms = $this->topdataDeviceRepository->search(
                (new Criteria($deviceSynonymIds))
                    ->addFilter(new EqualsFilter('enabled', true))
                    ->addAssociations(['brand', 'type', 'series']),
                $context->getContext()
            )->getEntities();

            if (count($synonyms)) {
                $page->addExtension('deviceSynonyms', $synonyms);
            }
        }
    }

    /**
     * Set SEO information for the device page.
     *
     * @param DevicePage $page The device page
     */
    private function _setSeoInformation(DevicePage $page): void
    {
        $device = $page->getDevice();

        $page->setPageTitle($this->translator->trans('topdata-topfinder.SEO.devicePageTitle', [
            '%brand%'  => $device->getBrand()->getName(),
            '%device%' => $device->getModel(),
        ]));

        $page->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.deviceMetaTitle', [
            '%brand%'  => $device->getBrand()->getName(),
            '%device%' => $device->getModel(),
        ]));

        $page->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.deviceMetaDescription', [
            '%brand%'  => $device->getBrand()->getName(),
            '%device%' => $device->getModel(),
        ]));

        if ($page->activeTabId) {
            $page->getMetaInformation()->setRobots('noindex,follow');
        } else {
            $page->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.deviceMetaRobots'));
        }

        $page->addBreadcrumb(
            $this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'),
            $this->router->generate('frontend.top-finder.brands')
        );

        $page->addBreadcrumb(
            $this->translator->trans('topdata-topfinder.SEO.brandPageTitle', ['%brand%' => $device->getBrand()->getName()]),
            $this->router->generate('frontend.top-finder.brand', ['brandCode' => $device->getBrand()->getCode()])
        );

        $page->addBreadcrumb(
            $page->getPageTitle(),
            $this->router->generate('frontend.top-finder.device', ['deviceCode' => $device->getCode()])
        );
    }
}
