<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder;

use Doctrine\DBAL\Connection;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\ContainsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;

/**
 * Class DeviceSearchPageLoader
 *
 * This class is responsible for loading the device search page.
 */
class DeviceSearchPageLoader
{
    public const DEVICES_LIMIT = 500;

    private ?array $deviceList = null;

    /**
     * DeviceSearchPageLoader constructor.
     *
     * @param GenericPageLoader $genericPageLoader
     * @param EntityRepository $topdataDeviceRepository
     * @param Connection $connection
     * @param SettingsService $settingsService
     */
    public function __construct(
        private readonly GenericPageLoader $genericPageLoader,
        private readonly EntityRepository  $topdataDeviceRepository,
        private readonly Connection        $connection,
        private readonly SettingsService   $settingsService,
    )
    {
    }

    /**
     * Loads the device search page.
     *
     * @param Request $request
     * @param SalesChannelContext $salesChannelContext
     * @return DeviceSearchPage
     */
    public function loadPage(Request $request, SalesChannelContext $salesChannelContext): DeviceSearchPage
    {
        $page = DeviceSearchPage::createFrom($this->genericPageLoader->load($request, $salesChannelContext));
        $page->setDevices($this->searchDevices($request, $salesChannelContext));
        $page->setDeviceSearchTerm($request->query->get('term'));

        //        $this->eventDispatcher->dispatch(
        //            new DeviceListPageLoadedEvent($page, $salesChannelContext, $request)
        //        );

        if ($this->settingsService->getInt('searchCompactLimit', true)) {
            $page->setCompactModeLimit($this->settingsService->getInt('searchCompactLimit'));
        }

        $page->offset = (int)$request->query->get('offset');
        $page->limit = self::DEVICES_LIMIT;
        $page->deviceListDevices = $this->getDeviceList($salesChannelContext);

        return $page;
    }

    /**
     * Searches for devices based on the request parameters.
     *
     * @param Request $request
     * @param SalesChannelContext $context
     * @return EntitySearchResult
     * @throws \Symfony\Component\HttpKernel\Exception\BadRequestHttpException
     */
    private function searchDevices($request, SalesChannelContext $context): EntitySearchResult
    {
        $term = $request->query->get('term');
        $offset = (int)$request->query->get('offset');

        if (empty($term)) {
            throw new \Symfony\Component\HttpKernel\Exception\BadRequestHttpException('term');
        }

        $search = mb_strtolower(trim($term));

        if (empty($search)) {
            throw new \Symfony\Component\HttpKernel\Exception\BadRequestHttpException('term');
        }

        $criteria = (new Criteria())
            ->addFilter(new EqualsFilter('enabled', true))
            ->addFilter(new ContainsFilter('keywords', $search))
            ->addAssociations(['media', 'brand', 'type', 'series'])
            ->addSorting(new FieldSorting('brand.name', FieldSorting::ASCENDING))
            ->addSorting(new FieldSorting('series.label', FieldSorting::ASCENDING))
            ->addSorting(new FieldSorting('model', FieldSorting::ASCENDING));

        $criteria->setTotalCountMode(Criteria::TOTAL_COUNT_MODE_EXACT);

        $criteria->setLimit(self::DEVICES_LIMIT);
        if ($offset) {
            $criteria->setOffset($offset);
        }

        $devices = $this->topdataDeviceRepository->search($criteria, $context->getContext());

        foreach ($devices as $device) {
            $device->setInDeviceList(in_array($device->getId(), $this->getDeviceList($context)));
        }

        return $devices;
    }

    /**
     * Retrieves the list of devices associated with the current customer.
     *
     * @param SalesChannelContext $context
     * @return array
     */
    private function getDeviceList(SalesChannelContext $context): array
    {
        if (null === $this->deviceList) {
            $this->deviceList = [];
            if ($context->getCustomer() && !($context->getCustomer()->getGuest())) {
                $connection = $this->connection;
                $rez = $connection->createQueryBuilder()
                    ->select(['device_id'])
                    ->from('topdata_device_to_customer')
                    ->where('customer_id = 0x' . $context->getCustomer()->getId())
                    ->execute()
                    ->fetchAllAssociative();
                foreach ($rez as $val) {
                    $this->deviceList[] = bin2hex($val['device_id']);
                }
            }
        }

        return $this->deviceList;
    }
}