<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\Page\PopupDeviceInfoWidgetPage;

use Doctrine\DBAL\Connection;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Topdata\TopdataConnectorSW6\Core\Content\Device\Agregate\DeviceCustomer\DeviceCustomerEntity;

/**
 * Class PopupDeviceInfoWidgetPageLoader
 *
 * Loads the popup device info widget page with device information.
 *
 * @package Topdata\TopdataTopFinderProSW6\Storefront\Page\PopupDeviceInfoWidgetPage
 */
class PopupDeviceInfoWidgetPageLoader
{
    /**
     * Constructor for PopupDeviceInfoWidgetPageLoader.
     *
     * @param GenericPageLoader $genericLoader The generic page loader.
     * @param EntityRepository $topdataDeviceRepository The repository for topdata devices.
     * @param Connection $connection The database connection.
     */
    public function __construct(
        private readonly GenericPageLoader $genericLoader,
        private readonly EntityRepository  $topdataDeviceRepository,
        private readonly Connection        $connection,
    )
    {
    }

    /**
     * ==== MAIN ====
     *
     * Loads the popup device info widget page.
     *
     * @param Request $request The HTTP request.
     * @param SalesChannelContext $salesChannelContext The sales channel context.
     * @param string $deviceId The ID of the device.
     * @param string $userId The ID of the user.
     * @return PopupDeviceInfoWidgetPage The loaded popup device info widget page.
     */
    public function load(
        Request             $request,
        SalesChannelContext $salesChannelContext,
        string              $deviceId,
        string              $userId
    ): PopupDeviceInfoWidgetPage
    {
        // Create the page from the generic loader
        $popupDeviceInfoWidgetPage = PopupDeviceInfoWidgetPage::createFrom(
            $this->genericLoader->load($request, $salesChannelContext)
        );

        // Fetch device information from the database
        $deviceInfoJson = $this->connection->createQueryBuilder()
            ->select('extra_info')
            ->from('topdata_device_to_customer')
            ->where('(device_id = 0x' . $deviceId . ') AND (customer_id = 0x' . $userId . ')')
            ->execute()
            ->fetchOne();
        $popupDeviceInfoWidgetPage->setDeviceInfo($deviceInfoJson ? json_decode($deviceInfoJson, true) : DeviceCustomerEntity::defaultExtraInfo());

        // Create criteria to search for the device
        $criteria = (new Criteria([$deviceId]))
            ->addFilter(new EqualsFilter('enabled', true))
            ->addAssociations(['media', 'brand', 'series', 'type']);

        // Fetch the device from the repository
        $popupDeviceInfoWidgetPage->setDevice($this->topdataDeviceRepository->search($criteria, $salesChannelContext->getContext())->getEntities()->first());

        return $popupDeviceInfoWidgetPage;
    }
}