<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Controller;

use Doctrine\DBAL\Connection;
use Shopware\Core\Framework\Context;
use Shopware\Storefront\Controller\StorefrontController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\HttpFoundation\JsonResponse;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Validation\DataBag\RequestDataBag;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Topdata\TopdataTopFinderProSW6\Constants\CookieKeyNameConstants;
use Topdata\TopdataTopFinderProSW6\Service\DeviceToCustomerService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\TopfinderPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\DeviceListPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\DeviceSuggestPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\DeviceSearchPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\TopfinderPage;
use Shopware\Core\Content\Product\SalesChannel\Listing\ProductListingResult;
use Topdata\TopdataConnectorSW6\Core\Content\Device\DeviceEntity;
use Topdata\TopdataTopFinderProSW6\Component\Collection;
use Shopware\Core\Framework\DataAbstractionLayer\EntityCollection;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\ListPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\CompatibleDevicesWidget\CompatibleDevicesWidgetLoader;
use Topdata\TopdataConnectorSW6\Service\EntitiesHelperService;
use Symfony\Component\HttpFoundation\Cookie;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\PopupFormWidget\PopupFormWidgetLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\PopupDeviceWidget\PopupDeviceWidgetLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\BrandLetter\BrandLetterPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\OneBrandLetter\OneBrandLetterPageLoader;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Topdata\TopdataTopFinderProSW6\Service\HelperService;
use Topdata\TopdataConnectorSW6\Core\Content\Device\Agregate\DeviceCustomer\DeviceCustomerEntity;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\PopupDeviceInfoWidgetPage\PopupDeviceInfoWidgetPageLoader;
use Shopware\Core\Framework\Uuid\Uuid;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\ListProducts\ListProductsLoader;
use Topdata\TopdataTopFinderProSW6\Content\Product\Events\DeviceProductsCriteriaEvent;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Device\DevicePage;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Device\DevicePageLoader;
use Shopware\Storefront\Page\Page;
use Shopware\Core\Checkout\Cart\Exception\CustomerNotLoggedInException;
use Symfony\Component\HttpKernel\EventListener\AbstractSessionListener;
use Topdata\TopdataTopFinderProSW6\Util\ControllerUtil;
use Topdata\TopdataFoundationSW6\Util\UtilUuid;

/**
 * Controller for handling TopFinder API requests.
 *
 * This controller provides various endpoints for the TopFinder functionality,
 * including retrieving series, types, devices, and handling device-related actions.
 * It was created in 10/2024, extracted from TopFinderController.
 */
#[Route(defaults: ['_routeScope' => ['storefront']])]
class TopFinderApiController extends StorefrontController
{
    public function __construct(
        private readonly EntityRepository              $topdataBrandRepository,
        private readonly EntityRepository              $topdataSeriesRepository,
        private readonly EntityRepository              $topdataDeviceTypeRepository,
        private readonly DeviceListPageLoader          $deviceListPageLoader,
        private readonly DeviceSearchPageLoader        $deviceSearchPageLoader,
        private readonly ListPageLoader                $listPageLoader,
        private readonly CompatibleDevicesWidgetLoader $compatibleDevicesWidgetLoader,
        private readonly PopupDeviceWidgetLoader       $popupDeviceWidgetLoader,
        private readonly BrandLetterPageLoader         $brandLetterPageLoader,
        private readonly OneBrandLetterPageLoader      $oneBrandLetterPageLoader,
        private readonly DevicePageLoader              $devicePageLoader,
        private readonly HelperService                 $helperService,
        private readonly DeviceToCustomerService       $deviceToCustomerService,
    )
    {
    }

    /**
     * Retrieves series for a given brand code.
     *
     * This method is called via AJAX to populate a dropdown with the series of a chosen brand.
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return JsonResponse JSON response containing series data or not found status
     */
    #[Route('/top-finder-api/get-series/{brandCode}', name: 'frontend.top-finder-api.get-series', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function getSeriesAction(string $brandCode, SalesChannelContext $salesChannelContext): JsonResponse
    {
        if (!trim($brandCode)) {
            return new JsonResponse(['found' => false]);
        }

        $brand = $this
            ->topdataBrandRepository
            ->search(
                (new Criteria())
                    ->addFilter(new EqualsFilter('enabled', true))
                    ->addFilter(new EqualsFilter('code', $brandCode)),
                $salesChannelContext->getContext()
            )
            ->getEntities()
            ->first();

        if (!$brand) {
            return new JsonResponse(['found' => false]);
        }

        $brandId = $brand->getId();
        $seriesMap = $this->_getSeriesMap($brandId);

        if (!count($seriesMap)) {
            return new JsonResponse(['found' => false]);
        }

        return new JsonResponse([
            'found' => true,
            'items' => $seriesMap,
        ]);
    }

    /**
     * Retrieves types for a given brand code.
     *
     * This method is called via AJAX to populate a dropdown with the types of a chosen brand.
     *
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return JsonResponse JSON response containing types data or not found status
     */
    #[Route('/top-finder-api/get-types/{brandCode}', name: 'frontend.top-finder-api.get-types', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function getTypes(string $brandCode, SalesChannelContext $salesChannelContext): JsonResponse
    {
        if (!trim($brandCode)) {
            return new JsonResponse(['found' => false]);
        }

        $brand = $this
            ->topdataBrandRepository
            ->search(
                (new Criteria())
                    ->addFilter(new EqualsFilter('enabled', true))
                    ->addFilter(new EqualsFilter('code', $brandCode)),
                $salesChannelContext->getContext()
            )
            ->getEntities()
            ->first();

        if (!$brand) {
            return new JsonResponse(['found' => false]);
        }

        $brandId = $brand->getId();

        $typesArray = $this->_getTypesArray($brandId);

        if (!count($typesArray)) {
            return new JsonResponse(['found' => false]);
        }

        return new JsonResponse([
            'found' => true,
            'items' => $typesArray,
        ]);
    }

    /**
     * Retrieves devices for a given brand, type, and series.
     *
     * This loads the devices of a brand (eg "canon") and optionally a series (eg "pixma") and a type (eg "printer")
     * it is called via ajax to populate a dropdown with the devices of a chosen brand
     *
     * This method is called via AJAX to populate a dropdown with the devices of a chosen brand, type, and series.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return JsonResponse JSON response containing devices data or not found status
     */
    #[Route('/top-finder-api/get-devices/{brandCode}/{typeCode}/{seriesCode}', name: 'frontend.top-finder-api.get-devices', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function getDevices(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $brandCode = $this->_resolveCode((string)$request->get('brandCode'), $this->topdataBrandRepository, $salesChannelContext->getContext());
        $seriesCode = $this->_resolveCode((string)$request->get('seriesCode'), $this->topdataSeriesRepository, $salesChannelContext->getContext());
        $typeCode = $this->_resolveCode((string)$request->get('typeCode'), $this->topdataDeviceTypeRepository, $salesChannelContext->getContext());

        if (($brandCode === false) || ($seriesCode === false) || ($typeCode === false)) {
            return new JsonResponse(['found' => false]);
        }

        $devicesArray = $this->helperService->getFilteredDevices($brandCode, $seriesCode, $typeCode);

        if (!count($devicesArray)) {
            return new JsonResponse(['found' => false]);
        }

        return new JsonResponse([
            'found' => true,
            'items' => $devicesArray,
        ]);
    }

    /**
     * Load more devices in the device search results.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response containing the HTML for additional search results
     */
    #[Route('/top-finder-api/device-search-more', name: 'frontend.top-finder-api.device-search-more', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function deviceSearchMore(Request $request, SalesChannelContext $context): JsonResponse
    {
        $page = $this->deviceSearchPageLoader->loadPage($request, $context);

        $result = ['success' => true];
        $result['html'] = $this->renderStorefront(
            '@Storefront/storefront/page/topfinder/device-search-more.html.twig',
            ['page' => $page]
        )->getContent();

        if ($result) {
            return new JsonResponse($result);
        }
    }

    /**
     * List devices for a specific brand.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChanelContext The sales channel context
     * @return JsonResponse JSON response containing the HTML for the brand devices list
     */
    #[Route('/top-finder-api/brand-devices/{brandCode}', name: 'frontend.top-finder-api.brand-devices', methods: ['GET', 'POST'], defaults: ['productid' => '', 'XmlHttpRequest' => true])]
    public function listBrandDevices(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $brandCode = $request->get('brandCode');
        if (!preg_match('/^[a-z0-9_-]+$/', $brandCode)) {
            throw new BadRequestHttpException('brandCode'/*, '/brandCode'*/);
        }

        $mode = $request->get('mode');
        if ($mode === 'series') {
            $view = '@Storefront/storefront/page/topfinder/one-brand-letter-series.html.twig';
        } elseif ($mode === 'types') {
            $view = '@Storefront/storefront/page/topfinder/one-brand-letter-types.html.twig';
        } else {
            $view = '@Storefront/storefront/page/topfinder/one-brand-letter-devices.html.twig';
        }

        $page = $this->oneBrandLetterPageLoader->load(
            $request,
            $salesChanelContext,
            $brandCode
        );

        $result = ['success' => true];
        $result['html'] = $this->renderStorefront($view, ['page' => $page])->getContent();

        if ($result) {
            return new JsonResponse($result);
        }
    }

    /**
     * Get compatible devices for a specific product.
     *
     * @param string $productid The ID of the product
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return JsonResponse JSON response containing the HTML for compatible devices
     */
    #[Route('/top-finder-api/compatible-devices/{productid}', name: 'frontend.top-finder-api.compatible-devices', methods: ['GET', 'POST'], defaults: ['productid' => '', 'XmlHttpRequest' => true])]
    public function compatibleDevices(string $productid, Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        if (!$productid) {
            return new JsonResponse(['success' => false, 'text' => 'Something went wrong']);
        }

        $result = ['success' => true];
        $page = $this->compatibleDevicesWidgetLoader->load($request, $salesChannelContext, $productid);
        $result['html'] = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/compatible-devices/index.html.twig', [
            'page' => $page,
        ])->getContent();
        $result['title'] = $page->product->getName() . ' ' . $this->trans('topdata-topfinder.productDevices');

        if ($result) {
            return new JsonResponse($result);
        }
    }

    /**
     * Get information for a new device popup.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return JsonResponse JSON response containing the HTML and data for the new device popup
     */
    #[Route('/top-finder-api/get-popup-device-new', name: 'frontend.top-finder-api.popup-device-new', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function popupDeviceNew(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $page = $this
            ->popupDeviceWidgetLoader
            ->loadNew($request, $salesChannelContext);

        $result = ['success' => true];

        if (!$page->device) {
            return new JsonResponse(['status' => 'false', 'message' => 'no Device Found'], 500);
        }

        $result['deviceId'] = $page->device->getId();
        $result['deviceUrl'] = $page->deviceUrl;
        $result['inDeviceList'] = $page->device->getInDeviceList();
        $result['userLogged'] = $page->other['userLogged'];
        if ($result['userLogged']) {
            $result['pathListAdd'] = $page->other['pathListAdd'];
            $result['pathListRemove'] = $page->other['pathListRemove'];
        }

        //        $devices = $this->getDevicesArray(
        //            $page->device->getBrand()->getId(),
        //            $page->device->getSeries() ? $page->device->getSeries()->getId() : 0,
        //            0
        //            //$page->device->getType() ? $page->device->getType()->getId() : false
        //        );
        //
        //        $series = $this->getSeriesArray($page->device->getBrand()->getId());
        //        $types = $this->getTypesArray($page->device->getBrand()->getId());

        //        $popupHome = [
        //            'brandskey' => $page->device->getBrand()->getCode(),
        //            'series' => $series,
        //            'serieskey' => $page->device->getSeries() ? $page->device->getSeries()->getCode() : null,
        //            'types' => $types,
        //            'typeskey' => $page->device->getType() ? $page->device->getType()->getCode() : null,
        //            'devices' => $devices,
        //            'deviceskey' => $page->device->getCode()
        //        ];
        //
        //        $result['popupHome'] = $popupHome;

        $devicelist = $this->deviceToCustomerService->getDeviceListInfo($salesChannelContext);

        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/box-device.html.twig',
                [
                    'page'       => $page,
                    'devicelist' => $devicelist,
                ]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    /**
     * List devices for a specific letter in a popup.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChanelContext The sales channel context
     * @return JsonResponse JSON response containing the HTML for the letter-specific device list
     */
    #[Route('/top-finder-api/popup-letter/{letter}', name: 'frontend.top-finder-api.popup-letter', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function listPopupLetter(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $letter = $request->get('letter');
        if (!preg_match('/^[a-z0]{1}$/', $letter)) {
            throw new BadRequestHttpException('letter'/*, '/letter'*/);
        }

        $page = $this->brandLetterPageLoader->loadJson($request, $salesChanelContext, $letter);
        $result = [];
        $result['success'] = true;
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-letter.html.twig',
                ['page' => $page]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder-api/popup-all-brands', name: 'frontend.top-finder-api.popup-all-brands', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function listPopupAllBrands(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $page = $this->listPageLoader->loadJsonBrands($request, $salesChanelContext);

        $result = [];
        $result['success'] = true;
        $result['html'] = $this->renderStorefront(
            '@TopdataTopFinderProSW6/storefront/widget/popup-form/all-brands.html.twig',
            ['page' => $page]
        )->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    /**
     * FIXME? UNUSED?
     */
    #[Route('/top-finder-api/popup-brandq/{code}', name: 'frontend.top-finder.popup_brandq', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function listPopupBrandDevices(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $code = $request->get('code');
        if (!preg_match('/^[a-z0-9_-]+$/', $code)) {
            throw new BadRequestHttpException('code'/*, '/code'*/);
        }

        //        $page = $this->listPageLoader->loadBrandDevicesJson($request, $salesChanelContext);

        $page = $this->oneBrandLetterPageLoader->loadJson(
            $request,
            $salesChanelContext,
            $code,
            'all'
        );

        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);

        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);

        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-devices.html.twig',
                ['page' => $page, 'mode' => 'all', 'listMode' => $listMode]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder-api/popup-series/{code}', name: 'frontend.top-finder-api.popup-series', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function listPopupSerieDevices($code, Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        if (!preg_match('/^[a-z0-9_-]+$/', $code)) {
            throw new BadRequestHttpException('code'/*, '/code'*/);
        }
        $page = $this->listPageLoader->loadSerieDevicesJson($request, $salesChanelContext);

        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);

        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devices.html.twig',
                ['page' => $page, 'listMode' => $listMode, 'hideMenuSeries' => true]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder-api/popup-type/{code}', name: 'frontend.top-finder-api.popup-type', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function listPopupTypeDevices($code, Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        if (!preg_match('/^[a-z0-9_-]+$/', $code)) {
            throw new BadRequestHttpException('code'/*, '/code'*/);
        }
        $page = $this->listPageLoader->loadTypeDevicesJson($request, $salesChanelContext);

        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);

        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devices.html.twig',
                ['page' => $page, 'listMode' => $listMode]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder-api/popup-devicelist', name: 'frontend.top-finder-api.popup-devicelist', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function popupDeviceList(Request $request, SalesChannelContext $context): JsonResponse
    {
        $page = $this->deviceListPageLoader->loadJson($request, $context);
        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);
        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/index.html.twig',
                ['page' => $page, 'listMode' => $listMode]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder-api/popup-devicelist-device', name: 'frontend.top-finder-api.popup-devicelist-device', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function popupDeviceListDevice(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');

        if (false === Uuid::isValid($id)) {
            return new JsonResponse(['success' => false]);
        }

        $page = $this->deviceListPageLoader->loadDeviceJson($request, $context);

        if (is_null($page->device)) {
            return new JsonResponse(['success' => false]);
        }

        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/device.html.twig',
                ['page' => $page]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder-api/popup-devicelist-subdevice-add', name: 'frontend.top-finder-api.popup-devicelist-subdevice-add', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function popupDeviceListDeviceSubdeviceAdd(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');

        if (false === Uuid::isValid($id)) {
            return new JsonResponse(['success' => false]);
        }

        $page = $this->deviceListPageLoader->loadAddSubdeviceJson($request, $context);

        if (is_null($page->device)) {
            return new JsonResponse(['success' => false]);
        }

        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/subdevice-add.html.twig',
                ['page' => $page]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder-api/popup-devicelist-subdevice-add-confirmed', name: 'frontend.top-finder-api.popup-devicelist-subdevice-add-confirmed', methods: ['POST'], defaults: ['XmlHttpRequest' => true])]
    public function popupDeviceListDeviceSubdeviceAddConfirmed(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');

        if (false === Uuid::isValid($id)) {
            return new JsonResponse(['success' => false]);
        }

        $page = $this->deviceListPageLoader->addSubdeviceJson($request, $context);

        if (is_null($page->device)) {
            return new JsonResponse(['success' => false]);
        }

        $result = [];
        $result['success'] = true;
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/device.html.twig',
                ['page' => $page]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder-api/popup-devicelist-subdevice-edit', name: 'frontend.top-finder-api.popup-devicelist-subdevice-edit', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function popupDeviceListDeviceSubdeviceEdit(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');

        if (false === Uuid::isValid($id)) {
            return new JsonResponse(['success' => false]);
        }

        $page = $this->deviceListPageLoader->loadEditSubdeviceJson($request, $context);

        if (is_null($page->device)) {
            return new JsonResponse(['success' => false]);
        }

        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/subdevice-edit.html.twig',
                ['page' => $page]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder-api/popup-devicelist-subdevice-edit-confirmed', name: 'frontend.top-finder-api.popup-devicelist-subdevice-edit-confirmed', methods: ['POST'], defaults: ['XmlHttpRequest' => true])]
    public function popupDeviceListDeviceSubdeviceEditConfirmed(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');

        if (false === Uuid::isValid($id)) {
            return new JsonResponse(['success' => false]);
        }

        $page = $this->deviceListPageLoader->editSubdeviceJson($request, $context);

        if (is_null($page->device)) {
            return new JsonResponse(['success' => false]);
        }

        $result = [];
        $result['success'] = true;
        $result['html'] = $this
            ->renderStorefront(
            //                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/device.html.twig',
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/subdevice-edit.html.twig',
                ['page' => $page]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder-api/popup-devicelist-subdevice-remove', name: 'frontend.top-finder-api.popup-devicelist-subdevice-remove', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function popupDeviceListDeviceSubdeviceRemove(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');

        if (false === Uuid::isValid($id)) {
            return new JsonResponse(['success' => false]);
        }

        $page = $this->deviceListPageLoader->removeSubdeviceJson($request, $context);

        //        if(is_null($page->device)) {
        //            return new JsonResponse(['success'=>false]);
        //        }

        $result = [];
        $result['success'] = true;
        if ($page->fallBack) {
            $result['html'] = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/index.html.twig',
                    ['page' => $page]
                )
                ->getContent();
        } else {
            $result['html'] = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/device.html.twig',
                    ['page' => $page]
                )
                ->getContent();
        }

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder-api/popup-history', name: 'frontend.top-finder-api.popup-history', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function listPopupHistory(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $page = $this->listPageLoader->loadHistoryDevicesJson($request, $salesChanelContext);

        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);

        $result = [];
        $result['success'] = true;
        $result['returnPath'] = ControllerUtil::getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devices.html.twig',
                ['page' => $page, 'listMode' => $listMode]
            )
            ->getContent();

        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder-api/device-filters/{deviceCode}/{propertyGroupOptionId}', name: 'frontend.top-finder-api.device-tab-filters', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function getDeviceTabFilters(Request $request, SalesChannelContext $context): JsonResponse
    {
        // Allows to fetch only aggregations over the gateway.
        $request->request->set('only-aggregations', true);

        // Allows to convert all post-filters to filters. This leads to the fact that only aggregation values are returned, which are combinable with the previous applied filters.
        $request->request->set('reduce-aggregations', true);

        /** @var DevicePage $page */
        $page = $this->devicePageLoader->loadDevicePageData($request, $context, true);
        //$listing = $page->listing->getResult();
        $listing = $page->listing;

        $mapped = [];
        foreach ($listing->getAggregations() as $aggregation) {
            $mapped[$aggregation->getName()] = $aggregation;
        }

        $response = new JsonResponse($mapped);
        $response->headers->set(AbstractSessionListener::NO_AUTO_CACHE_CONTROL_HEADER, '1');

        return $response;
    }

    #[Route('/top-finder-api/device-filters/{deviceCode}', name: 'frontend.top-finder-api.device-filters', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function getDeviceFilters(Request $request, SalesChannelContext $context): JsonResponse
    {
        // Allows to fetch only aggregations over the gateway.
        $request->request->set('only-aggregations', true);

        // Allows to convert all post-filters to filters. This leads to the fact that only aggregation values are returned, which are combinable with the previous applied filters.
        $request->request->set('reduce-aggregations', true);

        /** @var DevicePage $page */
        $page = $this->devicePageLoader->loadDevicePageData($request, $context, true);
        //$listing = $page->listing->getResult();
        $listing = $page->listing;

        $mapped = [];
        foreach ($listing->getAggregations() as $aggregation) {
            $mapped[$aggregation->getName()] = $aggregation;
        }

        $response = new JsonResponse($mapped);
        $response->headers->set(AbstractSessionListener::NO_AUTO_CACHE_CONTROL_HEADER, '1');

        return $response;
    }

    /**
     * private helper
     * Retrieves an array of types for a given brand ID.
     *
     * @param string $brandId The ID of the brand
     * @return array An array of types, with type code as key and label as value
     */
    private function _getTypesArray(string $brandId): array
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('enabled', true));
        $criteria->addFilter(new EqualsFilter('brandId', $brandId));
        $criteria->addSorting(new FieldSorting('label', FieldSorting::ASCENDING));
        $types = $this->topdataDeviceTypeRepository->search($criteria, Context::createDefaultContext())->getEntities();

        if (!count($types)) {
            return [];
        }

        $typesArray = [];
        foreach ($types as $type) {
            $typesArray[$type->getCode()] = $type->getLabel();
        }

        return $typesArray;
    }

    /**
     * Private helper function to resolve a code to an id.
     *
     * @param string $code The code to resolve
     * @param EntityRepository $repository The repository to search in
     * @param Context $context The context for the search
     * @return string|bool The resolved ID or false if not found
     */
    private function _resolveCode($code, $repository, $context)
    {
        if ($code == '0') {
            return '0';
        }

        if ($code == '-1') {
            return '-1';
        }

        if (!trim($code)) {
            return false;
        }

        $entity = $repository
            ->search(
                (new Criteria())
                    ->addFilter(new EqualsFilter('enabled', true))
                    ->addFilter(new EqualsFilter('code', $code)),
                $context
            )
            ->getEntities()
            ->first();

        if (!$entity) {
            return false;
        }

        return $entity->getId();
    }

    /**
     * private helper
     * Retrieves an array of series for a given brand ID.
     *
     * @param string $brandId The ID of the brand
     * @return array An array of series, with series code as key and label as value
     */
    private function _getSeriesMap(string $brandId): array
    {
        $seriesArray = [];

        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('enabled', true));
        $criteria->addFilter(new EqualsFilter('brandId', $brandId));
        $criteria->addSorting(new FieldSorting('label', FieldSorting::ASCENDING));
        $series = $this->topdataSeriesRepository->search($criteria, Context::createDefaultContext())->getEntities();

        if (!count($series)) {
            return $seriesArray;
        }

        foreach ($series as $serie) {
            $seriesArray[$serie->getCode()] = $serie->getLabel();
        }

        return $seriesArray;
    }
}
