<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Controller;

use Doctrine\DBAL\Connection;
use Shopware\Core\Checkout\Cart\Exception\CustomerNotLoggedInException;
use Shopware\Core\Content\Product\SalesChannel\Listing\ProductListingResult;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityCollection;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\Framework\Validation\DataBag\RequestDataBag;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Controller\StorefrontController;
use Shopware\Storefront\Page\Page;
use Symfony\Component\HttpFoundation\Cookie;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Topdata\TopdataConnectorSW6\Core\Content\Device\DeviceEntity;
use Topdata\TopdataTopFinderProSW6\Component\Collection;
use Topdata\TopdataTopFinderProSW6\Constants\CookieKeyNameConstants;
use Topdata\TopdataTopFinderProSW6\Content\Product\Events\DeviceProductsCriteriaEvent;
use Topdata\TopdataTopFinderProSW6\Service\DeviceToCustomerService;
use Topdata\TopdataTopFinderProSW6\Service\HelperService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\BrandLetter\BrandLetterPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Device\DevicePage;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Device\DevicePageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\ListProducts\ListProductsLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\OneBrandLetter\OneBrandLetterPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\PopupDeviceInfoWidgetPage\PopupDeviceInfoWidgetPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\PopupDeviceWidget\PopupDeviceWidgetLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\PopupFormWidget\PopupFormWidgetLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\DeviceListPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\DeviceSearchPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\DeviceSuggestPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\ListPageLoader;
use Topdata\TopdataTopFinderProSW6\Util\ControllerUtil;
use Topdata\TopdataFoundationSW6\Util\UtilUuid;

/**
 * Controller for handling TopFinder related actions in the storefront.
 *
 * This controller manages various functionalities related to device search,
 * listing, and management in the TopFinder Pro plugin for Shopware 6.
 *
 * TODO: this controller is too big and should be split into multiple controllers
 */
#[Route(defaults: ['_routeScope' => ['storefront']])]
class TopFinderController extends StorefrontController
{
    public const POPUP_DEVICES_PRELOAD_LENGTH = 20;


    public function __construct(
        private readonly EntityRepository                $topdataBrandRepository,
        private readonly EntityRepository                $topdataDeviceRepository,
        private readonly EntityRepository                $topdataSeriesRepository,
        private readonly EntityRepository                $topdataDeviceTypeRepository,
        private readonly DeviceListPageLoader            $deviceListPageLoader,
        private readonly DeviceSuggestPageLoader         $deviceSuggestPageLoader,
        private readonly DeviceSearchPageLoader          $deviceSearchPageLoader,
        private readonly SettingsService                 $settingsService,
        private readonly ListPageLoader                  $listPageLoader,
        private readonly PopupDeviceWidgetLoader         $popupDeviceWidgetLoader,
        private readonly PopupFormWidgetLoader           $popupFormWidgetLoader,
        private readonly BrandLetterPageLoader           $brandLetterPageLoader,
        private readonly OneBrandLetterPageLoader        $oneBrandLetterPageLoader,
        private readonly PopupDeviceInfoWidgetPageLoader $popupDeviceInfoWidgetPageLoader,
        private readonly ListProductsLoader              $listProductsLoader,
        private readonly DevicePageLoader                $devicePageLoader,
        private readonly HelperService                   $helperService,
        private readonly DeviceToCustomerService         $deviceToCustomerService,
    )
    {
    }

    /**
     * Retrieves an entity collection based on the provided parameters and sorting field.
     *
     * This method constructs a Criteria object with the given parameters and sorting field,
     * and then performs a search on the provided repository to return the matching entities.
     *
     * @param EntityRepository $repository The repository to search in.
     * @param array $params The parameters to filter the entities.
     * @param string|null $sortingField The field to sort the entities by. Defaults to null.
     * @param bool $deleteNullParams Whether to exclude parameters with null values. Defaults to true.
     * @return EntityCollection The collection of entities that match the criteria.
     */
    private function _getEntityCollection(EntityRepository $repository, array $params = [], ?string $sortingField = null, bool $deleteNullParams = true): EntityCollection
    {
        $criteria = new Criteria();
        foreach ($params as $key => $value) {
            if (($value === null) && $deleteNullParams) {
                continue;
            }
            $criteria->addFilter(new EqualsFilter($key, $value));
        }
        if ($sortingField !== null) {
            $criteria->addSorting(new FieldSorting($sortingField, FieldSorting::ASCENDING));
        }
        $criteria->setLimit(999);

        return $repository->search($criteria, Context::createDefaultContext())->getEntities();
    }

    /**
     * Handles the request to get a device page.
     *
     * This method is responsible for loading the device page based on the provided device code.
     * It checks if the device is in the device list, sets the device history if applicable,
     * fills select boxes with relevant data, and renders the device page.
     *
     * @param Request $request The HTTP request object.
     * @param SalesChannelContext $salesChannelContext The sales channel context.
     * @return Response The HTTP response object containing the rendered device page.
     */
    #[Route('/top-finder/device/{deviceCode}', name: 'frontend.top-finder.device', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function getDeviceNew(Request $request, SalesChannelContext $salesChannelContext): Response
    {
        $devicePage = $this->devicePageLoader->loadDevicePageData($request, $salesChannelContext);

        // Check if the device is in the device list and set the flag accordingly.
        $devicePage->getDevice()->setInDeviceList(
            in_array($devicePage->getDeviceId(), $this->deviceToCustomerService->getCustomerDeviceIds($salesChannelContext))
        );

        // If the request is not an XMLHttpRequest and device history is enabled, set the device to history.
        if ($request->isXmlHttpRequest() === false && $this->settingsService->getBool('showDeviceHistory')) {
            $this->_setDeviceToHistory($request, $devicePage);
        }

        // If the request is not an XMLHttpRequest, fill select boxes with relevant data.
        if ($request->isXmlHttpRequest() === false) {
            $this->_fillSelectboxes($devicePage, $devicePage->getDevice(), $salesChannelContext, $request);
        }

        // Render the device page.
        $response = $this->renderStorefront(
            '@TopdataTopFinderProSW6/storefront/page/topfinder/device-page.html.twig',
            ['page' => $devicePage]
        );

        // If the request is not an XMLHttpRequest and device history is enabled, set a cookie for device history.
        if ($request->isXmlHttpRequest() === false) {
            if ($this->settingsService->getBool('showDeviceHistory')) {
                $response->headers->setCookie(
                    Cookie::create(
                        CookieKeyNameConstants::DEVICE_HISTORY,
                        implode(',', $devicePage->deviceHistory),
                        time() + 3600 * 24 * 30
                    )
                );
            }
        }

        return $response;
    }

    #[Route('/top-finder/device/{deviceCode}/{propertyGroupOptionId}', name: 'frontend.top-finder.device_tab', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function getDeviceTab(Request $request, SalesChannelContext $context): Response
    {
        $devicePage = $this->devicePageLoader->loadDevicePageData($request, $context);

        $devicePage->getDevice()->setInDeviceList(in_array($devicePage->getDeviceId(), $this->deviceToCustomerService->getCustomerDeviceIds($context)));

        if ($request->isXmlHttpRequest() === false
            && $this->settingsService->getBool('showDeviceHistory')) {
            $this->_setDeviceToHistory($request, $devicePage);
        }

        if ($request->isXmlHttpRequest() === false) {
            $this->_fillSelectboxes($devicePage, $devicePage->getDevice(), $context, $request);
        }
        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/device-page.html.twig', ['page' => $devicePage]);

        if ($request->isXmlHttpRequest() === false) {
            if ($this->settingsService->getBool('showDeviceHistory')) {
                $response->headers->setCookie(
                    Cookie::create(
                        CookieKeyNameConstants::DEVICE_HISTORY,
                        implode(',', $devicePage->deviceHistory),
                        time() + 3600 * 24 * 30
                    )
                );
            }
        }

        return $response;
    }

    /**
     * Adds the current device to the device history.
     *
     * This method retrieves the device history from the request cookies, updates it with the current device,
     * and ensures the history does not exceed the configured limit. The updated history is then set in the page object.
     *
     * @param Request $request The HTTP request object.
     * @param Page $devicePage The page object containing the device information.
     * @return void
     */
    private function _setDeviceToHistory(Request $request, DevicePage $devicePage): void
    {
        $deviceId = $devicePage->getDeviceId();

        // Retrieve the device history from the request cookies.
        $cokie = $request->cookies->get(CookieKeyNameConstants::DEVICE_HISTORY);

        // Initialize the device history array.
        $devicesHistory = [];

        // If the cookie exists, split it into an array of device IDs.
        if ($cokie) {
            $devicesHistory = explode(',', $cokie);
        }

        // Ensure the device history is an array.
        if (!is_array($devicesHistory)) {
            $devicesHistory = [];
        }

        // Check if the current device is already in the history and remove it if found.
        $historyKey = array_search($deviceId, $devicesHistory);
        if ($historyKey !== false) {
            unset($devicesHistory[$historyKey]);
        }

        // Add the current device to the end of the history array.
        $devicesHistory[] = $deviceId;

        // Ensure the device history does not exceed the configured limit.
        while (count($devicesHistory) > $this->settingsService->getInt('deviceHistoryElements')) {
            array_shift($devicesHistory);
        }

        // Set the updated device history in the page object.
        $devicePage->deviceHistory = $devicesHistory;
    }

//    /**
//     * 10/2024 commented out (unused)
//     *
//     *
//     * Retrieves the product listing for a given device.
//     *
//     * This method creates a Criteria object using the product IDs associated with the given device,
//     * dispatches a DeviceProductsCriteriaEvent to allow for modifications to the criteria,
//     * and then searches the product repository with the modified criteria.
//     *
//     * @param DeviceEntity $device The device entity containing the product IDs.
//     * @param SalesChannelContext $context The sales channel context.
//     * @return ProductListingResult The resulting product listing.
//     */
//    private function _getProductListingByDevice(DeviceEntity $device, SalesChannelContext $context): ProductListingResult
//    {
//        $criteria = new Criteria($device->getProducts()->getIds());
//
//        $this->eventDispatcher->dispatch(
//            new DeviceProductsCriteriaEvent($criteria, $context)
//        );
//
//        $result = $this->productRepository->search($criteria, $context);
//        $result = ProductListingResult::createFrom($result);
//
//        return $result;
//    }

    /**
     * Fills the select boxes on the device page with relevant data.
     *
     * This method populates the select boxes on the device page with data such as brand, series, type, and device history.
     * It retrieves the necessary data from the repositories and sets the appropriate values in the finder structure.
     *
     * @param Page $page The page object containing the header and other page data.
     * @param DeviceEntity $device The device entity containing the device details.
     * @param SalesChannelContext $context The sales channel context.
     * @param Request $request The HTTP request object.
     * @return void
     */
    private function _fillSelectboxes(Page $page, DeviceEntity $device, SalesChannelContext $context, Request $request): void
    {
        $finderStruct = $page->getHeader()->getExtension('finder');
        if ($finderStruct === null) {
            $finderStruct = new Collection();
        }

        if ($device->getBrandId()) {
            $brand = $this
                ->topdataBrandRepository
                ->search(
                    (new Criteria([$device->getBrandId()]))
                        ->addFilter(new EqualsFilter('enabled', true)),
                    $context->getContext()
                )
                ->getEntities()
                ->first();
            if ($brand) {
                $finderStruct->set('brandCode', $brand->getCode());
            }
        }

        if ($device->getSeriesId()) {
            $series = $this
                ->topdataSeriesRepository
                ->search(
                    (new Criteria([$device->getSeriesId()]))
                        ->addFilter(new EqualsFilter('enabled', true)),
                    $context->getContext()
                )
                ->getEntities()
                ->first();
            if ($series) {
                $finderStruct->set('seriesCode', $series->getCode());
            }
        }

        if ($device->getTypeId()) {
            $type = $this
                ->topdataDeviceTypeRepository
                ->search(
                    (new Criteria([$device->getTypeId()]))
                        ->addFilter(new EqualsFilter('enabled', true)),
                    $context->getContext()
                )
                ->getEntities()
                ->first();
            if ($type) {
                $finderStruct->set('typeCode', $type->getCode());
            }
        }

        $finderStruct->set('deviceCode', $device->getCode());

        if (is_array($page->deviceHistory) && count($page->deviceHistory)
            && ($this->settingsService->getBool('showDevicesSlider')
                || $this->settingsService->getBool('showDeviceHistorySelect'))
        ) {
            $criteria = (new Criteria(array_reverse($page->deviceHistory)))
                ->addFilter(new EqualsFilter('enabled', true))
                ->addAssociations(['brand', 'media']);
            $devicesHistory = $this->topdataDeviceRepository->search($criteria, $context->getContext())->getElements();
            foreach ($devicesHistory as $dev) {
                if ($this->settingsService->getBool('showDevicelist')) {
                    $dev->setInDeviceList(
                        in_array(
                            $dev->getId(),
                            $this->deviceToCustomerService->getCustomerDeviceIds($context)
                        )
                    );
                } else {
                    $dev->setInDeviceList(false);
                }
            }
            $devicesHistoryStruct = new Collection();
            $devicesHistoryStruct->set('devices', $devicesHistory);
            if ($this->settingsService->getBool('showDeviceHistorySelect')) {
                $page->getHeader()->removeExtension('deviceHistory');
                $page->getHeader()->addExtension('deviceHistory', $devicesHistoryStruct);
            }
            if ($this->settingsService->getBool('showDevicesSlider')) {
                $devicesHistoryStruct->set('userHiddable', $this->settingsService->getBool('deviceSliderUserHide'));
                $devicesHistoryStruct->set('hidden', $request->cookies->get(CookieKeyNameConstants::DEVICE_SLIDER) === 'hidden');

                $page->getHeader()->removeExtension('deviceSlider');
                $page->getHeader()->addExtension('deviceSlider', $devicesHistoryStruct);
            }
        }

        if ($this->settingsService->getConfig('showSeries')) {
            $series = $this->_getEntityCollection(
                $this->topdataSeriesRepository,
                [
                    'enabled' => true,
                    'brandId' => $device->getBrandId(),
                ],
                'label'
            );
            $finderStruct->set('series', $series);
        }

        if ($this->settingsService->getConfig('showTypes')) {
            $types = $this->_getEntityCollection(
                $this->topdataDeviceTypeRepository,
                [
                    'enabled' => true,
                    'brandId' => $device->getBrandId(),
                ],
                'label'
            );
            $finderStruct->set('types', $types);
        }
        $devices = $this->helperService->getFilteredDevices($device->getBrandId(), $device->getSeriesId(), $device->getTypeId(), false);
        //        $devices = $this->getEntityCollection($this->topdataDeviceRepository, $deviceParams, 'model', false);
        $finderStruct->set('devices', $devices);

        //        $finderStruct->set('test', true);
        //        $page->getHeader()->removeExtension('finder');
        //        $page->getHeader()->addExtension('finder', $finderStruct);
    }


    /**
     * renders "account/device-list.html.twig"
     *
     * Handles the request to display the device list for the logged-in customer.
     *
     * This method checks if the customer is logged in and not a guest. If the customer is not logged in,
     * it throws a CustomerNotLoggedInException. If the customer is logged in, it loads the device list page
     * and sets the meta information to prevent indexing by search engines. Finally, it renders the device list page.
     *
     * @param Request $request The HTTP request object.
     * @param SalesChannelContext $context The sales channel context.
     * @return Response The HTTP response object containing the rendered device list page.
     * @throws CustomerNotLoggedInException If the customer is not logged in or is a guest.
     */
    #[Route('/account/devicelist', name: 'frontend.account.devicelist.page', methods: ['GET'], defaults: ['_loginRequired' => true])]
    public function deviceList(Request $request, SalesChannelContext $context): Response
    {
        if (!$context->getCustomer() || $context->getCustomer()->getGuest()) {
            throw new CustomerNotLoggedInException(
                Response::HTTP_FORBIDDEN,
                'Customer is not logged in.',
                'Customer is not logged in.'
            );
        }

        $page = $this->deviceListPageLoader->loadPage($request, $context);

        $page->getMetaInformation()->setRobots('noindex,nofollow');

        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/account/device-list.html.twig', [
            'page' => $page,
        ]);
    }

    /**
     * Handles the device suggestion request.
     *
     * This method loads the device suggestion page based on the provided request and context,
     * renders the corresponding Twig template, and sets the appropriate HTTP headers.
     *
     * @param Request $request The HTTP request object.
     * @param SalesChannelContext $context The sales channel context.
     * @return Response The HTTP response object containing the rendered device suggestion page.
     */
    #[Route('/device-suggest', name: 'frontend.top-finder.device_suuggest', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function deviceSuggest(Request $request, SalesChannelContext $context): Response
    {
        $page = $this->deviceSuggestPageLoader->load($request, $context);

        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/device-suggest/index.html.twig', ['page' => $page]);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }


    /**
     * Handles the device search request.
     *
     * This method loads the device search page based on the provided request and context,
     * renders the corresponding Twig template, and sets the appropriate HTTP headers.
     *
     * @param Request $request The HTTP request object.
     * @param SalesChannelContext $context The sales channel context.
     * @return Response The HTTP response object containing the rendered device search page.
     */
    #[Route('/device-search', name: 'frontend.top-finder.device_search.page', methods: ['GET'])]
    public function deviceSearch(Request $request, SalesChannelContext $context): Response
    {
        $page = $this->deviceSearchPageLoader->loadPage($request, $context);

        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/device-search.html.twig', ['page' => $page]);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }


    /**
     * Handles the request to list all brands.
     *
     * This method loads the brand listing page based on the provided request and context,
     * and renders the corresponding Twig template.
     *
     * @param Request $request The HTTP request object.
     * @param SalesChannelContext $salesChanelContext The sales channel context.
     * @return Response The HTTP response object containing the rendered brand listing page.
     */
    #[Route('/top-finder/brands', name: 'frontend.top-finder.brands', methods: ['GET'])]
    public function listBrands(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $page = $this->listPageLoader->loadPage($request, $salesChanelContext, ListPageLoader::TYPE_BRANDS);

        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list.html.twig', ['page' => $page]);
    }


    /**
     * renders "one-brand-letter.html.twig"
     *
     * Handles the request to list a specific brand
     *
     * This method loads the brand listing page based on the provided request and context,
     * and renders the corresponding Twig template. It also sets the brand code and retrieves
     * series and types if configured to do so.
     *
     * @param Request $request The HTTP request object.
     * @param SalesChannelContext $salesChanelContext The sales channel context.
     * @return Response The HTTP response object containing the rendered brand listing page.
     * @throws BadRequestHttpException If the brand code is invalid.
     */
    #[Route('/top-finder/brand/{brandCode}', name: 'frontend.top-finder.brand', methods: ['GET'])]
    public function listBrand(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $brandCode = $request->get('brandCode');
        if (!preg_match('/^[a-z0-9_-]+$/', $brandCode)) {
            throw new BadRequestHttpException('brandCode'/*, '/brandCode'*/);
        }

        $page = $this->oneBrandLetterPageLoader->load(
            $request,
            $salesChanelContext,
            $brandCode
        );

        $finderStruct = $page->getHeader()->getExtension('finder');
        if ($finderStruct !== null) {
            $finderStruct->set('brandCode', $page->brand['code']);

            if ($this->settingsService->getConfig('showSeries')) {
                $series = $this->_getEntityCollection(
                    $this->topdataSeriesRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page->brand['id'],
                    ],
                    'label'
                );
                $finderStruct->set('series', $series);
            }

            if ($this->settingsService->getConfig('showTypes')) {
                $types = $this->_getEntityCollection(
                    $this->topdataDeviceTypeRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page->brand['id'],
                    ],
                    'label'
                );
                $finderStruct->set('types', $types);
            }

            $devicesArray = $this->helperService->getFilteredDevices($page->brand['id'], 0, 0, false);
            $finderStruct->set('devices', $devicesArray);
        }

        return $this->renderStorefront(
            '@TopdataTopFinderProSW6/storefront/page/topfinder/one-brand-letter.html.twig',
            ['page' => $page]
        );
    }

    /**
     * renders "list.html.twig"
     *
     * Handles the request to list a specific brand.
     *
     * This method loads the brand listing page based on the provided request and context,
     * and renders the corresponding Twig template. It also sets the brand code and retrieves
     * series and types if configured to do so.
     *
     * @param Request $request The HTTP request object.
     * @param SalesChannelContext $salesChanelContext The sales channel context.
     * @return Response The HTTP response object containing the rendered brand listing page.
     * @throws BadRequestHttpException If the brand code is invalid.
     */
    #[Route('/top-finder/brandq/{code}', name: 'frontend.top-finder.brandq', methods: ['GET'])]
    public function listBrandQ(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $brandCode = $request->get('code');
        if (!preg_match('/^[a-z0-9_-]+$/', $brandCode)) {
            throw new BadRequestHttpException('brandCode'/*, '/brandCode'*/);
        }
        $page = $this->listPageLoader->loadPage($request, $salesChanelContext, ListPageLoader::TYPE_BRAND);

        $finderStruct = $page->getHeader()->getExtension('finder');
        if ($finderStruct !== null) {
            $finderStruct->set('brandCode', $page->model->getCode());

            if ($this->settingsService->getConfig('showSeries')) {
                $series = $this->_getEntityCollection(
                    $this->topdataSeriesRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page->model->getId(),
                    ],
                    'label'
                );
                $finderStruct->set('series', $series);
            }

            if ($this->settingsService->getConfig('showTypes')) {
                $types = $this->_getEntityCollection(
                    $this->topdataDeviceTypeRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page->model->getId(),
                    ],
                    'label'
                );
                $finderStruct->set('types', $types);
            }

            $devicesArray = $this->helperService->getFilteredDevices($page->model->getId(), 0, 0, false);
            //$devicesArray = $this->getDevicesArray($page->model->getId(), -1, -1, false);
            $finderStruct->set('devices', $devicesArray);
        }

        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list.html.twig', ['page' => $page]);
    }

    /**
     * Handles the request to list all series for a given brand.
     *
     * This method loads the series listing page based on the provided request and context,
     * and renders the corresponding Twig template. It also sets the brand code and series code,
     * and retrieves series and types if configured to do so.
     *
     * @param Request $request The HTTP request object.
     * @param RequestDataBag $data The data bag containing request data.
     * @param SalesChannelContext $context The sales channel context.
     * @return Response The HTTP response object containing the rendered series listing page.
     */
    #[Route('/top-finder/series/{code}', name: 'frontend.top-finder.series', methods: ['GET'])]
    public function listSeries(Request $request, RequestDataBag $data, SalesChannelContext $context): Response
    {
        // Load the series listing page.
        $page = $this->listPageLoader->loadPage($request, $context, ListPageLoader::TYPE_SERIES);

        // Get the finder structure from the page header.
        $finderStruct = $page->getHeader()->getExtension('finder');
        if ($finderStruct !== null) {
            // Set the brand code and series code in the finder structure.
            $finderStruct->set('brandCode', $page->model->getBrand()->getCode());
            $finderStruct->set('seriesCode', $page->model->getCode());

            // Retrieve and set the series if configured to do so.
            if ($this->settingsService->getConfig('showSeries')) {
                $series = $this->_getEntityCollection(
                    $this->topdataSeriesRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page->model->getBrand()->getId(),
                    ],
                    'label'
                );
                $finderStruct->set('series', $series);
            }

            // Retrieve and set the types if configured to do so.
            if ($this->settingsService->getConfig('showTypes')) {
                $types = $this->_getEntityCollection(
                    $this->topdataDeviceTypeRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page->model->getBrand()->getId(),
                    ],
                    'label'
                );
                $finderStruct->set('types', $types);
            }

            // Retrieve and set the devices array.
            $devicesArray = $this->helperService->getFilteredDevices(
                $page->model->getBrand()->getId(),
                $page->model->getId(),
                0,
                false
            );
            $finderStruct->set('devices', $devicesArray);
        }

        // Render the series listing page.
        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list.html.twig', ['page' => $page]);
    }

    /**
     * Handles the request to list all types for a given brand.
     *
     * This method loads the type listing page based on the provided request and context,
     * and renders the corresponding Twig template. It also sets the brand code and type code,
     * and retrieves series and types if configured to do so.
     *
     * @param Request $request The HTTP request object.
     * @param RequestDataBag $data The data bag containing request data.
     * @param SalesChannelContext $context The sales channel context.
     * @return Response The HTTP response object containing the rendered type listing page.
     */
    #[Route('/top-finder/type/{code}', name: 'frontend.top-finder.type', methods: ['GET'])]
    public function listType(Request $request, RequestDataBag $data, SalesChannelContext $context): Response
    {
        // Load the type listing page.
        $page = $this->listPageLoader->loadPage($request, $context, ListPageLoader::TYPE_TYPE);
        $finderStruct = $page->getHeader()->getExtension('finder');
        if ($finderStruct !== null) {
            // Set the brand code and type code in the finder structure.
            $finderStruct->set('brandCode', $page->model->getBrand()->getCode());
            $finderStruct->set('typeCode', $page->model->getCode());

            // Retrieve and set the series if configured to do so.
            if ($this->settingsService->getConfig('showSeries')) {
                $series = $this->_getEntityCollection(
                    $this->topdataSeriesRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page->model->getBrand()->getId(),
                    ],
                    'label'
                );
                $finderStruct->set('series', $series);
            }

            // Retrieve and set the types if configured to do so.
            if ($this->settingsService->getConfig('showTypes')) {
                $types = $this->_getEntityCollection(
                    $this->topdataDeviceTypeRepository,
                    [
                        'enabled' => true,
                        'brandId' => $page->model->getBrand()->getId(),
                    ],
                    'label'
                );
                $finderStruct->set('types', $types);
            }

            // Retrieve and set the devices array.
            $devicesArray = $this->helperService->getFilteredDevices(
                $page->model->getBrand()->getId(),
                0,
                $page->model->getId(),
                false
            );
            $finderStruct->set('devices', $devicesArray);
        }

        // Render the type listing page.
        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list.html.twig', ['page' => $page]);
    }

    #[Route('/top-finder/popupForm', name: 'frontend.top-finder.popup_form', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function popupForm(Request $request, SalesChannelContext $salesChannelContext): Response
    {
        $page = $this->popupFormWidgetLoader->load($request, $salesChannelContext);
        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/popup-form/index.html.twig', ['page' => $page]);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder/get-popup-device', name: 'frontend.top-finder.popup_device', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function popupDevice(Request $request, SalesChannelContext $salesChannelContext): Response
    {
        $page = $this->popupDeviceWidgetLoader->load($request, $salesChannelContext);
        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/popup-device/index.html.twig', ['page' => $page]);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder/letter/{letter}', name: 'frontend.top-finder.letter', methods: ['GET'])]
    public function listLetter(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $letter = $request->get('letter');
        if (!preg_match('/^[a-z0]{1}$/', $letter)) {
            throw new BadRequestHttpException('letter'/*, '/letter'*/);
        }

        $page = $this->brandLetterPageLoader->load($request, $salesChanelContext, $letter);

        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/brand-letter.html.twig', ['page' => $page]);
    }

    #[Route('/top-finder/letter-brand/{brandCode}', name: 'frontend.top-finder.letter-brand', methods: ['GET'])]
    public function listLetterBrand(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $brandCode = $request->get('brandCode');
        if (!preg_match('/^[a-z0-9_-]+$/', $brandCode)) {
            throw new BadRequestHttpException('brandCode'/*, '/brandCode'*/);
        }

        $page = $this->oneBrandLetterPageLoader->load($request, $salesChanelContext, $brandCode);

        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/one-brand-letter.html.twig', ['page' => $page]);
    }

    #[Route('/top-finder/history-remove/{deviceId}', name: 'frontend.top-finder.history-remove', methods: ['GET'])]
    public function historyRemove(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $deviceId = $request->get('deviceId');
        if (!UtilUuid::isValidUuid($deviceId)) {
            throw new BadRequestHttpException('deviceId');
        }

        $referer = $request->headers->get('referer');

        $cokie = $request->cookies->get(CookieKeyNameConstants::DEVICE_HISTORY);

        $devicesHistory = [];

        if ($cokie) {
            $devicesHistory = explode(',', $cokie);
        }

        if (!is_array($devicesHistory)) {
            $devicesHistory = [];
        }

        $historyKey = array_search($deviceId, $devicesHistory);

        if ($historyKey !== false) {
            unset($devicesHistory[$historyKey]);
        }

        if (!$referer) {
            $referer = '/';
        }

        $response = new RedirectResponse($referer);

        $response->headers->setCookie(
            Cookie::create(
                CookieKeyNameConstants::DEVICE_HISTORY,
                implode(',', $devicesHistory),
                time() + 3600 * 24 * 30
            )
        );

        return $response;
    }

    #[Route('/top-finder/popup-device-info/{deviceId}', name: 'frontend.top-finder.popup_device_info', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function popupDeviceInfo(Request $request, SalesChannelContext $salesChannelContext): Response
    {
        $deviceId = $request->get('deviceId');

        if (!UtilUuid::isValidUuid($deviceId)) {
            throw new BadRequestHttpException('deviceId');
        }

        if (!$salesChannelContext->getCustomer() || $salesChannelContext->getCustomer()->getGuest()) {
            throw new BadRequestHttpException('userId');
        }

        $userId = $salesChannelContext->getCustomer()->getId();

        $page = $this->popupDeviceInfoWidgetPageLoader->load($request, $salesChannelContext, $deviceId, $userId);
        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/popup-device-info/index.html.twig', ['page' => $page]);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    /**
     * @throws CustomerNotLoggedInException
     */
    #[Route('/top-finder/devicelist-find', name: 'frontend.top-finder.devicelist-find', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function deviceListFind(Request $request, SalesChannelContext $context): Response
    {
        $page = $this->deviceListPageLoader->loadPage($request, $context);

        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/account/device-list-find.html.twig', ['page' => $page]);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder/list-products', name: 'frontend.top-finder.list-products', methods: ['GET'])]
    public function listProducts(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $productIds = explode('-', $request->get('ids'));

        foreach ($productIds as $key => $id) {
            if (Uuid::isValid((string)$id) === false) {
                unset($productIds[$key]);
            }
        }
        if (!count($productIds)) {
            throw new BadRequestHttpException('ids');
        }

        $page = $this->listProductsLoader->loadPage($request, $salesChanelContext, $productIds);

        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list-products.html.twig', ['page' => $page]);
    }

    #[Route('/top-finder/popup-brand/{brandCode}', name: 'frontend.top-finder.popup_brand', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function listPopupBrand(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $brandCode = $request->get('brandCode');
        if (!preg_match('/^[a-z0-9_-]+$/', $brandCode)) {
            throw new BadRequestHttpException('brandCode'/*, '/brandCode'*/);
        }

        $page = $this->oneBrandLetterPageLoader->loadJson(
            $request,
            $salesChanelContext,
            $brandCode,
            'all'
        );

        $offset = $request->get('offset');
        $defaultOffset = self::POPUP_DEVICES_PRELOAD_LENGTH;
        if ($offset) {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/_listed-devices.html.twig',
                    ['page' => $page, 'offset' => $offset]
                );
        } else {
            $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-devices.html.twig',
                    [
                        'page'     => $page,
                        'mode'     => 'all',
                        'listMode' => $listMode,
                        'offset'   => $defaultOffset,
                    ]
                );
            $response->headers->set('returnPath', ControllerUtil::getReturnPath($request));
            $response->headers->set('loadOffset', (string)$defaultOffset);
        }

        //        $response = new JsonResponse($result);
        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder/popup-brand-series/{brandCode}', name: 'frontend.top-finder.popup_brand_series', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function listPopupBrandSeries(Request $request, SalesChannelContext $salesChanelContext)
    {
        $brandCode = $request->get('brandCode');
        if (!preg_match('/^[a-z0-9_-]+$/', $brandCode)) {
            throw new BadRequestHttpException('brandCode'/*, '/brandCode'*/);
        }

        $page = $this->oneBrandLetterPageLoader->loadJson(
            $request,
            $salesChanelContext,
            $brandCode,
            'series'
        );

        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);
        $offset = $request->get('offset');
        $defaultOffset = self::POPUP_DEVICES_PRELOAD_LENGTH;
        if ($offset) {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/_listed-series-devices.html.twig',
                    ['page' => $page, 'offset' => $offset, 'hideMenuSeries' => true]
                );
        } else {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-devices.html.twig',
                    [
                        'page'           => $page,
                        'mode'           => 'series',
                        'listMode'       => $listMode,
                        'offset'         => $defaultOffset,
                        'hideMenuSeries' => true,
                    ]
                );
            $response->headers->set('returnPath', ControllerUtil::getReturnPath($request));
            $response->headers->set('loadOffset', (string)$defaultOffset);
        }

        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder/popup-brand-types/{brandCode}', name: 'frontend.top-finder.popup_brand_types', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function listPopupBrandTypes(Request $request, SalesChannelContext $salesChanelContext)
    {
        $brandCode = $request->get('brandCode');
        if (!preg_match('/^[a-z0-9_-]+$/', $brandCode)) {
            throw new BadRequestHttpException('brandCode'/*, '/brandCode'*/);
        }

        $page = $this->oneBrandLetterPageLoader->loadJson(
            $request,
            $salesChanelContext,
            $brandCode,
            'types'
        );

        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);
        $offset = $request->get('offset');
        $defaultOffset = self::POPUP_DEVICES_PRELOAD_LENGTH;
        if ($offset) {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/_listed-types-devices.html.twig',
                    ['page' => $page, 'offset' => $offset]
                );
        } else {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-devices.html.twig',
                    [
                        'page'     => $page,
                        'mode'     => 'types',
                        'listMode' => $listMode,
                        'offset'   => $defaultOffset,
                    ]
                );

            $response->headers->set('returnPath', ControllerUtil::getReturnPath($request));
            $response->headers->set('loadOffset', (string)$defaultOffset);
        }

        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }

    #[Route('/top-finder/popup-search', name: 'frontend.top-finder.popup_search', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function listPopupSearch(Request $request, SalesChannelContext $salesChanelContext)
    {
        $page = $this->listPageLoader->loadSearchDevicesJson($request, $salesChanelContext);
        $listMode = $request->cookies->get(CookieKeyNameConstants::POPUP_DEVICELIST_MODE);
        $offset = $request->query->get('offset');
        $defaultOffset = self::POPUP_DEVICES_PRELOAD_LENGTH;

        if ($offset) {
            $response = $this->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/_listed-devices.html.twig',
                ['page' => $page]
            );
        } else {
            $response = $this->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devices.html.twig',
                ['page' => $page, 'listMode' => $listMode]
            );
            $response->headers->set('loadOffset', (string)$defaultOffset);
        }

        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');
        $response->headers->set('returnPath', ControllerUtil::getReturnPath($request));

        return $response;
    }

    #[Route('/top-finder/popup-search-suggest', name: 'frontend.top-finder.popup_search_suggest', methods: ['GET'], defaults: ['XmlHttpRequest' => true])]
    public function suggestPopupSearch(Request $request, SalesChannelContext $salesChanelContext)
    {
        $page = $this->listPageLoader->loadSearchDevicesJson($request, $salesChanelContext);

        $response = $this->renderStorefront(
            '@TopdataTopFinderProSW6/storefront/widget/popup-form/suggest-devices.html.twig',
            ['page' => $page]
        );

        $response->headers->set('X-Robots-Tag', 'noindex, nofollow');

        return $response;
    }
}
