<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Service;

use Doctrine\DBAL\Connection;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Symfony\Component\HttpFoundation\Request;
use Topdata\TopdataTopFinderProSW6\Constants\CookieKeyNameConstants;
use Topdata\TopdataTopFinderProSW6\Util\UtilSearchTerm;
use Topdata\TopdataTopFinderProSW6\Util\UtilUuid;

/**
 * Helper class for Topdata TopFinder Pro plugin
 *
 * This class provides various utility methods for handling devices, brands.
 *
 *
 * 10/2024 renamed Helper --> HelperService
 */
class HelperService
{

    public function __construct(
        protected Connection       $connection,
        protected SettingsService  $settingsService,
        protected EntityRepository $topdataDeviceRepository
    )
    {
    }

    /**
     * 10/2024 moved from TopFinderController::getDevicesArray to HelperService::getFilteredDevices
     *
     * private helper
     * Get topdata devices filtered by brand, series and type
     */
    public function getFilteredDevices($brandId, $seriesId, $typeId, bool $codeAsKey = true): array
    {
        $return = [];
        $params = [];

        $params[] = 'is_enabled = 1';

        if (UtilUuid::isValidUuid($brandId)) {
            $params[] = 'brand_id = 0x' . $brandId;
        } else {
            return $return;
        }

        if ($seriesId == -1) {
            $params[] = 'series_id is null';
        } elseif (UtilUuid::isValidUuid($seriesId)) {
            $params[] = 'series_id = 0x' . $seriesId;
        }

        if ($typeId == -1) {
            $params[] = 'type_id is null';
        } elseif (UtilUuid::isValidUuid($typeId)) {
            $params[] = 'type_id = 0x' . $typeId;
        }

        $devices = $this->connection->fetchAllAssociative(
            'SELECT code, model FROM `topdata_device`'
            . ' WHERE (' . implode(') AND (', $params) . ')'
            . ' ORDER BY model ASC'
        );

        if ($codeAsKey) {
            foreach ($devices as $device) {
                $return[$device['code']] = $device['model'];
            }
        } else {
            $return = $devices;
        }

        return $return;
    }

    /**
     * Get the device history from the request cookies
     *
     * @param Request $request The current request
     * @param Context $context The current context
     * @return array An array of device entities from the history
     */
    public function getDeviceHistory(Request $request, Context $context): array
    {
        $cokie = $request->cookies->get(CookieKeyNameConstants::DEVICE_HISTORY);

        $devicesHistory = [];

        if ($cokie) {
            $devicesHistory = explode(',', $cokie);
        }

        if (!is_array($devicesHistory)) {
            $devicesHistory = [];
        }

        if (count($devicesHistory)) {
            $criteria = (new Criteria(array_reverse($devicesHistory)))
                ->addFilter(new EqualsFilter('enabled', true))
                ->addAssociations(['brand', 'media']);
            $devicesHistory = $this->topdataDeviceRepository->search($criteria, $context)->getElements();
        }

        return $devicesHistory;
    }

    /**
     * [['id'=>hexid, 'name'=>brandname], ...]
     */
    /**
     * Get all enabled brands
     *
     * @return array An array of brand data, each containing 'id' and 'name'
     */
    public function getBrands(): array
    {
        return $this->connection->createQueryBuilder()
            ->select('LOWER(HEX(id)) as id, label as name')
            ->from('topdata_brand')
            ->where('is_enabled = 1')
            ->orderBy('sort DESC, label')
            ->execute()
            ->fetchAllAssociative();
    }

    /**
     * Find devices based on a search term
     *
     * @param string $term The search term
     * @param Connection $connection Database connection
     * @param int $limit Maximum number of results to return (0 for no limit)
     * @return array An array of matching devices
     */
    public static function findDevices(string $term, Connection $connection, int $limit = 0): array
    {
        $filteredTerm = UtilSearchTerm::filterTerm($term);
        if ($filteredTerm !== $term) {
            $term = $filteredTerm;
        }

        if (!$term) {
            return [];
        }

        $SQLadd = ($limit > 0) ? " LIMIT $limit" : '';

        $terms = UtilSearchTerm::getTermsFromString($term);

        $WhereAdd = '';
        if (count($terms)) {
            $WhereAdd = 'OR(';
            foreach ($terms as $key => $value) {
                $terms[$key] = '(topdata_device.keywords LIKE "%' . $value . '%")';
            }
            $WhereAdd .= implode('AND', $terms);
            $WhereAdd .= ')';
        }

        $devices = $connection->fetchAllAssociative('
SELECT topdata_device.model as name, 
       LOWER(HEX(topdata_device.id)) as id, 
       LOWER(HEX(topdata_device.brand_id)) as brand_id, 
       LOWER(HEX(topdata_device.series_id)) as series_id, 
       LOWER(HEX(topdata_device.type_id)) as type_id, 
       topdata_device.code as code, 
       topdata_brand.label as brand_name,
       topdata_brand.code as brand_code,
       topdata_series.code as series_code,
       LOWER(HEX(topdata_device.media_id)) as media_id,
       topdata_series.label as series_name,
       topdata_device_type.label as type_name
  FROM topdata_device 
  LEFT JOIN topdata_brand ON (topdata_brand.id=topdata_device.brand_id)
  LEFT JOIN topdata_series ON (topdata_series.id=topdata_device.series_id)
  LEFT JOIN topdata_device_type ON (topdata_device_type.id=topdata_device.type_id)
  WHERE (topdata_device.is_enabled=1)AND((topdata_device.keywords LIKE "%' . $term . '%")' . $WhereAdd . ')
        ORDER BY `topdata_device`.`code` ASC ' . $SQLadd);

        return count($devices) ? $devices : [];
    }

    /**
     * Count the number of devices matching a search term
     *
     * @param string $term The search term
     * @param Connection $connection Database connection
     * @return int The number of matching devices
     */
    public static function countFindDevices(string $term, Connection $connection): int
    {
        $filteredTerm = UtilSearchTerm::filterTerm($term);
        if ($filteredTerm !== $term) {
            $term = $filteredTerm;
        }

        if (!$term) {
            return 0;
        }

        $terms = UtilSearchTerm::getTermsFromString($term);

        $WhereAdd = '';
        if (count($terms)) {
            $WhereAdd = 'OR(';
            foreach ($terms as $key => $value) {
                $terms[$key] = '(topdata_device.keywords LIKE "%' . $value . '%")';
            }
            $WhereAdd .= implode('AND', $terms);
            $WhereAdd .= ')';
        }

        $devicesCount = $connection->fetchAllAssociative('
SELECT COUNT(*) as cnt
  FROM topdata_device 
  WHERE (topdata_device.is_enabled=1)AND((topdata_device.keywords LIKE "%' . $term . '%")' . $WhereAdd . ')
            ');

        return isset($devicesCount[0]['cnt']) ? (int)$devicesCount[0]['cnt'] : 0;
    }
}
