<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Controller;

use Doctrine\DBAL\Connection;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Validation\DataBag\RequestDataBag;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Controller\StorefrontController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Topdata\TopdataConnectorSW6\Core\Content\Device\Agregate\DeviceCustomer\DeviceCustomerEntity;
use Topdata\TopdataConnectorSW6\Core\Content\Device\DeviceEntity;
use Topdata\TopdataTopFinderProSW6\Service\DeviceToCustomerService;
use Topdata\TopdataTopFinderProSW6\Util\UtilUuid;

#[Route(defaults: ['_routeScope' => ['storefront']])]
class DeviceToCustomerApiController extends StorefrontController
{
    public function __construct(
        private readonly EntityRepository $topdataDeviceRepository,
        private readonly EntityRepository $topdataDeviceToCustomerRepository,
        private readonly Connection $connection,
        private readonly DeviceToCustomerService $deviceToCustomerService,
    ) {
    }

    /**
     * Add device to customer's personal Device List.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response indicating success or failure
     */
    #[Route('/top-finder-api/add-device/{deviceId}', name: 'frontend.top-finder-api.add-device', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function addDevice(Request $request, SalesChannelContext $context): JsonResponse
    {
        $deviceId    = $request->get('deviceId');
        $returnError = new JsonResponse([
            'success' => false,
            'info'    => 'Something went wrong',
        ]);
        $returnSuccess = new JsonResponse([
            'success'  => true,
            'deviceId' => $deviceId,
        ]);

        if (!UtilUuid::isValidUuid($deviceId)) {
            return $returnError;
        }

        if (!$context->getCustomer() || $context->getCustomer()->getGuest()) {
            return $returnError;
        }

        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('id', $deviceId))
            ->addFilter(new EqualsFilter('enabled', true))
            ->setLimit(1);
        /** @var DeviceEntity $device */
        $device = $this->topdataDeviceRepository->search($criteria, $context->getContext())->getEntities()->first();

        if ($device === null) {
            return $returnError;
        }

        if ($this->deviceToCustomerService->deviceToCustomerId($deviceId, $context->getCustomer()->getId())) {
            return $returnSuccess;
        }

        $this->topdataDeviceToCustomerRepository->create([
            [
                'deviceId'   => $deviceId,
                'customerId' => $context->getCustomer()->getId(),
                'extraInfo'  => json_encode(DeviceCustomerEntity::defaultExtraInfo()),
            ],
        ], $context->getContext());

        if ($this->deviceToCustomerService->deviceToCustomerId($deviceId, $context->getCustomer()->getId())) {
            return $returnSuccess;
        } else {
            return $returnError;
        }
    }

    /**
     * Remove device from customer's personal Device List.
     *
     * @param Request $request The current request
     * @param RequestDataBag $data The request data bag
     * @param SalesChannelContext $context The sales channel context
     * @return JsonResponse JSON response indicating success or failure
     */
    #[Route('/top-finder-api/remove-device/{deviceId}', name: 'frontend.top-finder-api.remove-device', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function removeDevice(Request $request, RequestDataBag $data, SalesChannelContext $context): JsonResponse
    {
        $deviceId    = $request->get('deviceId');
        $returnError = new JsonResponse([
            'success' => false,
            'info'    => 'Something went wrong',
        ]);
        $returnSuccess = new JsonResponse([
            'success'  => true,
            'deviceId' => $deviceId,
        ]);

        if (!UtilUuid::isValidUuid($deviceId)) {
            return $returnError;
        }

        if (!$context->getCustomer() || $context->getCustomer()->getGuest()) {
            return $returnError;
        }

        $criteria = (new Criteria())
            ->addFilter(new EqualsFilter('id', $deviceId))
            ->setLimit(1);
        /** @var DeviceEntity $device */
        $device = $this->topdataDeviceRepository->search($criteria, $context->getContext())->getEntities()->first();

        if ($device === null) {
            return $returnError;
        }

        $this->connection->createQueryBuilder()
            ->delete('topdata_device_to_customer')
            ->where('(device_id=0x' . $deviceId . ') AND (customer_id = 0x' . $context->getCustomer()->getId() . ')')
            ->execute();

        return $returnSuccess;
    }

    /**
     * Update device information
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return JsonResponse JSON response indicating success or failure of the update
     */
    #[Route('/top-finder-api/device-info-update', name: 'frontend.top-finder-api.device-info-update', methods: ['GET', 'POST'], defaults: ['XmlHttpRequest' => true])]
    public function popupDeviceInfoUpdate(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $deviceId = $request->get('deviceId');

        if (!UtilUuid::isValidUuid($deviceId)) {
            return new JsonResponse(['success' => false, $deviceId . ' invalid']);
        }

        if (!$salesChannelContext->getCustomer() || $salesChannelContext->getCustomer()->getGuest()) {
            return new JsonResponse(['success' => false, 'no customer']);
        }

        $userId = $salesChannelContext->getCustomer()->getId();
        $data   = $request->get('formData');

        $deviceToCustomerId = $this->deviceToCustomerService->deviceToCustomerId($deviceId, $userId);

        if (!$deviceToCustomerId) {
            return new JsonResponse(['success' => false, 'no device to customer']);
        }

        $this->topdataDeviceToCustomerRepository->update([
            [
                'id'        => $deviceToCustomerId,
                'extraInfo' => json_encode($data),
            ],
        ], $salesChannelContext->getContext());

        return new JsonResponse([
            'success' => true,
        ]);
    }
}
