<?php declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder;

use Shopware\Core\Content\Category\Exception\CategoryNotFoundException;
use Shopware\Core\Content\Product\SalesChannel\Search\ProductSearchGatewayInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Topdata\TopdataConnectorSW6\Core\Content\Device\DeviceCollection;
use Shopware\Core\Framework\Routing\Exception\MissingRequestParameterException;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\ContainsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Doctrine\DBAL\Connection;
use Topdata\TopdataTopFinderProSW6\Component\SettingsService;
use Shopware\Core\Framework\DataAbstractionLayer\EntityCollection;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Core\Framework\Uuid\Uuid;
use Topdata\TopdataTopFinderProSW6\Component\Helper;
use Topdata\TopdataConnectorSW6\Core\Content\Device\Agregate\DeviceCustomer\DeviceCustomerEntity;
use Topdata\TopdataTopFinderProSW6\Controller\TopFinderController;

class ListPageLoader
{
    const TYPE_BRANDS = 'brands';
    const TYPE_BRAND = 'brand';
    const TYPE_TYPE = 'type';
    const TYPE_SERIES = 'series';
    
    private Translator $translator;
    private Request $_request;
    private ListPage $_page;
    private SalesChannelContext $_salesChannelContext;
    private SettingsService $settings;
    
    /**
     * Stores device ids of current user
     * @var ?array
     */
    private $deviceList = null;
    private GenericPageLoader $genericLoader;
    private EventDispatcherInterface $eventDispatcher;
    private EntityRepository $deviceRepository;
    private EntityRepository $brandRepository;
    private EntityRepository $typeRepository;
    private EntityRepository $seriesRepository;
    private Connection $connection;
    private UrlGeneratorInterface $router;
    private EntityRepository $mediaRepository;

    public function __construct(
        GenericPageLoader $genericLoader,
        EventDispatcherInterface $eventDispatcher,
        EntityRepository $deviceRepository,
        EntityRepository $brandRepository,
        EntityRepository $typeRepository,
        EntityRepository $seriesRepository,
        Connection $connection,
        SettingsService $settings,
        UrlGeneratorInterface $router,
        Translator $translator,
        EntityRepository $mediaRepository
    ) {
        $this->genericLoader = $genericLoader;
        $this->eventDispatcher = $eventDispatcher;
        $this->deviceRepository = $deviceRepository;
        $this->brandRepository = $brandRepository;
        $this->typeRepository = $typeRepository;
        $this->seriesRepository = $seriesRepository;
        $this->connection = $connection;
        $this->settings = $settings;
        $this->router = $router;
        $this->translator = $translator;
        $this->mediaRepository = $mediaRepository;
    }

    /**
     * @throws CategoryNotFoundException
     * @throws InconsistentCriteriaIdsException
     */
    public function load(Request $request, SalesChannelContext $salesChannelContext, string $type = self::TYPE_BRANDS): ListPage
    {
        $this->_salesChannelContext = $salesChannelContext;
        $page = $this->genericLoader->load($request, $salesChannelContext);
        $this->_page = ListPage::createFrom($page);
        $this->_page->setPageType($type);
        $this->_request = $request;
        
        switch ($type) {
            case self::TYPE_SERIES:
                $this->loadSeries();
                break;
            case self::TYPE_TYPE:
                $this->loadType();
                break;
            case self::TYPE_BRAND:
                $this->loadBrand($request);
                break;
            case self::TYPE_BRANDS:
                $this->loadBrands();
                break;
            default:
                /**
                 * @todo throw unknown list
                 */
                break;
        }
        
        if($this->settings->getInt('searchCompactLimit', true)) {
            $this->_page->setCompactModeLimit($this->settings->getInt('searchCompactLimit'));
        }

        return $this->_page;
    }
    
    
    private function getEntityCollection(
        EntityRepository $repository,
        array $params = [], 
        array $sortingFields = [], 
        array $associations = [],
        bool $deleteNullParams = true) : EntityCollection
    {
        $criteria = new Criteria();
        foreach ($params as $key => $value) {
            if(($value === null) && $deleteNullParams) {
                continue;
            }
            $criteria->addFilter(new EqualsFilter($key, $value));
        }
        if($associations) {
            $criteria->addAssociations($associations);
        }
        foreach ($sortingFields as $sortingField) {
            $criteria->addSorting(new FieldSorting($sortingField, FieldSorting::ASCENDING));
        }
        return $repository->search($criteria, $this->_salesChannelContext->getContext())->getEntities();
    }
    
    
    private function loadSeries() : void
    {
        $series = $this->getEntityCollection(
            $this->seriesRepository, 
            [
                'enabled' => true,
                'code' => $this->_request->get('code')
            ], 
            [], 
            ['brand']
        )->first();

        if(!$series) {
        /**
         * @todo throw not found series
         */
        }
        
        $this->_page->model = $series;
        

        $this->_page->setShow(['devices'=>true]);
        
        $devices = $this->getDevicesArray($series->getBrand()->getCode(), $series->getId());
        $this->_page->setDevices( $devices );
        
        
        $this->_page->setTitle($this->translator->trans('topdata-topfinder.SEO.seriesPageTitle', [
            '%brand%' => $series->getBrand()->getName(),
            '%series%' => $series->getLabel()
        ]));
        
        $this->_page->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.seriesMetaTitle', [
            '%brand%' => $series->getBrand()->getName(),
            '%series%' => $series->getLabel()
        ]));
        
        $this->_page->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.seriesMetaDescription', [
            '%brand%' => $series->getBrand()->getName(),
            '%series%' => $series->getLabel()
        ]));
        
        $this->_page->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.seriesMetaRobots'));
        
        $this->_page->addBreadcrumb($this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'), $this->router->generate('frontend.top_finder.brands'));
        $this->_page->addBreadcrumb(
            $this->translator->trans('topdata-topfinder.SEO.brandPageTitle', ['%brand%' => $series->getBrand()->getName()]),
            $this->router->generate('frontend.top_finder.brandq', ['code'=>$series->getBrand()->getCode()])
        );
        $this->_page->addBreadcrumb($this->_page->getTitle(), $this->router->generate('frontend.top_finder.series', ['code'=>$series->getCode()]));
    }
    
    
    private function loadType() : void
    {
        $type = $this->getEntityCollection(
            $this->typeRepository, 
            [
                'enabled' => true,
                'code' => $this->_request->get('code')
            ], 
            [], 
            ['brand']
        )->first();

        if(!$type) {
        /**
         * @todo throw not found
         */
        }
        
        $this->_page->model = $type;
        
        
        $this->_page->setShow(['devices'=>true]);
        $devices = $this->getDevicesArray($type->getBrand()->getCode(), 0, $type->getId());
        $this->_page->setDevices( $devices );
        
        
        $this->_page->setTitle($this->translator->trans('topdata-topfinder.SEO.typePageTitle', [
            '%brand%' => $type->getBrand()->getName(),
            '%type%' => $type->getLabel()
        ]));
        
        $this->_page->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.typeMetaTitle', [
            '%brand%' => $type->getBrand()->getName(),
            '%type%' => $type->getLabel()
        ]));
        
        $this->_page->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.typeMetaDescription', [
            '%brand%' => $type->getBrand()->getName(),
            '%type%' => $type->getLabel()
        ]));
        
        $this->_page->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.typeMetaRobots'));
        
        $this->_page->addBreadcrumb($this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'), $this->router->generate('frontend.top_finder.brands'));
        $this->_page->addBreadcrumb(
            $this->translator->trans('topdata-topfinder.SEO.brandPageTitle', ['%brand%' => $type->getBrand()->getName()]), 
            $this->router->generate('frontend.top_finder.brandq', ['code'=>$type->getBrand()->getCode()])
        );
        $this->_page->addBreadcrumb($this->_page->getTitle(), $this->router->generate('frontend.top_finder.type', ['code'=>$type->getCode()]));
    }
    
    
    private function loadBrand(Request $request) : void
    {
        $brand = $this->getEntityCollection(
            $this->brandRepository, 
            [
                'enabled' => true,
                'code' => $this->_request->get('code')
            ]
        )->first();

        if(!$brand) {
        /**
         * @todo throw not found
         */
        }
        
        $this->_page->model = $brand;
        
        $pageShow = [];
        
        $pageTitle = $brand->getName();
        
        if($this->settings->getString('selectboxesMode') == 'both') {
            $switch = $request->cookies->get(Helper::COOKIE_FINDER_SWITCH);
            if($switch == 'types') {
                $pageShow['types'] = true;
                $types = $this->getEntityCollection(
                    $this->typeRepository, 
                    ['enabled' => true, 'brandId' => $brand->getId()], 
                    ['label']
                );
                $this->_page->setTypes($types);

                $devicesTypeNull = $this->getDevicesArray($brand->getCode(), 0, null);
                if(count($devicesTypeNull)) {
                    $pageShow['devices'] = true;
                    $this->_page->setDevices($devicesTypeNull);
                }
            }
            else {
                $pageShow['series'] = true;
                $series = $this->getEntityCollection(
                    $this->seriesRepository, 
                    ['enabled' => true, 'brandId' => $brand->getId()], 
                    ['label']
                );
                $this->_page->setSeries($series);

                $devicesSeriesNull = $this->getDevicesArray($brand->getCode(), null);

                if(count($devicesSeriesNull)) {
                    $pageShow['devices'] = true;
                    $this->_page->setDevices($devicesSeriesNull);
                }
            }
        } 
        elseif($this->settings->getBool('showSeries')) {
            $pageShow['series'] = true;
            $series = $this->getEntityCollection(
                $this->seriesRepository, 
                ['enabled' => true, 'brandId' => $brand->getId()], 
                ['label']
            );
            $this->_page->setSeries($series);
            
            $devicesSeriesNull = $this->getDevicesArray($brand->getCode(), null);
            
            if(count($devicesSeriesNull)) {
                $pageShow['devices'] = true;
                $this->_page->setDevices($devicesSeriesNull);
            }
        } 
        elseif($this->settings->getBool('showTypes')) {
            $pageShow['types'] = true;
            $types = $this->getEntityCollection(
                $this->typeRepository, 
                ['enabled' => true, 'brandId' => $brand->getId()], 
                ['label']
            );
            $this->_page->setTypes($types);
            
            $devicesTypeNull = $this->getDevicesArray($brand->getCode(), 0, null);
            if(count($devicesTypeNull)) {
                $pageShow['devices'] = true;
                $this->_page->setDevices($devicesTypeNull);
            }
        }
        
        if(!$this->settings->getBool('showSeries') && !$this->settings->getBool('showTypes')) {
            $pageShow['devices'] = true;
            $devices = $this->getDevicesArray($brand->getCode());
            $this->_page->setDevices( $devices );
//            $pageTitle .= ' ' . $this->translator->trans('topdata-topfinder.devices');
        }
        
        $this->_page->setShow($pageShow);
        
        
        $this->_page->setTitle($this->translator->trans('topdata-topfinder.SEO.brandPageTitle', [
            '%brand%' => $brand->getName()
        ]));
        
        $this->_page->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.brandMetaTitle', [
            '%brand%' => $brand->getName()
        ]));
        
        $this->_page->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.brandMetaDescription', [
            '%brand%' => $brand->getName()
        ]));
        
        $this->_page->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.brandMetaRobots'));
        
        $this->_page->addBreadcrumb($this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'), $this->router->generate('frontend.top_finder.brands'));
        $this->_page->addBreadcrumb($this->_page->getTitle(), $this->router->generate('frontend.top_finder.brandq', ['code'=>$brand->getCode()]));
    }
    
    
    private function loadBrands() : void
    {
        $this->_page->setShow(['brands'=>true]);
        
        $brands = $this->connection->fetchAllAssociative(
            'SELECT code, label as name, sort FROM `topdata_brand`'
            .' WHERE is_enabled = 1'
            .' ORDER BY sort DESC, name ASC');
        
        $this->_page->setBrands($brands);
        
        $this->_page->setTitle($this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'));
        $this->_page->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.brandsMetaTitle'));
        $this->_page->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.brandsMetaDescription'));
        $this->_page->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.brandsMetaRobots'));
        
        $this->_page->addBreadcrumb(
            $this->_page->getTitle(),
            $this->router->generate('frontend.top_finder.brands')
        );
    }
    
    
    
    private function getDevices(array $params, bool $deleteNullParams = true) : EntityCollection
    {
        $devices = $this->getEntityCollection(
            $this->deviceRepository, 
            array_merge(['enabled' => true], $params), 
            ['brand.name', 'series.label', 'model'], 
            ['media','brand','type','series'],
            $deleteNullParams
        );

        foreach ($devices as $device) {
            $device->setInDeviceList(in_array($device->getId(), $this->getDeviceList()));
        }
        
        return $devices;
    }
    
    private function getDevicesArray($brandCode, $seriesId = 0, $typeId = 0)
    {
        $conditions = [];
        $conditions[] = '(topdata_device.is_enabled=1)';
        $conditions[] = '(topdata_brand.code = "'.$brandCode.'")';
        
        if($seriesId === null) {
            $conditions[] = '(topdata_device.series_id IS NULL)';
        }
        elseif(Uuid::isValid((string)$seriesId)) {
            $conditions[] = '(topdata_device.series_id = 0x'.$seriesId.')';
        }
        
        if($typeId === null) {
            $conditions[] = '(topdata_device.type_id IS NULL)';
        }
        elseif(Uuid::isValid((string)$typeId)) {
            $conditions[] = '(topdata_device.type_id = 0x'.$typeId.')';
        }
        
        $devices = $this->connection->fetchAllAssociative('
SELECT topdata_device.model as name, 
       LOWER(HEX(topdata_device.id)) as id, 
       LOWER(HEX(topdata_device.brand_id)) as brand_id, 
       LOWER(HEX(topdata_device.series_id)) as series_id, 
       LOWER(HEX(topdata_device.type_id)) as type_id, 
       topdata_device.code as code, 
       topdata_brand.label as brand_name,
       topdata_brand.code as brand_code,
       topdata_series.code as series_code,
       LOWER(HEX(topdata_device.media_id)) as media_id,
       topdata_series.label as series_name,
       topdata_device_type.label as type_name
  FROM topdata_device 
  LEFT JOIN topdata_brand ON (topdata_brand.id=topdata_device.brand_id)
  LEFT JOIN topdata_series ON (topdata_series.id=topdata_device.series_id)
  LEFT JOIN topdata_device_type ON (topdata_device_type.id=topdata_device.type_id)
  WHERE '.implode('AND', $conditions).'
  ORDER BY topdata_device.code
            ');
        
        return $devices;
    }
    
    public function loadJsonBrands(Request $request, SalesChannelContext $salesChannelContext): ListPage
    {
        $this->_salesChannelContext = $salesChannelContext;
        $page = $this->genericLoader->load($request, $salesChannelContext);
        $this->_page = ListPage::createFrom($page);
        $this->_page->setPageType(self::TYPE_BRANDS);
        $this->_request = $request;

        $this->_page->setShow(['brands'=>true]);
        
        $brands = $this->connection->fetchAllAssociative(
            'SELECT code, label as name, sort FROM `topdata_brand`'
            .' WHERE is_enabled = 1'
            .' ORDER BY sort DESC, name ASC');
        
        $this->_page->setBrands($brands);
        
        $this->_page->setTitle($this->translator->trans('topdata-topfinder.popup.allBrands'));
        
        $this->_page->popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.allBrands'),
//            'path' => $this->router->generate('frontend.top_finder.popup_all_brands')
        ];

        return $this->_page;
    }
    
    public function loadBrandDevicesJson(Request $request, SalesChannelContext $salesChannelContext): ListPage
    {
        $this->_salesChannelContext = $salesChannelContext;
        $page = $this->genericLoader->load($request, $salesChannelContext);
        $this->_page = ListPage::createFrom($page);
        $this->_request = $request;
        
        $brand = $this->getEntityCollection(
            $this->brandRepository, 
            [
                'enabled' => true,
                'code' => $this->_request->get('code')
            ]
        )->first();
        
        if(!$brand) {
            return $this->_page;
        }
        
        $this->_page->setTitle($this->translator->trans('topdata-topfinder.SEO.brandPageTitle', [
            '%brand%' => '<b>' . $brand->getName() . '</b>' 
        ]));
        
        $devices = $this->getDevicesArray($brand->getCode());
        $this->loadDevicesContent($devices, $salesChannelContext, 'brand');
        $this->_page->setDevices( $devices );

        return $this->_page;
    }
    
    
    public function loadSerieDevicesJson(Request $request, SalesChannelContext $salesChannelContext): ListPage
    {
        $this->_salesChannelContext = $salesChannelContext;
        $page = $this->genericLoader->load($request, $salesChannelContext);
        $this->_page = ListPage::createFrom($page);
        $this->_request = $request;

        $series = $this->getEntityCollection(
            $this->seriesRepository, 
            [
                'enabled' => true,
                'code' => $this->_request->get('code')
            ], 
            [], 
            ['brand']
        )->first();

        if(!$series) {
            return $this->_page;
        }
        
        $this->_page->setTitle($this->translator->trans('topdata-topfinder.SEO.seriesPageTitle', [
            '%brand%' => '<b>' . $series->getBrand()->getName() . '</b>',
            '%series%' => $series->getLabel()
        ]));
        
        $devices = $this->getDevicesArray($series->getBrand()->getCode(), $series->getId());
        $this->loadDevicesContent($devices, $salesChannelContext, 'series');
        $this->_page->setDevices( $devices );
        
        $letter = ($series->getBrand()->getCode())[0];
        if(preg_match('/^[0-9]{1}$/', $letter)) {
            $letter = '0';
            $letterStr = '0-9';
        }
        else {
            $letterStr = strtoupper($letter);
        }
        
        $popupPath = [];
        
        $popupPath[] = [
            'name' => $letterStr,
            'path' => $this->router->generate('frontend.top_finder.popup_letter', ['letter'=>$letter])
        ];
        
        $popupPath[] = [
            'name' => $series->getBrand()->getName(),
            'path' => $this->router->generate('frontend.top_finder.popup_brand', ['brandCode'=>$series->getBrand()->getCode()])
        ];
        
        $popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.brandSeries'),
            'path' => $this->router->generate('frontend.top_finder.popup_brand_series', ['brandCode'=>$series->getBrand()->getCode()])
        ];
        
        $popupPath[] = [
            'name' => $series->getLabel()
        ];
        
        $this->_page->popupPath = $popupPath;

        return $this->_page;
    }
    
    public function loadTypeDevicesJson(Request $request, SalesChannelContext $salesChannelContext): ListPage
    {
        $this->_salesChannelContext = $salesChannelContext;
        $page = $this->genericLoader->load($request, $salesChannelContext);
        $this->_page = ListPage::createFrom($page);
        $this->_request = $request;
        
        $type = $this->getEntityCollection(
            $this->typeRepository, 
            [
                'enabled' => true,
                'code' => $this->_request->get('code')
            ], 
            [], 
            ['brand']
        )->first();

        if(!$type) {
            return $this->_page;
        }
        
        $this->_page->setTitle($this->translator->trans('topdata-topfinder.SEO.typePageTitle', [
            '%brand%' => '<b>' . $type->getBrand()->getName() . '</b>',
            '%type%' => $type->getLabel()
        ]));
        
        $devices = $this->getDevicesArray($type->getBrand()->getCode(), 0, $type->getId());
        $this->loadDevicesContent($devices, $salesChannelContext, 'types');
        $this->_page->setDevices( $devices );
        
        $letter = ($type->getBrand()->getCode())[0];
        if(preg_match('/^[0-9]{1}$/', $letter)) {
            $letter = '0';
            $letterStr = '0-9';
        }
        else {
            $letterStr = strtoupper($letter);
        }
        
        $popupPath = [];
        
        $popupPath[] = [
            'name' => $letterStr,
            'path' => $this->router->generate('frontend.top_finder.popup_letter', ['letter'=>$letter])
        ];
        
        $popupPath[] = [
            'name' => $type->getBrand()->getName(),
            'path' => $this->router->generate('frontend.top_finder.popup_brand', ['brandCode'=>$type->getBrand()->getCode()])
        ];
        
        $popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.brandTypes'),
            'path' => $this->router->generate('frontend.top_finder.popup_brand_types', ['brandCode'=>$type->getBrand()->getCode()])
        ];
        
        $popupPath[] = [
            'name' => $type->getLabel()
        ];
        
        $this->_page->popupPath = $popupPath;

        return $this->_page;
    }
    
    
    private function getDeviceList(SalesChannelContext $salesChannelContext) : array
    {
        if(!$salesChannelContext->getCustomer() || $salesChannelContext->getCustomer()->getGuest()) {
            $this->deviceList = [];
        }
        elseif(null === $this->deviceList) {
            $this->deviceList = [];
            
            $rez = $this->connection
                ->createQueryBuilder()
                ->select ('LOWER(HEX(device_id)) as device_id, extra_info')
                ->from('topdata_device_to_customer')
                ->where('customer_id = 0x' . $salesChannelContext->getCustomer()->getId() )
                //')AND(is_dealer_managed=0)'
                ->execute()
                ->fetchAllAssociative();
            foreach ($rez as $val) {
                $this->deviceList[$val['device_id']] = $val['extra_info'] ? json_decode($val['extra_info'], true) : DeviceCustomerEntity::defaultExtraInfo();
            }
        }
        
        return $this->deviceList;
    }
    
    
    private function loadDevicesContent(&$devices, SalesChannelContext $salesChannelContext, string $listType='brand', string $searchTerm = '')
    {
        $mediaIds = [];
        $devicelist = $this->getDeviceList($salesChannelContext);
        foreach ($devices as $key=>$device) {
            if($listType=='search' && $searchTerm) {
                $devices[$key]['path'] = $this->router->generate('frontend.top_finder.popup_device_new', [
                    'deviceCode'=>$device['code'],
                    'listType' => $listType,
                    'term' => $searchTerm
                ]);
            }
            else {
                $devices[$key]['path'] = $this->router->generate('frontend.top_finder.popup_device_new', [
                    'deviceCode'=>$device['code'],
                    'listType' => $listType
                ]);
            }
            if($device['media_id']) {
                $mediaIds[] = $device['media_id'];
            }
            
            if(isset($devicelist[$device['id']]) && isset($devicelist[$device['id']]['devices'])) {
                $devices[$key]['devicelist'] = count($devicelist[$device['id']]['devices']);
            }
            else {
                $devices[$key]['devicelist'] = -1;
            }
        }
        
        
        if(count($mediaIds)) {
            $medias = $this->mediaRepository->search(new Criteria($mediaIds), $salesChannelContext->getContext());
            
            foreach ($devices as $key => $device) {
                if($device['media_id'] && $medias->get($device['media_id'])) {
                    $devices[$key]['media_url'] = $medias->get($device['media_id'])->getUrl();
                }
            }
        }
    }
    
    
    public function loadHistoryDevicesJson(Request $request, SalesChannelContext $salesChannelContext): ListPage
    {
        $this->_salesChannelContext = $salesChannelContext;
        $page = $this->genericLoader->load($request, $salesChannelContext);
        $this->_page = ListPage::createFrom($page);
        $this->_request = $request;
        
        $this->_page->setTitle($this->translator->trans('topdata-topfinder.popup.deviceHistoryTitle'));
        
        $popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.deviceHistoryTitle')
        ];
        
        $this->_page->popupPath = $popupPath;
        
        if(!$this->settings->getBool('showDeviceHistory')) {
            return $this->_page;
        }
        
        
        $cokie = $request->cookies->get(Helper::COOKIE_DEVICE_HISTORY);
        $cookieIds = [];
        $deviceIds = [];
        if($cokie) {
            $cookieIds = explode(',', $cokie);
        }
        
        foreach ($cookieIds as $value) {
            if(Uuid::isValid($value)) {
                $deviceIds[] = $value;
            }
        }
        
        if(!$deviceIds) {
            return $this->_page;
        }
        
        
        $devices = $this->connection->fetchAllAssociative('
SELECT topdata_device.model as name, 
       LOWER(HEX(topdata_device.id)) as id, 
       LOWER(HEX(topdata_device.brand_id)) as brand_id, 
       LOWER(HEX(topdata_device.series_id)) as series_id, 
       LOWER(HEX(topdata_device.type_id)) as type_id, 
       topdata_device.code as code, 
       topdata_brand.label as brand_name,
       topdata_brand.code as brand_code,
       topdata_series.code as series_code,
       LOWER(HEX(topdata_device.media_id)) as media_id,
       topdata_series.label as series_name,
       topdata_device_type.label as type_name
  FROM topdata_device 
  LEFT JOIN topdata_brand ON (topdata_brand.id=topdata_device.brand_id)
  LEFT JOIN topdata_series ON (topdata_series.id=topdata_device.series_id)
  LEFT JOIN topdata_device_type ON (topdata_device_type.id=topdata_device.type_id)
  WHERE topdata_device.id IN (0x'.implode(',0x', $deviceIds).')
            ');
        $deviceIds = array_reverse($deviceIds);
        
        $sortedDevices = [];
        foreach ($deviceIds as $id) {
            foreach ($devices as $device) {
                if($id === $device['id']) {
                    $sortedDevices[] = $device;
                }
            }
        }
        
        $this->loadDevicesContent($sortedDevices, $salesChannelContext, 'history');
        $this->_page->setDevices( $sortedDevices );
        

        return $this->_page;
    }
    
    
    public function loadSearchDevicesJson(Request $request, SalesChannelContext $salesChannelContext): ListPage
    {
        $defaultOffset = TopFinderController::POPUP_DEVICES_PRELOAD_LENGTH;
        
        $this->_salesChannelContext = $salesChannelContext;
        $page = $this->genericLoader->load($request, $salesChannelContext);
        $this->_page = ListPage::createFrom($page);
        $this->_request = $request;
        
        $offset = 1*$request->query->get('offset');
        
        $term = $request->query->get('term');
        
        $term = Helper::filterTerm($term);
        
        $this->_page->setTitle($this->translator->trans('topdata-topfinder.popup.deviceSearchTitle', ['%term%'=>$term]));
        $this->_page->searchTerm = $term;
        
        $popupPath[] = [
            'name' => $this->translator->trans('topdata-topfinder.popup.deviceSearchTitle', ['%term%'=>$term])
        ];
        
        $this->_page->popupPath = $popupPath;
        
        if(!$term) {
            return $this->_page;
        }
        
        $this->_page->devicesTotal = Helper::countFindDevices($term, $this->connection);
        
        if($this->_page->devicesTotal) {
            if($offset == $defaultOffset) {
                $devices = Helper::findDevices($term, $this->connection);
                $devices = array_slice($devices, $offset);
            }
            else {
                $devices = Helper::findDevices($term, $this->connection, $defaultOffset);
            }
        }
        
        $this->loadDevicesContent($devices, $salesChannelContext, 'search', $term);
        $this->_page->setDevices( $devices );
        

        return $this->_page;
    }
}
