<?php declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder;

use Shopware\Core\Content\Category\Exception\CategoryNotFoundException;
use Shopware\Core\Content\Product\SalesChannel\Search\ProductSearchGatewayInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Exception\InconsistentCriteriaIdsException;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\DataAbstractionLayer\Search\EntitySearchResult;
use Topdata\TopdataConnectorSW6\Core\Content\Device\DeviceCollection;
use Shopware\Core\Framework\Routing\Exception\MissingRequestParameterException;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\ContainsFilter;
use Topdata\TopdataTopFinderProSW6\Component\SettingsService;

class DeviceSearchPageLoader
{
    const DEVICES_LIMIT = 500;
    
    /**
     * @var SettingsService
     */
    private $settings;
    
    /**
     * @var ?array
     */
    private $deviceList = null;
    
    /**
     * @var GenericPageLoader
     */
    private $genericLoader;

    /**
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;
        
    private $deviceRepository;
    
    private $connection;

    public function __construct(
        GenericPageLoader $genericLoader,
        EventDispatcherInterface $eventDispatcher,
        $deviceRepository,
        $connection,
        SettingsService $settings
    ) {
        $this->genericLoader = $genericLoader;
        $this->eventDispatcher = $eventDispatcher;
        $this->deviceRepository = $deviceRepository;
        $this->connection = $connection;
        $this->settings = $settings;
    }


    public function load(Request $request, SalesChannelContext $salesChannelContext): DeviceSearchPage
    {
        /** @var DeviceSearchPage $page */
        $page = $this->genericLoader->load($request, $salesChannelContext);
        $page = DeviceSearchPage::createFrom($page);
        $page->setDevices($this->searchDevices($request, $salesChannelContext));
        $page->setDeviceSearchTerm($request->query->get('term'));

//        $this->eventDispatcher->dispatch(
//            new DeviceListPageLoadedEvent($page, $salesChannelContext, $request)
//        );
        
        if($this->settings->getInt('searchCompactLimit', true)) {
            $page->setCompactModeLimit($this->settings->getInt('searchCompactLimit'));
        }
        
        $page->offset = (int)$request->query->get('offset');
        $page->limit = self::DEVICES_LIMIT;
        $page->deviceListDevices = $this->getDeviceList($salesChannelContext);
        
        return $page;
    }
    
    
    private function searchDevices($request, SalesChannelContext $context) : EntitySearchResult
    {
        $term = $request->query->get('term');
        $offset = (int)$request->query->get('offset');
        
        if (empty($term)) {
            throw new MissingRequestParameterException('term');
        }

        $search = mb_strtolower(trim($term));
        
        if (empty($search)) {
            throw new MissingRequestParameterException('term');
        }
        
        $criteria = (new Criteria())
            ->addFilter(new EqualsFilter('enabled', true))
            ->addFilter(new ContainsFilter('keywords', $search))
            ->addAssociations(['media','brand','type','series'])
            ->addSorting(new FieldSorting('brand.name', FieldSorting::ASCENDING))
            ->addSorting(new FieldSorting('series.label', FieldSorting::ASCENDING))
            ->addSorting(new FieldSorting('model', FieldSorting::ASCENDING));
        
        $criteria->setTotalCountMode(Criteria::TOTAL_COUNT_MODE_EXACT);
        
        $criteria->setLimit(self::DEVICES_LIMIT);
        if($offset) {
            $criteria->setOffset($offset);
        }

        $devices = $this->deviceRepository->search($criteria, $context->getContext());
        
        foreach ($devices as $device) {
            $device->setInDeviceList(in_array($device->getId(), $this->getDeviceList($context)));
        }
        
        return $devices;
    }
    
    private function getDeviceList(SalesChannelContext $context) : array
    {
        if(null === $this->deviceList) {
            $this->deviceList = [];
            if($context->getCustomer() && !($context->getCustomer()->getGuest())) {
                $connection = $this->connection;
                $rez = $connection->createQueryBuilder()
                    ->select(['device_id'])
                    ->from('topdata_device_to_customer')
                    ->where('customer_id = 0x' . $context->getCustomer()->getId() )
                    ->execute()
                    ->fetchAllAssociative();
                foreach ($rez as $val) {
                    $this->deviceList[] = bin2hex($val['device_id']);
                }
            }
        }
        
        return $this->deviceList;
    }
}
