<?php declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\Page\OneBrandLetter;

use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Doctrine\DBAL\Connection;
use Topdata\TopdataTopFinderProSW6\Component\SettingsService;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\OneBrandLetter\OneBrandLetterPage;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;

class OneBrandLetterPageLoader
{
    /**
     * @var Translator
     */
    private $translator;
    
    /**
     * @var SettingsService
     */
    private $settingsService;
    
    /**
     * @var GenericPageLoader
     */
    private $genericLoader;
    
    /**
     * @var Connection
     */
    private $connection;
    
    /**
     * @var UrlGeneratorInterface
     */
    private $router;
    
    private EntityRepository $mediaRepository;
    private ?array $deviceList = null;

    public function __construct(
        GenericPageLoader $genericLoader,
        Connection $connection,
        SettingsService $settingsService,
        UrlGeneratorInterface $router,
        Translator $translator,
        EntityRepository $mediaRepository
    ) {
        $this->genericLoader = $genericLoader;
        $this->connection = $connection;
        $this->settingsService = $settingsService;
        $this->router = $router;
        $this->translator = $translator;
        $this->mediaRepository = $mediaRepository;
        
    }

    
    public function load(Request $request, SalesChannelContext $salesChannelContext, string $brandCode): OneBrandLetterPage
    {
        $page = $this->genericLoader->load($request, $salesChannelContext);
        /** @var OneBrandLetterPage $page */
        $page = OneBrandLetterPage::createFrom($page);
        $this->setElements($page, $brandCode);
        
        if($request->isXmlHttpRequest() === false) {
            $page->setTitle($this->translator->trans('topdata-topfinder.SEO.brandPageTitle', [
                '%brand%' => $page->brand['name']
            ]));

            $page->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.brandMetaTitle', [
                '%brand%' => $page->brand['name']
            ]));

            $page->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.brandMetaDescription', [
                '%brand%' => $page->brand['name']
            ]));

            $page->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.brandMetaRobots'));



            $page->addBreadcrumb(
                $this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'),
                $this->router->generate('frontend.top_finder.brands')
            );

            $page->addBreadcrumb(
                $page->getTitle(),
                $this->router->generate('frontend.top_finder.brand', ['brandCode'=>$page->brand['code']])
            );
        }
        
        return $page;
    }
    
    private function setElements(OneBrandLetterPage $page, string $brandCode) : void
    {
        $return = [];
        
        $page->brand = $this->getBrand($brandCode);
        if(!$page->brand) {
            throw new \Exception('Device brand not found!');
        }
        
        $devices = $this->connection->fetchAllAssociative('
SELECT topdata_device.model as name, 
       LOWER(HEX(topdata_device.id)) as id, 
       LOWER(HEX(topdata_device.brand_id)) as brand_id, 
       LOWER(HEX(topdata_device.series_id)) as series_id, 
       LOWER(HEX(topdata_device.type_id)) as type_id, 
       topdata_device.code as code, 
       topdata_brand.label as brand_name,
       topdata_brand.code as brand_code,
       LOWER(HEX(topdata_device.media_id)) as media_id
  FROM topdata_brand,
       topdata_device
  WHERE (topdata_brand.id=topdata_device.brand_id)
        AND(topdata_brand.code = "'.$brandCode.'")
        AND(topdata_device.is_enabled=1)
  ORDER BY topdata_device.code
            ');
        
        $current_brand = [
            'brand'=>$devices[0]['brand_name'],
            'brand_id'=>$devices[0]['brand_id'],
            'brand_code'=>$devices[0]['brand_code'],
            'devices'=>[],
            'series'=>[],
            'types'=>[],
        ];
        
        $series = $this->getSeries();
        $types = $this->getTypes();
        
        $uniqueSeries = [];
        $uniqueTypes = [];
        
        foreach ($devices as $device) {
            if($device['series_id'] && isset($series[$device['series_id']])) {
                $device['series_name'] = $series[$device['series_id']]['name'];
                $device['series_code'] = $series[$device['series_id']]['code'];
            }
            else {
                $device['series_name'] = '';
            }
            
            if($device['type_id'] && isset($types[$device['type_id']])) {
                $device['type_name'] = $types[$device['type_id']]['name'];
                $device['type_code'] = $types[$device['type_id']]['code'];
            }
            else {
                $device['type_name'] = '';
            }
            
            if(!in_array($device['series_name'], $uniqueSeries)) {
                $uniqueSeries[] = $device['series_name'];
            }
            
            if(!in_array($device['type_name'], $uniqueTypes)) {
                $uniqueTypes[] = $device['type_name'];
            }
            $return[] = $device;
        }
        
        sort($uniqueSeries, SORT_STRING);
        if($uniqueSeries[0] == '') {
            unset($uniqueSeries[0]);
            $uniqueSeries[] = '';
        }
        sort($uniqueTypes, SORT_STRING);
        if($uniqueTypes[0] == '') {
            unset($uniqueTypes[0]);
            $uniqueTypes[] = '';
        }
        
        
        $page->series = $uniqueSeries;
        $page->types = $uniqueTypes;
        $page->devices = $return;
    }
    
    
    private function getSeries() : array
    {
        $return = [];
        
        $series = $this->connection->fetchAllAssociative('
SELECT label as name, 
       LOWER(HEX(id)) as id, 
       code as code
  FROM topdata_series
  WHERE is_enabled=1
  ORDER BY label
        ');
        
        foreach ($series as $serie) {
            $return[$serie['id']] = $serie;
        }
        
        return $return;
    }
    
    
    private function getTypes() : array
    {
        $return = [];
        
        $types = $this->connection->fetchAllAssociative('
SELECT label as name, 
       LOWER(HEX(id)) as id, 
       code as code
  FROM topdata_device_type
  WHERE is_enabled=1
  ORDER BY label
        ');
        
        foreach ($types as $type) {
            $return[$type['id']] = $type;
        }
        
        return $return;
    }
    
    
    private function getBrand(string $brandCode) : array
    {
        $brand = $this->connection->fetchAllAssociative('
SELECT LOWER(HEX(id)) as id, 
       code, 
       label as name
  FROM topdata_brand
  WHERE (is_enabled=1)
        AND(code = "'.$brandCode.'")
            ');
        
        return isset($brand[0]) ? $brand[0] : [];
    }
    
    public function loadJson(
        Request $request, 
        SalesChannelContext $salesChannelContext, 
        string $brandCode,
        string $displayMode = 'all'
    ): OneBrandLetterPage
    {
        $page = $this->genericLoader->load($request, $salesChannelContext);
        /** @var OneBrandLetterPage $page */
        $page = OneBrandLetterPage::createFrom($page);
        $this->setElements($page, $brandCode);
        
        $page->setTitle($this->translator->trans('topdata-topfinder.SEO.brandPageTitle', [
            '%brand%' => $page->brand['name']
        ]));
        
        $letter = $page->brand['code'][0];
        
        if(preg_match('/^[0-9]{1}$/', $letter)) {
            $letter = '0';
            $letterStr = '0-9';
        }
        else {
            $letterStr = strtoupper($letter);
        }
        
        $page->popupPath[] = [
            'name' => $letterStr,
            'path' => $this->router->generate('frontend.top_finder.popup_letter', ['letter'=>$letter])
        ];
        
        $page->popupPath[] = [
            'name' => $page->brand['name'],
        ];
        
        $listType = 'brand';
        if($displayMode == 'series') {
            usort($page->devices, function($a, $b) {
                if($a['series_name'] === '' && $b['series_name']) {
                    return 1;
                }
                if($a['series_name'] && $b['series_name'] === '') {
                    return -1;
                }
                return $a['series_name'].$a['code'] <=> $b['series_name'].$b['code'];
            });
            
            
            $page->popupPath[] = [
                'name' => $this->translator->trans('topdata-topfinder.popup.brandSeries'),
            ];
            $listType = 'series';
        }
        elseif($displayMode == 'types') {
            usort($page->devices, function($a, $b) {
                if($a['type_name'] === '' && $b['type_name']) {
                    return 1;
                }
                if($a['type_name'] && $b['type_name'] === '') {
                    return -1;
                }
                return $a['type_name'].$a['code'] <=> $b['type_name'].$b['code'];
            });
            $page->popupPath[] = [
                'name' => $this->translator->trans('topdata-topfinder.popup.brandTypes'),
            ];
            $listType = 'types';
        }
        else {
            $page->popupPath[] = [
                'name' => $this->translator->trans('topdata-topfinder.popup.allModels'),
            ];
        }
        
        $mediaIds = [];
        $devicelist = $this->getDeviceList($salesChannelContext);
        foreach ($page->devices as $key => $device) {
            $page->devices[$key]['path'] = $this->router->generate('frontend.top_finder.popup_device_new', [
                'deviceCode'=>$device['code'],
                'listType'=>$listType
            ]);
            if($device['media_id']) {
                $mediaIds[] = $device['media_id'];
            }
            
            if(isset($devicelist[$device['id']]) && isset($devicelist[$device['id']]['devices'])) {
                $page->devices[$key]['devicelist'] = count($devicelist[$device['id']]['devices']);
            }
            else {
                $page->devices[$key]['devicelist'] = -1;
            }
        }
        
        if(count($mediaIds)) {
            $medias = $this->mediaRepository->search(new Criteria($mediaIds), $salesChannelContext->getContext());
            
            foreach ($page->devices as $key => $device) {
                if($device['media_id'] && $medias->get($device['media_id'])) {
                    $page->devices[$key]['media_url'] = $medias->get($device['media_id'])->getUrl();
                }
            }
        }
        
        return $page;
    }
    
    
    private function getDeviceList(SalesChannelContext $salesChannelContext) : array
    {
        if(!$salesChannelContext->getCustomer() || $salesChannelContext->getCustomer()->getGuest()) {
            $this->deviceList = [];
        }
        elseif(null === $this->deviceList) {
            $this->deviceList = [];
            
            $rez = $this->connection
                ->createQueryBuilder()
                ->select ('LOWER(HEX(device_id)) as device_id, extra_info')
                ->from('topdata_device_to_customer')
                ->where('customer_id = 0x' . $salesChannelContext->getCustomer()->getId() )
                //')AND(is_dealer_managed=0)'
                ->execute()
                ->fetchAllAssociative();
            foreach ($rez as $val) {
                $this->deviceList[$val['device_id']] = $val['extra_info'] ? json_decode($val['extra_info'], true) : DeviceCustomerEntity::defaultExtraInfo();
            }
        }
        
        return $this->deviceList;
    }
}