<?php declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\Page\Device;

use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Doctrine\DBAL\Connection;
use Topdata\TopdataTopFinderProSW6\Component\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\ListProducts\ListProducts;

use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;

use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Topdata\TopdataConnectorSW6\Core\Content\Device\DeviceEntity;
use Shopware\Core\Content\Product\SalesChannel\Listing\ProductListingLoader;
use Shopware\Core\Content\Product\SalesChannel\ProductAvailableFilter;
use Shopware\Core\Content\Product\Aggregate\ProductVisibility\ProductVisibilityDefinition;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Shopware\Core\Content\Product\SalesChannel\Listing\ProductListingResult;
use Shopware\Core\Content\Product\Events\ProductListingResultEvent;
use Shopware\Core\Content\Product\Events\ProductListingCriteriaEvent;
use Topdata\TopdataConnectorSW6\Component\Helper\EntitiesHelper;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Struct\ArrayStruct;
use Topdata\TopdataTopFinderProSW6\Content\Product\SalesChannel\DeviceListing\ProductDeviceListingLoader;

class DevicePageLoader
{
    /**
     * @var GenericPageLoader
     */
    private $genericLoader;
    
    /**
     * @var EntityRepository
     */
    private $deviceRepository;
    
    /**
     * @var ProductDeviceListingLoader
     */
    private $listingLoader;

    /**
     * @var EventDispatcher
     */
    private $eventDispatcher;
    
    /**
     * @var EntitiesHelper
     */
    private $entitiesHelper;
    
    /**
     * @var Translator
     */
    private $translator;
    
    /**
     * @var UrlGeneratorInterface
     */
    private $router;
    
    private SettingsService $settings;
    private Connection $connection;
    private EntityRepository $propertyGroupOptionRepository;

    public function __construct(
        GenericPageLoader $genericLoader,
        EntityRepository $deviceRepository,
//        ProductDeviceListingLoader 
        $listingLoader,
        EventDispatcherInterface $eventDispatcher,
        EntitiesHelper $entitiesHelper,
        Translator $translator,
        UrlGeneratorInterface $router,
        SettingsService $settings,
        Connection $connection,
        EntityRepository $propertyGroupOptionRepository
    ) {
        $this->genericLoader = $genericLoader;
        $this->deviceRepository = $deviceRepository;
        $this->listingLoader = $listingLoader;
        $this->eventDispatcher = $eventDispatcher;
        $this->entitiesHelper = $entitiesHelper;
        $this->translator = $translator;
        $this->router = $router;
        $this->settings = $settings;
        $this->connection = $connection;
        $this->propertyGroupOptionRepository = $propertyGroupOptionRepository;
    }
    
    private function getProductsChildIds(array $productIds): array
    {
        $ids = [];
        if(!count($productIds)) {
            return $ids;
        }
        
        $productIds = '0x'.implode(',0x', $productIds);
        
        $rez = $this->connection->fetchAllAssociative("
 SELECT LOWER(HEX(id)) as id
  FROM `product`
  WHERE parent_id IN ($productIds)
            ");
        
        foreach ($rez as $item) {
            $ids[] = $item['id'];
        }
        
        return $ids;
    }

    public function load(Request $request, SalesChannelContext $context, bool $partialLoad = false): DevicePage
    {
        /** @var DevicePage $page */
        $page = DevicePage::createFrom(
                $this->genericLoader->load($request, $context)
            );
        
        /** @var ?DeviceEntity $device */
        $device = $this->loadDevice($request, $context);
        $page->device = $device;
        
        $this->loadTabs($page, $request, $context);
        
        $this->setActiveTab($page, $request, $context);
        
        if($request->isXmlHttpRequest() === false) {
            $this->loadDeviceSynonims($page, $context);
        }
        
        $productIds = $device->getProducts()->getIds();
        $productWithChildsIds = [];
        foreach ($device->getProducts() as $prod) {
            if($prod->getParentId()) {
                $productIds[] = $prod->getParentId();
            }
            elseif($prod->getChildCount()) {
                $productWithChildsIds[] = $prod->getId();
            }
        }
        
        $productIds = array_merge(
            $productIds, 
            $this->getProductsChildIds($productWithChildsIds)
        );
        
        $productIds = array_unique($productIds);
        
        $criteria = new Criteria($productIds);
        $criteria->setTitle('cms::product-listing');
        $this->eventDispatcher->dispatch(
            new ProductListingCriteriaEvent($request, $criteria, $context)
        );
        
        $criteria->addFilter(
            new ProductAvailableFilter($context->getSalesChannel()->getId(), ProductVisibilityDefinition::VISIBILITY_ALL)
        );
        
        if($page->activeTabId) {
            $criteria->addFilter(new EqualsFilter('properties.id', $page->activeTabId));
        }
        
        $page->listing = ProductListingResult::createFrom(
                $this->listingLoader->load($criteria, $context)
            );

        $this->eventDispatcher->dispatch(
            new ProductListingResultEvent($request, $page->listing, $context)
        );
        
        if($request->isXmlHttpRequest() === false) {
            $this->setSeoInformation($page);
        }
        
        return $page;
    }
    
    private function loadDevice(Request $request, SalesChannelContext $context) : ?DeviceEntity
    {
        $deviceCode = $request->get('deviceCode');
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('enabled', true));
        $criteria->addFilter(new EqualsFilter('code', $deviceCode));
        $criteria->addAssociations(['media', 'products', 'brand','type','series']);
        
        return $this->deviceRepository->search($criteria, $context->getContext())->getEntities()->first();
    }
    
    private function setActiveTab(DevicePage $page, Request $request, SalesChannelContext $context)
    {
//        if($page->tabs && $page->tabs->first()) {
//            $page->activeTabId = $page->tabs->first()->getId();
//        }
//        else {
//            return;
//        }
        
        $propertyGroupOptionId = $request->get('propertyGroupOptionId', '');
        if(!Uuid::isValid($propertyGroupOptionId)) {
            return;
        }
        
        if($page->tabs->get($propertyGroupOptionId)) {
            $page->activeTabId = $propertyGroupOptionId;
        }
    }
    
    private function loadTabs(DevicePage $page, Request $request, SalesChannelContext $context)
    {
        if(!$page->device) {
            return;
        }
        $deviceId = $page->device->getId();
        $propertyGroupId = $this->settings->getString('groupingTabPropery');
        
        if(!Uuid::isValid($propertyGroupId)) {
            return;
        }
        
        $rez = $this->connection->fetchAllAssociative("
 SELECT LOWER(HEX(pgo.id)) as pgo_id, 
        COUNT(pp.product_id) as cnt
  FROM 
  `product_property` as pp,
  `property_group_option` as pgo,
  `topdata_device_to_product` as tdp
  WHERE pp.property_group_option_id=pgo.id 
        AND tdp.product_id = pp.product_id 
	AND tdp.device_id = 0x$deviceId
	AND pgo.property_group_id = 0x$propertyGroupId
  GROUP BY pgo.id
            ");
        
        $page->allProductsCount = 0;
        $propertyGroupOptionIds = [];
        foreach ($rez as $row) {
            $propertyGroupOptionIds[$row['pgo_id']] = $row['cnt'];
            $page->allProductsCount += $row['cnt'];
        }
        
        if(!count($propertyGroupOptionIds)) {
            return;
        }
        
        $criteria = new Criteria(array_keys($propertyGroupOptionIds));
        $criteria->addSorting(new FieldSorting('name'));
        $page->tabs = $this->propertyGroupOptionRepository->search(
                $criteria, 
                $context->getContext()
            )->getEntities();
        foreach ($page->tabs as $entity) {
            $entity->addExtension('productCount', new ArrayStruct([$propertyGroupOptionIds[$entity->getId()]]));
        }
//        dump($page->tabs);
//        die();
    }
    
    
    private function loadDeviceSynonims(DevicePage $page, SalesChannelContext $context) : void
    {
        $device = $page->device;
        
        $deviceSynonymIds = $this->entitiesHelper->getDeviceSynonymsIds($device->getId());
        if($deviceSynonymIds) {
            $synonyms = $this->deviceRepository->search(
                (new Criteria($deviceSynonymIds))
                    ->addFilter(new EqualsFilter('enabled', true))
                    ->addAssociations(['brand','type','series']), 
                $context->getContext()
            )->getEntities();
            if(count($synonyms)) {
                $page->addExtension('deviceSynonyms', $synonyms);
            }
        }
    }
    
    private function setSeoInformation(DevicePage $page) : void
    {
        $device = $page->device;
        
        $page->setPageTitle($this->translator->trans('topdata-topfinder.SEO.devicePageTitle', [
            '%brand%' => $device->getBrand()->getName(),
            '%device%' => $device->getModel()
        ]));
        
        $page->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.deviceMetaTitle', [
            '%brand%' => $device->getBrand()->getName(),
            '%device%' => $device->getModel()
        ]));
        
        $page->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.deviceMetaDescription', [
            '%brand%' => $device->getBrand()->getName(),
            '%device%' => $device->getModel()
        ]));
        
        if($page->activeTabId) {
            $page->getMetaInformation()->setRobots("noindex,follow");
        } 
        else {
            $page->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.deviceMetaRobots'));
        }
        
        
        $page->addBreadcrumb(
            $this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'), 
            $this->router->generate('frontend.top_finder.brands')
        );
        
        $page->addBreadcrumb(
            $this->translator->trans('topdata-topfinder.SEO.brandPageTitle', ['%brand%' => $device->getBrand()->getName()]), 
            $this->router->generate('frontend.top_finder.brand', ['brandCode' => $device->getBrand()->getCode()])
        );
        
        $page->addBreadcrumb(
            $page->getPageTitle(), 
            $this->router->generate('frontend.top_finder.device', ['deviceCode' => $device->getCode()])
        );
    }
}