<?php declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Sitemap;

use Shopware\Core\Content\Sitemap\Provider\AbstractUrlProvider;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Topdata\TopdataTopFinderProSW6\Component\SettingsService;
use Topdata\TopdataConnectorSW6\Core\Content\Series\SeriesCollection;
use Topdata\TopdataConnectorSW6\Core\Content\Series\SeriesEntity;
use Shopware\Core\Content\Sitemap\Struct\Url;
use Shopware\Core\Content\Sitemap\Struct\UrlResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\Content\Sitemap\Service\ConfigHandler;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\NotFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsAnyFilter;
use Shopware\Core\Content\Seo\SeoUrlPlaceholderHandlerInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;

class SeriesUrlProvider extends AbstractUrlProvider
{
    public const CHANGE_FREQ = 'daily';

    /**
     * @var EntityRepository
     */
    private $seriesRepository;

    /**
     * @var SettingsService
     */
    private $settings;
    
    /**
     * @var SeoUrlPlaceholderHandlerInterface
     */
    private $seoUrlPlaceholderHandler;
    
    /**
     * @var ConfigHandler
     */
    private $configHandler;

    public function __construct(
        EntityRepository $seriesRepository,
        SettingsService $settings,
        SeoUrlPlaceholderHandlerInterface $seoUrlPlaceholderHandler,
        ConfigHandler $configHandler
    ) {
        $this->seriesRepository = $seriesRepository;
        $this->settings = $settings;
        $this->seoUrlPlaceholderHandler = $seoUrlPlaceholderHandler;
        $this->configHandler = $configHandler;
    }


    public function getDecorated(): AbstractUrlProvider
    {
        throw new DecorationPatternException(self::class);
    }

    public function getName(): string
    {
        return 'device_series';
    }

    /**
     * {@inheritdoc}
     */
    public function getUrls(SalesChannelContext $salesChannelContext, int $limit, ?int $offset = null): UrlResult
    {
        if($this->settings->getBool('showSeries')) {
            return new UrlResult([], null);
        }
        $series = $this->getSeries($salesChannelContext, $limit, $offset);
        $urls = [];
        $url = new Url();
        foreach ($series as $serie) {
            /** @var \DateTimeInterface $lastmod */
            $lastmod = $serie->getUpdatedAt() ?: $serie->getCreatedAt();

            $newUrl = clone $url;
            $newUrl->setLoc($this->seoUrlPlaceholderHandler->generate('frontend.top_finder.series', ['code'=>$serie->getCode()]));
            $newUrl->setLastmod($lastmod);
            $newUrl->setChangefreq(self::CHANGE_FREQ);
            $newUrl->setResource(SeriesEntity::class);
            $newUrl->setIdentifier($serie->getId());

            $urls[] = $newUrl;
        }

        if (\count($urls) < $limit) { // last run
            $nextOffset = null;
        } elseif ($offset === null) { // first run
            $nextOffset = $limit;
        } else { // 1+n run
            $nextOffset = $offset + $limit;
        }

        return new UrlResult($urls, $nextOffset);
    }

    private function getSeries(SalesChannelContext $salesChannelContext, int $limit, ?int $offset): SeriesCollection
    {
        $criteria = new Criteria();
        $criteria->setLimit($limit);

        if ($offset !== null) {
            $criteria->setOffset($offset);
        }
        
        $criteria->addFilter(new EqualsFilter('enabled', true));
        $criteria->addSorting(new FieldSorting('wsId'));
        
        $excludedSeriesIds = $this->getExcludedSeriesIds($salesChannelContext);
        if (!empty($excludedSeriesIds)) {
            $criteria->addFilter(new NotFilter(NotFilter::CONNECTION_AND, [new EqualsAnyFilter('id', $excludedSeriesIds)]));
        }
        
        return $this->seriesRepository->search($criteria, $salesChannelContext->getContext())->getEntities();
    }
    
    private function getExcludedSeriesIds(SalesChannelContext $salesChannelContext): array
    {
        $salesChannelId = $salesChannelContext->getSalesChannel()->getId();

        $excludedUrls = $this->configHandler->get(ConfigHandler::EXCLUDED_URLS_KEY);
        if (empty($excludedUrls)) {
            return [];
        }

        $excludedUrls = array_filter($excludedUrls, static function (array $excludedUrl) use ($salesChannelId) {
            if ($excludedUrl['resource'] !== SeriesEntity::class) {
                return false;
            }

            if ($excludedUrl['salesChannelId'] !== $salesChannelId) {
                return false;
            }

            return true;
        });

        return array_column($excludedUrls, 'identifier');
    }
}
