<?php declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Sitemap;

use Shopware\Core\Content\Sitemap\Provider\AbstractUrlProvider;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Topdata\TopdataTopFinderProSW6\Component\SettingsService;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataConnectorSW6\Core\Content\Device\DeviceCollection;
use Topdata\TopdataConnectorSW6\Core\Content\Device\DeviceEntity;
use Shopware\Core\Content\Sitemap\Struct\Url;
use Shopware\Core\Content\Sitemap\Struct\UrlResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\Content\Sitemap\Service\ConfigHandler;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\NotFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsAnyFilter;
use Shopware\Core\Content\Seo\SeoUrlPlaceholderHandlerInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;

class DeviceUrlProvider extends AbstractUrlProvider
{
    public const CHANGE_FREQ = 'daily';

    /**
     * @var EntityRepository
     */
    private $deviceRepository;

    /**
     * @var SettingsService
     */
    private $settings;
    
    /**
     * @var UrlGeneratorInterface
     */
    private $router;
    
    /**
     * @var SeoUrlPlaceholderHandlerInterface
     */
    private $seoUrlPlaceholderHandler;
    
    /**
     * @var ConfigHandler
     */
    private $configHandler;

    public function __construct(
        EntityRepository $deviceRepository,
        SettingsService $settings,
        UrlGeneratorInterface $router,
        SeoUrlPlaceholderHandlerInterface $seoUrlPlaceholderHandler,
        ConfigHandler $configHandler
    ) {
        $this->deviceRepository = $deviceRepository;
        $this->settings = $settings;
        $this->router = $router;
        $this->seoUrlPlaceholderHandler = $seoUrlPlaceholderHandler;
        $this->configHandler = $configHandler;
    }


    public function getDecorated(): AbstractUrlProvider
    {
        throw new DecorationPatternException(self::class);
    }

    public function getName(): string
    {
        return 'device';
    }

    /**
     * {@inheritdoc}
     */
    public function getUrls(SalesChannelContext $salesChannelContext, int $limit, ?int $offset = null): UrlResult
    {
        $devices = $this->getDevices($salesChannelContext, $limit, $offset);

        $urls = [];
        $url = new Url();
        foreach ($devices as $device) {
            /** @var \DateTimeInterface $lastmod */
            $lastmod = $device->getUpdatedAt() ?: $device->getCreatedAt();

            $newUrl = clone $url;
            //$this->router->generate('frontend.top_finder.device', ['deviceCode'=>$device->getCode()])
            $newUrl->setLoc($this->seoUrlPlaceholderHandler->generate('frontend.top_finder.device', ['deviceCode'=>$device->getCode()]));
            $newUrl->setLastmod($lastmod);
            $newUrl->setChangefreq(self::CHANGE_FREQ);
            $newUrl->setResource(DeviceEntity::class);
            $newUrl->setIdentifier($device->getId());

            $urls[] = $newUrl;
        }

        if (\count($urls) < $limit) { // last run
            $nextOffset = null;
        } elseif ($offset === null) { // first run
            $nextOffset = $limit;
        } else { // 1+n run
            $nextOffset = $offset + $limit;
        }

        return new UrlResult($urls, $nextOffset);
    }

    private function getDevices(SalesChannelContext $salesChannelContext, int $limit, ?int $offset): DeviceCollection
    {
        $criteria = new Criteria();
        $criteria->setLimit($limit);

        if ($offset !== null) {
            $criteria->setOffset($offset);
        }
        
        $criteria->addFilter(new EqualsFilter('enabled', true));
        $criteria->addSorting(new FieldSorting('wsId'));
        
        $excludedDeviceIds = $this->getExcludedDeviceIds($salesChannelContext);
        if (!empty($excludedDeviceIds)) {
            $criteria->addFilter(new NotFilter(NotFilter::CONNECTION_AND, [new EqualsAnyFilter('id', $excludedDeviceIds)]));
        }
        
        return $this->deviceRepository->search($criteria, $salesChannelContext->getContext())->getEntities();
    }
    
    private function getExcludedDeviceIds(SalesChannelContext $salesChannelContext): array
    {
        $salesChannelId = $salesChannelContext->getSalesChannel()->getId();

        $excludedUrls = $this->configHandler->get(ConfigHandler::EXCLUDED_URLS_KEY);
        if (empty($excludedUrls)) {
            return [];
        }

        $excludedUrls = array_filter($excludedUrls, static function (array $excludedUrl) use ($salesChannelId) {
            if ($excludedUrl['resource'] !== DeviceEntity::class) {
                return false;
            }

            if ($excludedUrl['salesChannelId'] !== $salesChannelId) {
                return false;
            }

            return true;
        });

        return array_column($excludedUrls, 'identifier');
    }
}
