<?php declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Sitemap;

use Shopware\Core\Content\Sitemap\Provider\AbstractUrlProvider;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Topdata\TopdataTopFinderProSW6\Component\SettingsService;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataConnectorSW6\Core\Content\Brand\BrandCollection;
use Topdata\TopdataConnectorSW6\Core\Content\Brand\BrandEntity;
use Shopware\Core\Content\Sitemap\Struct\Url;
use Shopware\Core\Content\Sitemap\Struct\UrlResult;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\Content\Sitemap\Service\ConfigHandler;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\NotFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsAnyFilter;
use Shopware\Core\Content\Seo\SeoUrlPlaceholderHandlerInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;

class BrandUrlProvider extends AbstractUrlProvider
{
    public const CHANGE_FREQ = 'daily';

    /**
     * @var EntityRepository
     */
    private $brandRepository;

    /**
     * @var SettingsService
     */
    private $settings;
    
    /**
     * @var UrlGeneratorInterface
     */
    private $router;
    
    /**
     * @var SeoUrlPlaceholderHandlerInterface
     */
    private $seoUrlPlaceholderHandler;
    
    /**
     * @var ConfigHandler
     */
    private $configHandler;

    public function __construct(
        EntityRepository $brandRepository,
        SettingsService $settings,
        UrlGeneratorInterface $router,
        SeoUrlPlaceholderHandlerInterface $seoUrlPlaceholderHandler,
        ConfigHandler $configHandler
    ) {
        $this->brandRepository = $brandRepository;
        $this->settings = $settings;
        $this->router = $router;
        $this->seoUrlPlaceholderHandler = $seoUrlPlaceholderHandler;
        $this->configHandler = $configHandler;
    }

    public function getDecorated(): AbstractUrlProvider
    {
        throw new DecorationPatternException(self::class);
    }

    public function getName(): string
    {
        return 'device_brand';
    }

    /**
     * {@inheritdoc}
     */
    public function getUrls(SalesChannelContext $salesChannelContext, int $limit, ?int $offset = null): UrlResult
    {
        $brands = $this->getBrands($salesChannelContext, $limit, $offset);

        $urls = [];
        $url = new Url();
        foreach ($brands as $brand) {
            /** @var \DateTimeInterface $lastmod */
            $lastmod = $brand->getUpdatedAt() ?: $brand->getCreatedAt();

            $newUrl = clone $url;
            //$this->router->generate('frontend.top_finder.device', ['deviceCode'=>$device->getCode()])
            $newUrl->setLoc($this->seoUrlPlaceholderHandler->generate('frontend.top_finder.brand', ['brandCode'=>$brand->getCode()]));
            $newUrl->setLastmod($lastmod);
            $newUrl->setChangefreq(self::CHANGE_FREQ);
            $newUrl->setResource(BrandEntity::class);
            $newUrl->setIdentifier($brand->getId());

            $urls[] = $newUrl;
        }

        if (\count($urls) < $limit) { // last run
            $nextOffset = null;
        } elseif ($offset === null) { // first run
            $nextOffset = $limit;
        } else { // 1+n run
            $nextOffset = $offset + $limit;
        }

        return new UrlResult($urls, $nextOffset);
    }

    private function getBrands(SalesChannelContext $salesChannelContext, int $limit, ?int $offset): BrandCollection
    {
        $criteria = new Criteria();
        $criteria->setLimit($limit);

        if ($offset !== null) {
            $criteria->setOffset($offset);
        }
        
        $criteria->addFilter(new EqualsFilter('enabled', true));
        $criteria->addSorting(new FieldSorting('wsId'));
        
        $excludedBrandIds = $this->getExcludedBrandIds($salesChannelContext);
        if (!empty($excludedBrandIds)) {
            $criteria->addFilter(new NotFilter(NotFilter::CONNECTION_AND, [new EqualsAnyFilter('id', $excludedBrandIds)]));
        }
        
        return $this->brandRepository->search($criteria, $salesChannelContext->getContext())->getEntities();
    }
    
    private function getExcludedBrandIds(SalesChannelContext $salesChannelContext): array
    {
        $salesChannelId = $salesChannelContext->getSalesChannel()->getId();

        $excludedUrls = $this->configHandler->get(ConfigHandler::EXCLUDED_URLS_KEY);
        if (empty($excludedUrls)) {
            return [];
        }

        $excludedUrls = array_filter($excludedUrls, static function (array $excludedUrl) use ($salesChannelId) {
            if ($excludedUrl['resource'] !== BrandEntity::class) {
                return false;
            }

            if ($excludedUrl['salesChannelId'] !== $salesChannelId) {
                return false;
            }

            return true;
        });

        return array_column($excludedUrls, 'identifier');
    }
}
