<?php declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Controller;

use Shopware\Core\Framework\Context;
use Shopware\Storefront\Controller\StorefrontController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\HttpFoundation\JsonResponse;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Validation\DataBag\RequestDataBag;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\TopfinderPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\DeviceListPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\DeviceSuggestPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\DeviceSearchPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\TopfinderPage;
use Shopware\Core\Content\Product\SalesChannel\Listing\ProductListingResult;
use Topdata\TopdataConnectorSW6\Core\Content\Device\DeviceEntity;
use Topdata\TopdataTopFinderProSW6\Component\Collection;
use Shopware\Core\Framework\DataAbstractionLayer\EntityCollection;
use Topdata\TopdataTopFinderProSW6\Component\SettingsService;
use Shopware\Core\System\SalesChannel\Entity\SalesChannelRepository;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\ListPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\CompatibleDevicesWidget\CompatibleDevicesWidgetLoader;
use Topdata\TopdataConnectorSW6\Component\Helper\EntitiesHelper;
use Symfony\Component\HttpFoundation\Cookie;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\PopupFormWidget\PopupFormWidgetLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\PopupDeviceWidget\PopupDeviceWidgetLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\BrandLetter\BrandLetterPageLoader;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\OneBrandLetter\OneBrandLetterPageLoader;
use Shopware\Core\Framework\Routing\Exception\MissingRequestParameterException;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Topdata\TopdataTopFinderProSW6\Component\Helper;
use Topdata\TopdataConnectorSW6\Core\Content\Device\Agregate\DeviceCustomer\DeviceCustomerEntity;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\PopupDeviceInfoWidget\PopupDeviceInfoWidgetLoader;
use Shopware\Core\Framework\Uuid\Uuid;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\ListProducts\ListProductsLoader;
use Topdata\TopdataTopFinderProSW6\Content\Product\Events\DeviceProductsCriteriaEvent;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Device\DevicePage;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Device\DevicePageLoader;
use Shopware\Storefront\Page\Page;
use Shopware\Core\Checkout\Cart\Exception\CustomerNotLoggedInException;
use Symfony\Component\HttpKernel\EventListener\AbstractSessionListener;

/**
 * @Route(defaults={"_routeScope"={"storefront"}})
 */
class TopFinderController extends StorefrontController
{
    const POPUP_DEVICES_PRELOAD_LENGTH = 20;
    
    /**
     * @var ?array
     */
    private $deviceList = null;
    
    /**
     * @var SalesChannelRepository
     */
    private $productRepository;
    
    /**
     * @var EntityRepository
     */
    private $brandRepository;
    
    /**
     * @var EntityRepository
     */
    private $deviceRepository;
    
    /**
     * @var EntityRepository
     */
    private $deviceCustomerRepository;
    
    /**
     * @var EntityRepository
     */
    private $seriesRepository;
    
    /**
     * @var EntityRepository
     */
    private $typeRepository;
    
    /**
     * @var TopfinderPageLoader
     */
    private $topfinderPageLoader;
    
    /**
     * @var DeviceListPageLoader
     */
    private $deviceListPageLoader;
    
    /**
     * @var DeviceSuggestPageLoader
     */
    private $deviceSuggestPageLoader;
    
    /**
     * @var DeviceSearchPageLoader
     */
    private $deviceSearchPageLoader;
    
    /**
     * @var SettingsService
     */
    private $settings;
    
    /**
     * @var ListPageLoader
     */
    private $listPageLoader;
    
    /**
     * @var CompatibleDevicesWidgetLoader
     */
    private $compatibleDevicesWidgetLoader;
    
    /**
     * @var type PopupDeviceWidgetLoader
     */
    private $popupDeviceWidgetLoader;
    
    /**
     * @var PopupFormWidgetLoader
     */
    private $popupFormWidgetLoader;
    
    /**
     * @var BrandLetterPageLoader
     */
    private $brandLetterPageLoader;
    
    /**
     * @var OneBrandLetterPageLoader
     */
    private $oneBrandLetterPageLoader;
    
    /**
     * @var PopupDeviceInfoWidgetLoader
     */
    private $popupDeviceInfoWidgetLoader;
    
    /**
     * @var ListProductsLoader
     */
    private $listProductsLoader;
    
    /**
     * @var DevicePageLoader
     */
    private $devicePageLoader;
    
    /**
     * @var EntitiesHelper
     */
    private $entitiesHelper;

    public function __construct(
        SalesChannelRepository $productRepository,
        EntityRepository $brandRepository,
        EntityRepository $deviceRepository,
        EntityRepository $seriesRepository,
        EntityRepository $typeRepository,
        EntityRepository $deviceCustomerRepository,
        TopfinderPageLoader $topfinderPageLoader,
        DeviceListPageLoader $deviceListPageLoader,
        DeviceSuggestPageLoader $deviceSuggestPageLoader,
        DeviceSearchPageLoader $deviceSearchPageLoader,
        SettingsService $settingsService,
        ListPageLoader $listPageLoader,
        CompatibleDevicesWidgetLoader $compatibleDevicesWidgetLoader,
        PopupFormWidgetLoader $popupFormWidgetLoader,
        PopupDeviceWidgetLoader $popupDeviceWidgetLoader,
        BrandLetterPageLoader $brandLetterPageLoader,
        OneBrandLetterPageLoader $oneBrandLetterPageLoader,
        PopupDeviceInfoWidgetLoader $popupDeviceInfoWidgetLoader,
        ListProductsLoader $listProductsLoader,
        DevicePageLoader $devicePageLoader,
        EntitiesHelper $entitiesHelper
    )
    {
        $this->productRepository = $productRepository;
        $this->brandRepository = $brandRepository;
        $this->deviceRepository = $deviceRepository;
        $this->seriesRepository = $seriesRepository;
        $this->typeRepository = $typeRepository;
        $this->topfinderPageLoader = $topfinderPageLoader;
        $this->deviceListPageLoader = $deviceListPageLoader;
        $this->deviceSuggestPageLoader = $deviceSuggestPageLoader;
        $this->deviceSearchPageLoader = $deviceSearchPageLoader;
        $this->settings = $settingsService;
        $this->listPageLoader = $listPageLoader;
        $this->compatibleDevicesWidgetLoader = $compatibleDevicesWidgetLoader;
        $this->popupFormWidgetLoader = $popupFormWidgetLoader;
        $this->popupDeviceWidgetLoader = $popupDeviceWidgetLoader;
        $this->brandLetterPageLoader = $brandLetterPageLoader;
        $this->oneBrandLetterPageLoader = $oneBrandLetterPageLoader;
        $this->deviceCustomerRepository = $deviceCustomerRepository;
        $this->popupDeviceInfoWidgetLoader = $popupDeviceInfoWidgetLoader;
        $this->listProductsLoader = $listProductsLoader;
        
        $this->devicePageLoader = $devicePageLoader;
        $this->entitiesHelper = $entitiesHelper;
    }
    
    private function getSeriesArray($brandId) : array
    {
        $seriesArray = [];
        
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('enabled', true));
        $criteria->addFilter(new EqualsFilter('brandId', $brandId));
        $criteria->addSorting(new FieldSorting('label', FieldSorting::ASCENDING));
        $series = $this->seriesRepository->search($criteria, Context::createDefaultContext())->getEntities();
        
        if(!count($series)) {
            return $seriesArray;
        }
        
        
        foreach ($series as $serie) {
            $seriesArray[$serie->getCode()] = $serie->getLabel();
        }
        
        return $seriesArray;
    }
    
    /**
     * @Route("/top-finder/get-series/{brandCode}", name="frontend.top_finder.get_series", methods={"GET","POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function getSeries(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $brandCode = $request->get('brandCode');
        
        if(!trim($brandCode)) {
            return new JsonResponse(['found' => false]);
        }
        
        $brand = $this
            ->brandRepository
            ->search(
                (new Criteria())
                    ->addFilter(new EqualsFilter('enabled', true))
                    ->addFilter(new EqualsFilter('code', $brandCode)),
                $salesChannelContext->getContext()
            )
            ->getEntities()
            ->first();
        
        if(!$brand) {
            return new JsonResponse(['found' => false]);
        }
        
        $brandId = $brand->getId();
        $seriesArray = $this->getSeriesArray($brandId);
        
        if(!count($seriesArray)) {
            return new JsonResponse(['found' => false]);
        }
        
        return new JsonResponse([
            'found' => true,
            'items' => $seriesArray
        ]);
    }
    
    
    public function getTypesArray($brandId) : array
    {
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('enabled', true));
        $criteria->addFilter(new EqualsFilter('brandId', $brandId));
        $criteria->addSorting(new FieldSorting('label', FieldSorting::ASCENDING));
        $types = $this->typeRepository->search($criteria, Context::createDefaultContext())->getEntities();
        
        if(!count($types)) {
            return [];
        }
        
        
        $typesArray = [];
        foreach ($types as $type) {
            $typesArray[$type->getCode()] = $type->getLabel();
        }
        
        return $typesArray;
    }
    
    
    /**
     * @Route("/top-finder/get-types/{brandCode}", name="frontend.top_finder.get_types", methods={"GET","POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function getTypes(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $brandCode = $request->get('brandCode');
        
        if(!trim($brandCode)) {
            return new JsonResponse(['found' => false]);
        }
        
        $brand = $this
            ->brandRepository
            ->search(
                (new Criteria())
                    ->addFilter(new EqualsFilter('enabled', true))
                    ->addFilter(new EqualsFilter('code', $brandCode)),
                $salesChannelContext->getContext()
            )
            ->getEntities()
            ->first();
        
        if(!$brand) {
            return new JsonResponse(['found' => false]);
        }
        
        $brandId = $brand->getId();
        
        $typesArray = $this->getTypesArray($brandId);
        
        if(!count($typesArray)) {
            return new JsonResponse(['found' => false]);
        }
        
        return new JsonResponse([
            'found' => true,
            'items' => $typesArray
        ]);
    }
    
    
    private function resolveCode($code, $repository, $context)
    {
        if($code == '0') {
            return '0';
        }
        
        if($code == '-1') {
            return '-1';
        }
        
        if(!trim($code)) {
            return false;
        }
        
        $entity = $repository
            ->search(
                (new Criteria())
                    ->addFilter(new EqualsFilter('enabled', true))
                    ->addFilter(new EqualsFilter('code', $code)),
                $context
            )
            ->getEntities()
            ->first();
        
        if(!$entity) {
            return false;
        }
        
        return $entity->getId();
    }
    
    /**
     * @Route("/top-finder/get-devices/{brandCode}/{typeCode}/{seriesCode}", name="frontend.top_finder.get_devices", methods={"GET","POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function getDevices(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $brandCode = $this->resolveCode((string)$request->get('brandCode'), $this->brandRepository, $salesChannelContext->getContext());
        $seriesCode = $this->resolveCode((string)$request->get('seriesCode'), $this->seriesRepository, $salesChannelContext->getContext());
        $typeCode = $this->resolveCode((string)$request->get('typeCode'), $this->typeRepository, $salesChannelContext->getContext());
        
        if(($brandCode === false) || ($seriesCode === false) || ($typeCode === false)) {
            return new JsonResponse(['found' => false]);
        }
        
        $devicesArray = $this->getDevicesArray($brandCode, $seriesCode, $typeCode);
        
        if(!count($devicesArray)) {
            return new JsonResponse(['found' => false]);
        }
        
        return new JsonResponse([
            'found' => true,
            'items' => $devicesArray
        ]);
    }
    
    private function getDevicesArray($brandId, $seriesId, $typeId, bool $codeAsKey = true) : array
    {
        $return = [];
        $params = [];
        /** @var \Doctrine\DBAL\Connection $connection */
        $connection = $this->container->get('Doctrine\DBAL\Connection');
        
        $params[] = 'is_enabled = 1';
        
        if(EntitiesHelper::isValidUuid($brandId)) {
            $params[] = 'brand_id = 0x'.$brandId;
        }
        else {
            return $return;
        }
        
        if($seriesId == -1) {
            $params[] = 'series_id is null';
        }
        elseif(EntitiesHelper::isValidUuid($seriesId)) {
            $params[] = 'series_id = 0x'.$seriesId;
        }
        
        if($typeId == -1) {
            $params[] = 'type_id is null';
        }
        elseif(EntitiesHelper::isValidUuid($typeId)) {
            $params[] = 'type_id = 0x'.$typeId;
        }
        
        $devices = $connection->fetchAllAssociative(
            'SELECT code, model FROM `topdata_device`'
            .' WHERE ('.implode(') AND (', $params).')'
            .' ORDER BY model ASC');
        
        if($codeAsKey) {
            foreach ($devices as $device) {
                $return[$device['code']] = $device['model'];
            }
        }
        else {
            $return = $devices;
        }
        
        return $return;
        
    }
    
    private function getEntityCollection(EntityRepository $repository, array $params = [], ?string $sortingField = null, bool $deleteNullParams = true) : EntityCollection
    {
        $criteria = new Criteria();
        foreach ($params as $key => $value) {
            if(($value === null) && $deleteNullParams) {
                continue;
            }
            $criteria->addFilter(new EqualsFilter($key, $value));
        }
        if($sortingField !== null) {
            $criteria->addSorting(new FieldSorting($sortingField, FieldSorting::ASCENDING));
        }
        $criteria->setLimit(999);
        return $repository->search($criteria, Context::createDefaultContext())->getEntities();
    }
    
    /**
     * @Route("/top-finder/device/{deviceCode}", name="frontend.top_finder.device", methods={"GET","POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function getDeviceNew(Request $request, SalesChannelContext $context): Response
    {
        /** @var DevicePage $page */
        $page = $this->devicePageLoader->load($request, $context);
        
        $page->device->setInDeviceList(
            in_array($page->device->getId(), 
            $this->getDeviceList($context))
        );
        
        if($request->isXmlHttpRequest() === false 
            && $this->settings->getBool('showDeviceHistory'))
        {
            $this->setDeviceToHistory($request, $page);
        }
        
        if($request->isXmlHttpRequest() === false) {
            $this->fillSelectboxes($page, $page->device, $context, $request);
        }
        $response = $this->renderStorefront(
            '@TopdataTopFinderProSW6/storefront/page/topfinder/device-page.html.twig', 
            ['page' => $page]
        );
        
        if($request->isXmlHttpRequest() === false) {
            if($this->settings->getBool('showDeviceHistory')) {
                $response->headers->setCookie(
                    Cookie::create(
                        Helper::COOKIE_DEVICE_HISTORY,
                        implode(',', $page->deviceHistory),
                        time() + 3600*24*30
                    )
                );
            }
        }
        
        return $response;
    }
    
    
    /**
     * @Route("/top-finder/device/{deviceCode}/{propertyGroupOptionId}", name="frontend.top_finder.device_tab", methods={"GET","POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function getDeviceTab(Request $request, SalesChannelContext $context): Response
    {
        /** @var DevicePage $page */
        $page = $this->devicePageLoader->load($request, $context);
        
        $page->device->setInDeviceList(in_array($page->device->getId(), $this->getDeviceList($context)));
        
        if($request->isXmlHttpRequest() === false 
            && $this->settings->getBool('showDeviceHistory'))
        {
            $this->setDeviceToHistory($request, $page);
        }
        
        if($request->isXmlHttpRequest() === false) {
            $this->fillSelectboxes($page, $page->device, $context, $request);
        }
        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/device-page.html.twig', ['page' => $page]);
        
        if($request->isXmlHttpRequest() === false) {
            if($this->settings->getBool('showDeviceHistory')) {
                $response->headers->setCookie(
                    Cookie::create(
                        Helper::COOKIE_DEVICE_HISTORY,
                        implode(',', $page->deviceHistory),
                        time() + 3600*24*30
                    )
                );
            }
        }
        
        return $response;
    }
    
    
    /**
     * @Route("/top-finder/device-filters/{deviceCode}/{propertyGroupOptionId}", name="frontend.top_finder.device_tab_filters", methods={"GET","POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function getDeviceTabFilters(Request $request, SalesChannelContext $context): Response
    {
        // Allows to fetch only aggregations over the gateway.
        $request->request->set('only-aggregations', true);

        // Allows to convert all post-filters to filters. This leads to the fact that only aggregation values are returned, which are combinable with the previous applied filters.
        $request->request->set('reduce-aggregations', true);

        /** @var DevicePage $page */
        $page = $this->devicePageLoader->load($request, $context, true);
        //$listing = $page->listing->getResult();
        $listing = $page->listing;

        $mapped = [];
        foreach ($listing->getAggregations() as $aggregation) {
            $mapped[$aggregation->getName()] = $aggregation;
        }

        $response = new JsonResponse($mapped);
        $response->headers->set(AbstractSessionListener::NO_AUTO_CACHE_CONTROL_HEADER, '1');

        return $response;
    }
    
    
    /**
     * @Route("/top-finder/device-filters/{deviceCode}", name="frontend.top_finder.device_filters", methods={"GET","POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function getDeviceFilters(Request $request, SalesChannelContext $context): Response
    {
        // Allows to fetch only aggregations over the gateway.
        $request->request->set('only-aggregations', true);

        // Allows to convert all post-filters to filters. This leads to the fact that only aggregation values are returned, which are combinable with the previous applied filters.
        $request->request->set('reduce-aggregations', true);

        /** @var DevicePage $page */
        $page = $this->devicePageLoader->load($request, $context, true);
        //$listing = $page->listing->getResult();
        $listing = $page->listing;

        $mapped = [];
        foreach ($listing->getAggregations() as $aggregation) {
            $mapped[$aggregation->getName()] = $aggregation;
        }

        $response = new JsonResponse($mapped);
        $response->headers->set(AbstractSessionListener::NO_AUTO_CACHE_CONTROL_HEADER, '1');

        return $response;
    }
    
    private function setDeviceToHistory($request, $page) : void
    {
        $device = $page->device;
        
        $cokie = $request->cookies->get(Helper::COOKIE_DEVICE_HISTORY);

        $devicesHistory = [];

        if($cokie) {
            $devicesHistory = explode(',', $cokie);
        }

        if(!is_array($devicesHistory)) {
            $devicesHistory = [];
        }

        $historyKey = array_search($device->getId(), $devicesHistory);

        if($historyKey !== false) {
            unset($devicesHistory[$historyKey]);
        }

        $devicesHistory[] = $device->getId();

        while(count($devicesHistory) > $this->settings->getInt('deviceHistoryElements')) {
            array_shift($devicesHistory);
        }

        $page->deviceHistory = $devicesHistory;
    }
    
    
    private function getProductListingByDevice(DeviceEntity $device, SalesChannelContext $context) : ProductListingResult
    {
        $criteria = new Criteria($device->getProducts()->getIds());
        
        /** @var EventDispatcherInterface $eventDispatcher */
        $eventDispatcher = $this->container->get('event_dispatcher');
        $eventDispatcher->dispatch(
            new DeviceProductsCriteriaEvent($criteria, $context)
        );
        
        $result = $this->productRepository->search($criteria, $context);
        $result = ProductListingResult::createFrom($result);
        
        return $result;
    }
    
    private function fillSelectboxes(Page $page, DeviceEntity $device, SalesChannelContext $context, Request $request) : void
    {
        $finderStruct = $page->getHeader()->getExtension('finder');
        if($finderStruct === null) {
            $finderStruct = new Collection();
        }
        
        if($device->getBrandId()) {
            $brand = $this
                ->brandRepository
                ->search(
                    (new Criteria([$device->getBrandId()]))
                        ->addFilter(new EqualsFilter('enabled', true)),
                    $context->getContext()
                )
                ->getEntities()
                ->first();
            if($brand) {
                $finderStruct->set('brandCode', $brand->getCode());
            }
        }
        
        if($device->getSeriesId()) {
            $series = $this
                ->seriesRepository
                ->search(
                    (new Criteria([$device->getSeriesId()]))
                        ->addFilter(new EqualsFilter('enabled', true)),
                    $context->getContext()
                )
                ->getEntities()
                ->first();
            if($series) {
                $finderStruct->set('seriesCode', $series->getCode());
            }
        }
        
        if($device->getTypeId()) {
            $type = $this
                ->typeRepository
                ->search(
                    (new Criteria([$device->getTypeId()]))
                        ->addFilter(new EqualsFilter('enabled', true)),
                    $context->getContext()
                )
                ->getEntities()
                ->first();
            if($type) {
                $finderStruct->set('typeCode', $type->getCode());
            }
        }
        
        $finderStruct->set('deviceCode', $device->getCode());
        
        if(is_array($page->deviceHistory) && count($page->deviceHistory)
                && ($this->settings->getBool('showDevicesSlider') 
                    || $this->settings->getBool('showDeviceHistorySelect'))
        ) {
            $criteria = (new Criteria(array_reverse($page->deviceHistory)))
                ->addFilter(new EqualsFilter('enabled', true))
                ->addAssociations(['brand','media']);
            $devicesHistory = $this->deviceRepository->search($criteria, $context->getContext())->getElements();
            foreach ($devicesHistory as $dev) {
                if($this->settings->getBool('showDevicelist')) {
                    $dev->setInDeviceList(
                        in_array(
                            $dev->getId(), 
                            $this->getDeviceList($context)
                        )
                    );
                } else {
                    $dev->setInDeviceList(false);
                }
            }
            $devicesHistoryStruct = new Collection();
            $devicesHistoryStruct->set('devices', $devicesHistory);
            if($this->settings->getBool('showDeviceHistorySelect')) {
                $page->getHeader()->removeExtension('deviceHistory');
                $page->getHeader()->addExtension('deviceHistory', $devicesHistoryStruct);
            }
            if($this->settings->getBool('showDevicesSlider')) {
                
                $devicesHistoryStruct->set('userHiddable', $this->settings->getBool('deviceSliderUserHide'));
                $devicesHistoryStruct->set('hidden', $request->cookies->get(Helper::COOKIE_DEVICE_SLIDER) === 'hidden');
                
                $page->getHeader()->removeExtension('deviceSlider');
                $page->getHeader()->addExtension('deviceSlider', $devicesHistoryStruct);
            }
        }
        
        if($this->settings->getConfig('showSeries')) {
            $series = $this->getEntityCollection(
                $this->seriesRepository, 
                [
                    'enabled'=>true,
                    'brandId'=>$device->getBrandId()
                ],
                'label'
            );
            $finderStruct->set('series', $series);
        }
        
        if($this->settings->getConfig('showTypes')) {
            $types = $this->getEntityCollection(
                $this->typeRepository, 
                [
                    'enabled'=>true,
                    'brandId'=>$device->getBrandId()
                ],
                'label'
            );
            $finderStruct->set('types', $types);
        }
        $devices = $this->getDevicesArray($device->getBrandId(), $device->getSeriesId(), $device->getTypeId(), false);
//        $devices = $this->getEntityCollection($this->deviceRepository, $deviceParams, 'model', false);
        $finderStruct->set('devices', $devices);
        
//        $finderStruct->set('test', true);
//        $page->getHeader()->removeExtension('finder');
//        $page->getHeader()->addExtension('finder', $finderStruct);
    }
    
    
    /**
     * Add device to customers personal Device List
     * @Route("/top-finder/add-device/{deviceId}", name="frontend.top_finder.add-device", methods={"GET","POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function addDevice(Request $request, RequestDataBag $data, SalesChannelContext $context): JsonResponse
    {
        $deviceId = $request->get('deviceId');
        $returnError = new JsonResponse([
            'success' => false,
            'info' => 'Something went wrong'
        ]);
        $returnSuccess = new JsonResponse([
            'success' => true,
            'deviceId' => $deviceId
        ]);
        
        if(!EntitiesHelper::isValidUuid($deviceId)) {
            return $returnError;
        }
        
        if(!$context->getCustomer() || $context->getCustomer()->getGuest()) {
            return $returnError;
        }
        
        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('id', $deviceId))
            ->addFilter(new EqualsFilter('enabled', true))
            ->setLimit(1);
        /** @var DeviceEntity $device */
        $device = $this->deviceRepository->search($criteria, $context->getContext())->getEntities()->first();
        
        if($device === null) {
            return $returnError;
        }
        
        $connection = $this->container->get('Doctrine\DBAL\Connection');
        if(Helper::deviceToCustomerId($deviceId, $context->getCustomer()->getId(), $connection)) {
            return $returnSuccess;
        }
        
        $this->deviceCustomerRepository->create([
            [
                'deviceId' => $deviceId,
                'customerId' => $context->getCustomer()->getId(),
                'extraInfo' => json_encode(DeviceCustomerEntity::defaultExtraInfo())
            ]
        ], $context->getContext());
        
        if(Helper::deviceToCustomerId($deviceId, $context->getCustomer()->getId(), $connection)) {
            return $returnSuccess;
        }
        else {
            return $returnError;
        }
    }
    
    
    /**
     * Remove device from customers personal Device List
     * @Route("/top-finder/remove-device/{deviceId}", name="frontend.top_finder.remove-device", methods={"GET","POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function removeDevice(Request $request, RequestDataBag $data, SalesChannelContext $context): JsonResponse
    {
        /** @var \Doctrine\DBAL\Connection $connection */
        $connection = $this->container->get('Doctrine\DBAL\Connection');
        
        $deviceId = $request->get('deviceId');
        $returnError = new JsonResponse([
            'success' => false,
            'info' => 'Something went wrong'
        ]);
        $returnSuccess = new JsonResponse([
            'success' => true,
            'deviceId' => $deviceId
        ]);
        
        if(!EntitiesHelper::isValidUuid($deviceId)) {
            return $returnError;
        }
        
        if(!$context->getCustomer() || $context->getCustomer()->getGuest()) {
            return $returnError;
        }
        
        $criteria = (new Criteria())
            ->addFilter(new EqualsFilter('id', $deviceId))
            ->setLimit(1);
        /** @var DeviceEntity $device */
        $device = $this->deviceRepository->search($criteria, $context->getContext())->getEntities()->first();
        
        if($device === null) {
            return $returnError;
        }
        
        $connection->createQueryBuilder()
            ->delete('topdata_device_to_customer')
            ->where('(device_id=0x'. $deviceId .') AND (customer_id = 0x'. $context->getCustomer()->getId() .')')
            ->execute();
        
        return $returnSuccess;
    }
    
    private function getDeviceList(SalesChannelContext $context) : array
    {
        if(null === $this->deviceList) {
            $this->deviceList = [];
            if($context->getCustomer() && !($context->getCustomer()->getGuest())) {
                $connection = $this->container->get('Doctrine\DBAL\Connection');
                $rez = $connection->createQueryBuilder()
                    ->select(['device_id'])
                    ->from('topdata_device_to_customer')
                    ->where('customer_id = 0x' . $context->getCustomer()->getId() )
                    ->execute()
                    ->fetchAllAssociative();
                foreach ($rez as $val) {
                    $this->deviceList[] = bin2hex($val['device_id']);
                }
            }
        }
        
        return $this->deviceList;
    }
    
    /**
     * @Route("/account/devicelist", name="frontend.account.devicelist.page", methods={"GET"}, defaults={"_loginRequired"=true})
     *
     * @throws CustomerNotLoggedInException
     */
    public function deviceList(Request $request, SalesChannelContext $context): Response
    {
        if (!$context->getCustomer()
            || $context->getCustomer()->getGuest()
        ) {
            throw new CustomerNotLoggedInException();
        }

        $page = $this->deviceListPageLoader->load($request, $context);
        
        $page->getMetaInformation()->setRobots('noindex,nofollow');

        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/account/device-list.html.twig', [
            'page' => $page
        ]);
    }
    
    /**
     * @Route("/device-suggest", name="frontend.top_finder.device_suuggest", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function deviceSuggest(Request $request, SalesChannelContext $context): Response
    {
        $page = $this->deviceSuggestPageLoader->load($request, $context);

        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/device-suggest/index.html.twig', ['page' => $page]);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/device-search", name="frontend.top_finder.device_search.page", methods={"GET"})
     */
    public function deviceSearch(Request $request, SalesChannelContext $context): Response
    {
        $page = $this->deviceSearchPageLoader->load($request, $context);

        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/device-search.html.twig', ['page' => $page]);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/device-search-more", name="frontend.top_finder.device_search_more", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function deviceSearchMore(Request $request, SalesChannelContext $context): JsonResponse
    {
        $page = $this->deviceSearchPageLoader->load($request, $context);

        $result = ['success' => true];
        $result['html'] = $this->renderStorefront(
            '@Storefront/storefront/page/topfinder/device-search-more.html.twig', 
            ['page' => $page]
        )->getContent();
        
        if($result) {
            return new JsonResponse($result);
        }
    }
    
    
    /**
     * @Route("/top-finder/brands", name="frontend.top_finder.brands", methods={"GET"})
     */
    public function listBrands(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $page = $this->listPageLoader->load($request, $salesChanelContext, ListPageLoader::TYPE_BRANDS);
        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list.html.twig', ['page' => $page]);
    }
    
    /**
     * @Route("/top-finder/brand/{brandCode}", name="frontend.top_finder.brand", methods={"GET"})
     */
    public function listBrand(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $brandCode = $request->get('brandCode');
        if(!preg_match('/^[a-z0-9_-]+$/', $brandCode)) {
            throw new MissingRequestParameterException('brandCode', '/brandCode');
        }
        
        $page = $this->oneBrandLetterPageLoader->load(
            $request, 
            $salesChanelContext, 
            $brandCode
        );
        
        $finderStruct = $page->getHeader()->getExtension('finder');
        if($finderStruct !== null) {
            $finderStruct->set('brandCode', $page->brand['code']);
            
            if($this->settings->getConfig('showSeries')) {
                $series = $this->getEntityCollection(
                    $this->seriesRepository, 
                    [
                        'enabled'=>true,
                        'brandId'=>$page->brand['id']
                    ],
                    'label'
                );
                $finderStruct->set('series', $series);
            }

            if($this->settings->getConfig('showTypes')) {
                $types = $this->getEntityCollection(
                    $this->typeRepository, 
                    [
                        'enabled'=>true,
                        'brandId'=>$page->brand['id']
                    ],
                    'label'
                );
                $finderStruct->set('types', $types);
            }
            
            $devicesArray = $this->getDevicesArray($page->brand['id'], 0, 0, false);
            $finderStruct->set('devices', $devicesArray);
        }
        
        
        return $this->renderStorefront(
            '@TopdataTopFinderProSW6/storefront/page/topfinder/one-brand-letter.html.twig', 
            ['page' => $page]
        );
    }
    
    /**
     * @Route("/top-finder/brand-devices/{brandCode}", name="frontend.top_finder.brand_devices", methods={"GET", "POST"}, defaults={"productid"="", "XmlHttpRequest"=true})
     */
    public function listBrandDevices(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $brandCode = $request->get('brandCode');
        if(!preg_match('/^[a-z0-9_-]+$/', $brandCode)) {
            throw new MissingRequestParameterException('brandCode', '/brandCode');
        }
        
        $mode = $request->get('mode');
        if($mode === 'series') {
            $view = '@Storefront/storefront/page/topfinder/one-brand-letter-series.html.twig';
        }
        elseif($mode === 'types') {
            $view = '@Storefront/storefront/page/topfinder/one-brand-letter-types.html.twig';
        }
        else {
            $view = '@Storefront/storefront/page/topfinder/one-brand-letter-devices.html.twig';
        }
        
        $page = $this->oneBrandLetterPageLoader->load(
            $request, 
            $salesChanelContext, 
            $brandCode
        );
        
        $result = ['success' => true];
        $result['html'] = $this->renderStorefront($view, ['page' => $page])->getContent();
        
        if($result) {
            return new JsonResponse($result);
        }
    }
    
    
    /**
     * @Route("/top-finder/brandq/{code}", name="frontend.top_finder.brandq", methods={"GET"})
     */
    public function listBrandQ(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $brandCode = $request->get('code');
        if(!preg_match('/^[a-z0-9_-]+$/', $brandCode)) {
            throw new MissingRequestParameterException('brandCode', '/brandCode');
        }
        $page = $this->listPageLoader->load($request, $salesChanelContext, ListPageLoader::TYPE_BRAND);
        
        $finderStruct = $page->getHeader()->getExtension('finder');
        if($finderStruct !== null) {
            $finderStruct->set('brandCode', $page->model->getCode());
            
            if($this->settings->getConfig('showSeries')) {
                $series = $this->getEntityCollection(
                    $this->seriesRepository, 
                    [
                        'enabled'=>true,
                        'brandId'=>$page->model->getId()
                    ],
                    'label'
                );
                $finderStruct->set('series', $series);
            }

            if($this->settings->getConfig('showTypes')) {
                $types = $this->getEntityCollection(
                    $this->typeRepository, 
                    [
                        'enabled'=>true,
                        'brandId'=>$page->model->getId()
                    ],
                    'label'
                );
                $finderStruct->set('types', $types);
            }
            
            $devicesArray = $this->getDevicesArray($page->model->getId(), 0, 0, false);
            //$devicesArray = $this->getDevicesArray($page->model->getId(), -1, -1, false);
            $finderStruct->set('devices', $devicesArray);
        }
        
        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list.html.twig', ['page' => $page]);
    }
    
    /**
     * @Route("/top-finder/series/{code}", name="frontend.top_finder.series", methods={"GET"})
     */
    public function listSeries(Request $request, RequestDataBag $data, SalesChannelContext $context): Response
    {
        $page = $this->listPageLoader->load($request, $context, ListPageLoader::TYPE_SERIES);
        
        $finderStruct = $page->getHeader()->getExtension('finder');
        if($finderStruct !== null) {
            $finderStruct->set('brandCode', $page->model->getBrand()->getCode());
            $finderStruct->set('seriesCode', $page->model->getCode());
//            $finderStruct->set('typeId', $device->getTypeId());
//            $finderStruct->set('deviceCode', $device->getCode());
            
            if($this->settings->getConfig('showSeries')) {
                $series = $this->getEntityCollection(
                    $this->seriesRepository, 
                    [
                        'enabled'=>true,
                        'brandId'=>$page->model->getBrand()->getId()
                    ],
                    'label'
                );
                $finderStruct->set('series', $series);
            }

            if($this->settings->getConfig('showTypes')) {
                $types = $this->getEntityCollection(
                    $this->typeRepository, 
                    [
                        'enabled'=>true,
                        'brandId'=>$page->model->getBrand()->getId()
                    ],
                    'label'
                );
                $finderStruct->set('types', $types);
            }
            
            $devicesArray = $this->getDevicesArray(
                $page->model->getBrand()->getId(), 
                $page->model->getId(), 
                0, 
                false);
            $finderStruct->set('devices', $devicesArray);
        }
        
        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list.html.twig', ['page' => $page]);
    }
    
    /**
     * @Route("/top-finder/type/{code}", name="frontend.top_finder.type", methods={"GET"})
     */
    public function listType(Request $request, RequestDataBag $data, SalesChannelContext $context): Response
    {
        $page = $this->listPageLoader->load($request, $context, ListPageLoader::TYPE_TYPE);
        $finderStruct = $page->getHeader()->getExtension('finder');
        if($finderStruct !== null) {
            $finderStruct->set('brandCode', $page->model->getBrand()->getCode());
//            $finderStruct->set('seriesId', $page->model->getId());
            $finderStruct->set('typeCode', $page->model->getCode());
//            $finderStruct->set('deviceCode', $device->getCode());
            
            if($this->settings->getConfig('showSeries')) {
                $series = $this->getEntityCollection(
                    $this->seriesRepository, 
                    [
                        'enabled'=>true,
                        'brandId'=>$page->model->getBrand()->getId()
                    ],
                    'label'
                );
                $finderStruct->set('series', $series);
            }

            if($this->settings->getConfig('showTypes')) {
                $types = $this->getEntityCollection(
                    $this->typeRepository, 
                    [
                        'enabled'=>true,
                        'brandId'=>$page->model->getBrand()->getId()
                    ],
                    'label'
                );
                $finderStruct->set('types', $types);
            }
            
            $devicesArray = $this->getDevicesArray(
                $page->model->getBrand()->getId(), 
                0, 
                $page->model->getId(), 
                false);
            $finderStruct->set('devices', $devicesArray);
        }
        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list.html.twig', ['page' => $page]);
    }
    
    /**
     * @Route("/top-finder/compatible-devices/{productid}", name="frontend.top_finder.compatible_devices", methods={"GET", "POST"}, defaults={"productid"="", "XmlHttpRequest"=true})
     */
    public function compatibleDevices(string $productid, Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        if (!$productid) {
            return new JsonResponse(['success' => false, 'text' => 'Something went wrong']);
        }
        
        $result = ['success' => true];
        $page = $this->compatibleDevicesWidgetLoader->load($request, $salesChannelContext, $productid);
        $result['html'] = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/compatible-devices/index.html.twig', [
                'page' => $page
            ])->getContent();
        $result['title'] = $page->product->getName() . ' ' . $this->trans('topdata-topfinder.productDevices');
        
        if($result) {
            return new JsonResponse($result);
        }
    }
    
    /**
     * @Route("/top-finder/popupForm", name="frontend.top_finder.popup_form", methods={"GET", "POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function popupForm(Request $request, SalesChannelContext $salesChannelContext): Response
    {
        $page = $this->popupFormWidgetLoader->load($request, $salesChannelContext);
        $response =  $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/popup-form/index.html.twig', ['page' => $page]);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/get-popup-device", name="frontend.top_finder.popup_device", methods={"GET", "POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function popupDevice(Request $request, SalesChannelContext $salesChannelContext): Response
    {
        $page = $this->popupDeviceWidgetLoader->load($request, $salesChannelContext);
        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/popup-device/index.html.twig', ['page' => $page]);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/letter/{letter}", name="frontend.top_finder.letter", methods={"GET"})
     */
    public function listLetter(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $letter = $request->get('letter');
        if(!preg_match('/^[a-z0]{1}$/', $letter)) {
            throw new MissingRequestParameterException('letter', '/letter');
        }
        
        $page = $this->brandLetterPageLoader->load($request, $salesChanelContext, $letter);
        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/brand-letter.html.twig', ['page' => $page]);
    }
    
    /**
     * @Route("/top-finder/letter-brand/{brandCode}", name="frontend.top_finder.letter-brand", methods={"GET"})
     */
    public function listLetterBrand(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $brandCode = $request->get('brandCode');
        if(!preg_match('/^[a-z0-9_-]+$/', $brandCode)) {
            throw new MissingRequestParameterException('brandCode', '/brandCode');
        }
        
        $page = $this->oneBrandLetterPageLoader->load($request, $salesChanelContext, $brandCode);
        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/one-brand-letter.html.twig', ['page' => $page]);
    }
    
    /**
     * @Route("/top-finder/history-remove/{deviceId}", name="frontend.top_finder.history-remove", methods={"GET"})
     */
    public function historyRemove(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $deviceId = $request->get('deviceId');
        if(!EntitiesHelper::isValidUuid($deviceId)) {
            throw new MissingRequestParameterException('deviceId');
        }
        
        $referer = $request->headers->get('referer'); 
        
        $cokie = $request->cookies->get(Helper::COOKIE_DEVICE_HISTORY);

        $devicesHistory = [];

        if($cokie) {
            $devicesHistory = explode(',', $cokie);
        }

        if(!is_array($devicesHistory)) {
            $devicesHistory = [];
        }

        $historyKey = array_search($deviceId, $devicesHistory);

        if($historyKey !== false) {
            unset($devicesHistory[$historyKey]);
        }
        
        if(!$referer) {
            $referer = '/';
        }
        
        $response = new RedirectResponse($referer);
        
        $response->headers->setCookie(
            Cookie::create(
                Helper::COOKIE_DEVICE_HISTORY,
                implode(',', $devicesHistory),
                time() + 3600*24*30
            )
        );
        
        return $response;
    }
    
    /**
     * @Route("/top-finder/popup-device-info/{deviceId}", name="frontend.top_finder.popup_device_info", methods={"GET", "POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function popupDeviceInfo(Request $request, SalesChannelContext $salesChannelContext): Response
    {
        $deviceId = $request->get('deviceId');
        
        if(!EntitiesHelper::isValidUuid($deviceId)) {
            throw new MissingRequestParameterException('deviceId');
        }
        
        if(!$salesChannelContext->getCustomer() || $salesChannelContext->getCustomer()->getGuest()) {
            throw new MissingRequestParameterException('userId');
        }
        
        $userId = $salesChannelContext->getCustomer()->getId();
        
        
        $page = $this->popupDeviceInfoWidgetLoader->load($request, $salesChannelContext, $deviceId, $userId);
        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/widget/popup-device-info/index.html.twig', ['page' => $page]);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/device-info-update", name="frontend.top_finder.device_info_update", methods={"GET", "POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function popupDeviceInfoUpdate(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $deviceId = $request->get('deviceId');
        
        if(!EntitiesHelper::isValidUuid($deviceId)) {
            return new JsonResponse(['success' => false, $deviceId.' invalid']);
        }
        
        if(!$salesChannelContext->getCustomer() || $salesChannelContext->getCustomer()->getGuest()) {
            return new JsonResponse(['success' => false, 'no customer']);
        }
        
        $userId = $salesChannelContext->getCustomer()->getId();
        $data = $request->get('formData');
        
        $connection = $this->container->get('Doctrine\DBAL\Connection');
        $deviceToCustomerId = Helper::deviceToCustomerId(
                $deviceId, 
                $userId, 
                $connection
            );
        
        if(!$deviceToCustomerId) {
            return new JsonResponse(['success' => false, 'no device to customer']);
        }
        
        $this->deviceCustomerRepository->update([
            [
                'id' => $deviceToCustomerId,
                'extraInfo' => json_encode($data)
            ]
        ], $salesChannelContext->getContext());
        
        return new JsonResponse([
            'success' => true
        ]);
    }
    
    /**
     * @Route("/top-finder/devicelist-find", name="frontend.top_finder.devicelist-find", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     *
     * @throws CustomerNotLoggedInException
     */
    public function deviceListFind(Request $request, SalesChannelContext $context): Response
    {
        $page = $this->deviceListPageLoader->load($request, $context);
        
        $response = $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/account/device-list-find.html.twig', ['page' => $page]);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/list-products", name="frontend.top_finder.list-products", methods={"GET"})
     */
    public function listProducts(Request $request, SalesChannelContext $salesChanelContext): Response
    {
        $productIds = explode('-', $request->get('ids'));
        
        foreach ($productIds as $key=>$id) {
            if(Uuid::isValid((string)$id) === false) {
                unset($productIds[$key]);
            }
        }
        if(!count($productIds)) {
            throw new MissingRequestParameterException('ids');
        }
        
        $page = $this->listProductsLoader->load($request, $salesChanelContext, $productIds);
        return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list-products.html.twig', ['page' => $page]);
    }
    
    /**
     * @Route("/top-finder/get-popup-device-new", name="frontend.top_finder.popup_device_new", methods={"GET", "POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function popupDeviceNew(Request $request, SalesChannelContext $salesChannelContext): JsonResponse
    {
        $page = $this
            ->popupDeviceWidgetLoader
            ->loadNew($request, $salesChannelContext);
        
        $result = ['success' => true];

        if(!$page->device) return new JsonResponse(['status'=>'false','message'=>"no Device Found"], 500);
        
        $result['deviceId'] = $page->device->getId();
        $result['deviceUrl'] = $page->deviceUrl;
        $result['inDeviceList'] = $page->device->getInDeviceList();
        $result['userLogged'] = $page->other['userLogged'];
        if($result['userLogged']) {
            $result['pathListAdd'] = $page->other['pathListAdd'];
            $result['pathListRemove'] = $page->other['pathListRemove'];
        }
        
//        $devices = $this->getDevicesArray(
//            $page->device->getBrand()->getId(),
//            $page->device->getSeries() ? $page->device->getSeries()->getId() : 0, 
//            0
//            //$page->device->getType() ? $page->device->getType()->getId() : false
//        );
//        
//        $series = $this->getSeriesArray($page->device->getBrand()->getId());
//        $types = $this->getTypesArray($page->device->getBrand()->getId());
        
        
//        $popupHome = [
//            'brandskey' => $page->device->getBrand()->getCode(),
//            'series' => $series,
//            'serieskey' => $page->device->getSeries() ? $page->device->getSeries()->getCode() : null,
//            'types' => $types,
//            'typeskey' => $page->device->getType() ? $page->device->getType()->getCode() : null,
//            'devices' => $devices,
//            'deviceskey' => $page->device->getCode()
//        ];
//        
//        $result['popupHome'] = $popupHome;
        
        $connection = $this->container->get('Doctrine\DBAL\Connection');
        $devicelist = Helper::getDeviceListInfo($salesChannelContext, $connection);
        
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/box-device.html.twig', 
                [
                    'page' => $page,
                    'devicelist' => $devicelist
                ]
            )
            ->getContent();
        
        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    
    /**
     * @Route("/top-finder/popup-letter/{letter}", name="frontend.top_finder.popup_letter", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function listPopupLetter(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $letter = $request->get('letter');
        if(!preg_match('/^[a-z0]{1}$/', $letter)) {
            throw new MissingRequestParameterException('letter', '/letter');
        }
        
        $page = $this->brandLetterPageLoader->loadJson($request, $salesChanelContext, $letter);
        $result = [];
        $result['success'] = true;
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-letter.html.twig', 
                ['page' => $page]
            )
            ->getContent();
        
        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/popup-all-brands", name="frontend.top_finder.popup_all_brands", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function listPopupAllBrands(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $page = $this->listPageLoader->loadJsonBrands($request, $salesChanelContext);
        //return $this->renderStorefront('@TopdataTopFinderProSW6/storefront/page/topfinder/list.html.twig', ['page' => $page]);
//        $page = $this->brandLetterPageLoader->loadJson($request, $salesChanelContext, $letter);
        $result = [];
        $result['success'] = true;
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/all-brands.html.twig', 
                ['page' => $page]
            )
            ->getContent();
        
        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/popup-brand/{brandCode}", name="frontend.top_finder.popup_brand", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function listPopupBrand(Request $request, SalesChannelContext $salesChanelContext)
    {
        $brandCode = $request->get('brandCode');
        if(!preg_match('/^[a-z0-9_-]+$/', $brandCode)) {
            throw new MissingRequestParameterException('brandCode', '/brandCode');
        }
        
        $page = $this->oneBrandLetterPageLoader->loadJson(
            $request, 
            $salesChanelContext, 
            $brandCode,
            'all'
        );
        
        $offset = $request->get('offset');
        $defaultOffset = self::POPUP_DEVICES_PRELOAD_LENGTH;
        if($offset) {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/_listed-devices.html.twig', 
                    ['page' => $page, 'offset' => $offset]
                );
        }
        else {
            $listMode = $request->cookies->get('topfinder-popup-devicelist-mode');
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-devices.html.twig', 
                    [
                        'page' => $page, 
                        'mode'=>'all',
                        'listMode' => $listMode,
                        'offset' => $defaultOffset
                    ]
                );
            $response->headers->set("returnPath", $this->getReturnPath($request));
            $response->headers->set("loadOffset", (string)$defaultOffset);
        }
        
//        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/popup-brand-series/{brandCode}", name="frontend.top_finder.popup_brand_series", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function listPopupBrandSeries(Request $request, SalesChannelContext $salesChanelContext)
    {
        $brandCode = $request->get('brandCode');
        if(!preg_match('/^[a-z0-9_-]+$/', $brandCode)) {
            throw new MissingRequestParameterException('brandCode', '/brandCode');
        }
        
        $page = $this->oneBrandLetterPageLoader->loadJson(
            $request, 
            $salesChanelContext, 
            $brandCode,
            'series'
        );
        
        $listMode = $request->cookies->get('topfinder-popup-devicelist-mode');        
        $offset = $request->get('offset');
        $defaultOffset = self::POPUP_DEVICES_PRELOAD_LENGTH;
        if($offset) {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/_listed-series-devices.html.twig', 
                    ['page' => $page, 'offset' => $offset, 'hideMenuSeries'=>true]
                );
        }
        else {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-devices.html.twig', 
                    [
                        'page' => $page, 
                        'mode'=>'series',
                        'listMode' => $listMode,
                        'offset' => $defaultOffset, 
                        'hideMenuSeries'=>true
                    ]
                );
            $response->headers->set("returnPath", $this->getReturnPath($request));
            $response->headers->set("loadOffset", (string)$defaultOffset);
        }
        
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/popup-brand-types/{brandCode}", name="frontend.top_finder.popup_brand_types", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function listPopupBrandTypes(Request $request, SalesChannelContext $salesChanelContext)
    {
        $brandCode = $request->get('brandCode');
        if(!preg_match('/^[a-z0-9_-]+$/', $brandCode)) {
            throw new MissingRequestParameterException('brandCode', '/brandCode');
        }
        
        $page = $this->oneBrandLetterPageLoader->loadJson(
            $request, 
            $salesChanelContext, 
            $brandCode,
            'types'
        );
        
        $listMode = $request->cookies->get('topfinder-popup-devicelist-mode');        
        $offset = $request->get('offset');
        $defaultOffset = self::POPUP_DEVICES_PRELOAD_LENGTH;
        if($offset) {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/_listed-types-devices.html.twig', 
                    ['page' => $page, 'offset' => $offset]
                );
        }
        else {
            $response = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-devices.html.twig', 
                    [
                        'page' => $page, 
                        'mode'=>'types',
                        'listMode' => $listMode,
                        'offset' => $defaultOffset
                    ]
                );
            
            $response->headers->set("returnPath", $this->getReturnPath($request));
            $response->headers->set("loadOffset", (string)$defaultOffset);
        }
        
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    
    /**
     * @Route("/top-finder/popup-brandq/{code}", name="frontend.top_finder.popup_brandq", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function listPopupBrandDevices(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $code = $request->get('code');
        if(!preg_match('/^[a-z0-9_-]+$/', $code)) {
            throw new MissingRequestParameterException('code', '/code');
        }
        
//        $page = $this->listPageLoader->loadBrandDevicesJson($request, $salesChanelContext);
        
        $page = $this->oneBrandLetterPageLoader->loadJson(
            $request, 
            $salesChanelContext, 
            $code,
            'all'
        );
        
        $listMode = $request->cookies->get('topfinder-popup-devicelist-mode');
        
        $result = [];
        $result['success'] = true;
        $result['returnPath'] = $this->getReturnPath($request);
        
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/brand-devices.html.twig', 
                ['page' => $page, 'mode'=>'all', 'listMode' => $listMode]
            )
            ->getContent();
        
        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/popup-series/{code}", name="frontend.top_finder.popup_series", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function listPopupSerieDevices(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $code = $request->get('code');
        if(!preg_match('/^[a-z0-9_-]+$/', $code)) {
            throw new MissingRequestParameterException('code', '/code');
        }
        $page = $this->listPageLoader->loadSerieDevicesJson($request, $salesChanelContext);
        
        $listMode = $request->cookies->get('topfinder-popup-devicelist-mode');
        
        $result = [];
        $result['success'] = true;
        $result['returnPath'] = $this->getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devices.html.twig', 
                ['page' => $page, 'listMode' => $listMode, 'hideMenuSeries' => true]
            )
            ->getContent();
        
        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/popup-type/{code}", name="frontend.top_finder.popup_type", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function listPopupTypeDevices(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $code = $request->get('code');
        if(!preg_match('/^[a-z0-9_-]+$/', $code)) {
            throw new MissingRequestParameterException('code', '/code');
        }
        $page = $this->listPageLoader->loadTypeDevicesJson($request, $salesChanelContext);
        
        $listMode = $request->cookies->get('topfinder-popup-devicelist-mode');
        
        $result = [];
        $result['success'] = true;
        $result['returnPath'] = $this->getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devices.html.twig', 
                ['page' => $page, 'listMode' => $listMode]
            )
            ->getContent();
        
        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/popup-devicelist", name="frontend.top_finder.popup_devicelist", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function popupDeviceList(Request $request, SalesChannelContext $context): JsonResponse
    {
        $page = $this->deviceListPageLoader->loadJson($request, $context);
        $listMode = $request->cookies->get('topfinder-popup-devicelist-mode');
        $result = [];
        $result['success'] = true;
        $result['returnPath'] = $this->getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/index.html.twig', 
                ['page' => $page, 'listMode' => $listMode]
            )
            ->getContent();
        
        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/popup-devicelist-device", name="frontend.top_finder.popup_devicelist_device", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function popupDeviceListDevice(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');
        
        if(false === Uuid::isValid($id)) {
            return new JsonResponse(['success'=>false]);
        }
        
        $page = $this->deviceListPageLoader->loadDeviceJson($request, $context);
        
        if(is_null($page->device)) {
            return new JsonResponse(['success'=>false]);
        }
        
        $result = [];
        $result['success'] = true;
        $result['returnPath'] = $this->getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/device.html.twig', 
                ['page' => $page]
            )
            ->getContent();
        
        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/popup-devicelist-subdevice-add", name="frontend.top_finder.popup_devicelist_subdevice_add", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function popupDeviceListDeviceSubdeviceAdd(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');
        
        if(false === Uuid::isValid($id)) {
            return new JsonResponse(['success'=>false]);
        }
        
        $page = $this->deviceListPageLoader->loadAddSubdeviceJson($request, $context);
        
        if(is_null($page->device)) {
            return new JsonResponse(['success'=>false]);
        }
        
        $result = [];
        $result['success'] = true;
        $result['returnPath'] = $this->getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/subdevice-add.html.twig', 
                ['page' => $page]
            )
            ->getContent();
        
        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/popup-devicelist-subdevice-add-confirmed", name="frontend.top_finder.popup_devicelist_subdevice_add_confirmed", methods={"POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function popupDeviceListDeviceSubdeviceAddConfirmed(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');
        
        if(false === Uuid::isValid($id)) {
            return new JsonResponse(['success'=>false]);
        }
        
        $page = $this->deviceListPageLoader->addSubdeviceJson($request, $context);
        
        if(is_null($page->device)) {
            return new JsonResponse(['success'=>false]);
        }
        
        $result = [];
        $result['success'] = true;
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/device.html.twig', 
                ['page' => $page]
            )
            ->getContent();
        
        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    
    /**
     * @Route("/top-finder/popup-devicelist-subdevice-edit", name="frontend.top_finder.popup_devicelist_subdevice_edit", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function popupDeviceListDeviceSubdeviceEdit(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');
        
        if(false === Uuid::isValid($id)) {
            return new JsonResponse(['success'=>false]);
        }
        
        $page = $this->deviceListPageLoader->loadEditSubdeviceJson($request, $context);
        
        if(is_null($page->device)) {
            return new JsonResponse(['success'=>false]);
        }
        
        $result = [];
        $result['success'] = true;
        $result['returnPath'] = $this->getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/subdevice-edit.html.twig', 
                ['page' => $page]
            )
            ->getContent();
        
        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    /**
     * @Route("/top-finder/popup-devicelist-subdevice-edit-confirmed", name="frontend.top_finder.popup_devicelist_subdevice_edit_confirmed", methods={"POST"}, defaults={"XmlHttpRequest"=true})
     */
    public function popupDeviceListDeviceSubdeviceEditConfirmed(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');
        
        if(false === Uuid::isValid($id)) {
            return new JsonResponse(['success'=>false]);
        }
        
        $page = $this->deviceListPageLoader->editSubdeviceJson($request, $context);
        
        if(is_null($page->device)) {
            return new JsonResponse(['success'=>false]);
        }
        
        $result = [];
        $result['success'] = true;
        $result['html'] = $this
            ->renderStorefront(
//                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/device.html.twig', 
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/subdevice-edit.html.twig', 
                ['page' => $page]
            )
            ->getContent();
        
        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    
    /**
     * @Route("/top-finder/popup-devicelist-subdevice-remove", name="frontend.top_finder.popup_devicelist_subdevice_remove", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function popupDeviceListDeviceSubdeviceRemove(Request $request, SalesChannelContext $context): JsonResponse
    {
        $id = $request->get('id');
        
        if(false === Uuid::isValid($id)) {
            return new JsonResponse(['success'=>false]);
        }
        
        $page = $this->deviceListPageLoader->removeSubdeviceJson($request, $context);
        
//        if(is_null($page->device)) {
//            return new JsonResponse(['success'=>false]);
//        }
        
        $result = [];
        $result['success'] = true;
        if($page->fallBack) {
            $result['html'] = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/index.html.twig', 
                    ['page' => $page]
                )
                ->getContent();
        }
        else {
            $result['html'] = $this
                ->renderStorefront(
                    '@TopdataTopFinderProSW6/storefront/widget/popup-form/devicelist/device.html.twig', 
                    ['page' => $page]
                )
                ->getContent();
        }
        
        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    
    private function getReturnPath($request)
    {
        $return = $request->getPathInfo();
        if($request->getQueryString()) {
            $return .= '?'.$request->getQueryString();
        }
        return $return;
    }
    
    
    /**
     * @Route("/top-finder/popup-history", name="frontend.top_finder.popup_history", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function listPopupHistory(Request $request, SalesChannelContext $salesChanelContext): JsonResponse
    {
        $page = $this->listPageLoader->loadHistoryDevicesJson($request, $salesChanelContext);
        
        $listMode = $request->cookies->get('topfinder-popup-devicelist-mode');
        
        $result = [];
        $result['success'] = true;
        $result['returnPath'] = $this->getReturnPath($request);
        $result['html'] = $this
            ->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devices.html.twig', 
                ['page' => $page, 'listMode' => $listMode]
            )
            ->getContent();
        
        $response = new JsonResponse($result);
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        return $response;
    }
    
    
    /**
     * @Route("/top-finder/popup-search", name="frontend.top_finder.popup_search", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function listPopupSearch(Request $request, SalesChannelContext $salesChanelContext)
    {
        $page = $this->listPageLoader->loadSearchDevicesJson($request, $salesChanelContext);
        $listMode = $request->cookies->get('topfinder-popup-devicelist-mode');
        $offset = $request->query->get('offset');
        $defaultOffset = self::POPUP_DEVICES_PRELOAD_LENGTH;
        
        if($offset) {
            $response = $this->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/_listed-devices.html.twig', 
                ['page' => $page]
            );
        }
        else {
            $response = $this->renderStorefront(
                '@TopdataTopFinderProSW6/storefront/widget/popup-form/devices.html.twig', 
                ['page' => $page, 'listMode' => $listMode]
            );
            $response->headers->set("loadOffset", (string)$defaultOffset);
        }
        
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
        $response->headers->set("returnPath", $this->getReturnPath($request));
        return $response;
    }
    
    /**
     * @Route("/top-finder/popup-search-suggest", name="frontend.top_finder.popup_search_suggest", methods={"GET"}, defaults={"XmlHttpRequest"=true})
     */
    public function suggestPopupSearch(Request $request, SalesChannelContext $salesChanelContext)
    {
        $page = $this->listPageLoader->loadSearchDevicesJson($request, $salesChanelContext);
       
        $response = $this->renderStorefront(
            '@TopdataTopFinderProSW6/storefront/widget/popup-form/suggest-devices.html.twig', 
            ['page' => $page]
        );
        
        $response->headers->set("X-Robots-Tag", "noindex, nofollow");
//        $response->headers->set("returnPath", $this->getReturnPath($request));
        return $response;
    }
}