<?php declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Component;

use Topdata\TopdataTopFinderProSW6\Component\SettingsService;
use Doctrine\DBAL\Connection;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Symfony\Component\HttpFoundation\Request;
use Shopware\Core\Framework\Context;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Sorting\FieldSorting;
use Shopware\Core\Framework\Uuid\Uuid;
use Topdata\TopdataConnectorSW6\Core\Content\Device\Agregate\DeviceCustomer\DeviceCustomerEntity;

class Helper
{
    const COOKIE_DEVICE_HISTORY = 'topdata-device-history';
    const COOKIE_DEVICE_SELECTBOXES = 'topdata-device-selectboxes';
    const COOKIE_DEVICE_SLIDER = 'topdata-device-slider';
    const COOKIE_FINDER_SWITCH = 'topdata-switch';
    
    /**
     * @var type Connection
     */
    protected $connection;
    
    /**
     * @var type SettingsService
     */
    protected $settingsService;
    
    /**
     * @var type EntityRepositoryInterface
     */
    protected $deviceRepository;

    public function __construct(
        Connection $connection,
        SettingsService $settingsService,
        EntityRepository $deviceRepository
    )
    {
        $this->connection = $connection;
        $this->settingsService = $settingsService;
        $this->deviceRepository = $deviceRepository;
    }
    
    public function getDeviceHistory(Request $request, Context $context) : array
    {
        $cokie = $request->cookies->get(self::COOKIE_DEVICE_HISTORY);

        $devicesHistory = [];

        if($cokie) {
            $devicesHistory = explode(',', $cokie);
        }

        if(!is_array($devicesHistory)) {
            $devicesHistory = [];
        }
        
        if(count($devicesHistory)) {
            $criteria = (new Criteria(array_reverse($devicesHistory)))
                ->addFilter(new EqualsFilter('enabled', true))
                ->addAssociations(['brand', 'media']);
            $devicesHistory = $this->deviceRepository->search($criteria, $context)->getElements();
        }

        return $devicesHistory;
    }
    
    /**
     * [['id'=>hexid, 'name'=>brandname], ...]
     */
    public function getBrands() : array
    {
        return $this->connection->createQueryBuilder()
            ->select ('LOWER(HEX(id)) as id, label as name')
            ->from('topdata_brand')
            ->where('is_enabled = 1' )
            ->orderBy('sort DESC, label')
            ->execute()
            ->fetchAllAssociative();
    }
    
    
    public static function deviceToCustomerId(
        string $deviceId, 
        string $customerId, 
        Connection $connection
    ) : ?string
    {
        if(!Uuid::isValid($deviceId)) {
            return null;
        }
        
        if(!Uuid::isValid($customerId)) {
            return null;
        }
        
        return (string) $connection->executeQuery('
            SELECT LOWER(HEX(id)) as id
                FROM topdata_device_to_customer
                WHERE (0x'.$deviceId.' = device_id) AND (0x'.$customerId.' = customer_id)
                LIMIT 1
            ')->fetchOne();
    }
    
    
    public function getDeviceList(SalesChannelContext $salesChannelContext)
    {
        $return = [];
        $deviceIds = [];
        if($salesChannelContext->getCustomer() && !($salesChannelContext->getCustomer()->getGuest())) {
            $deviceIdsAssoc =  $this->connection->createQueryBuilder()
                ->select ('LOWER(HEX(device_id)) as device_id')
                ->from('topdata_device_to_customer')
                ->where('customer_id = 0x' . $salesChannelContext->getCustomer()->getId() )
                ->execute()
                ->fetchAllAssociative();
            foreach ($deviceIdsAssoc as $val) {
                $deviceIds[] = $val['device_id'];
            }
        }
        if($deviceIds) {
            $criteria = (new Criteria($deviceIds))
                    ->addFilter(new EqualsFilter('enabled', true))
                    ->addAssociations(['media','brand','series','type'])
                    ->addSorting(new FieldSorting('model', FieldSorting::ASCENDING));

            $devices = $this->deviceRepository->search($criteria, $salesChannelContext->getContext());
            if($devices) {
                $return = $devices->getEntities();
                
            }
        }
        
        return $return;
    }
    
    public function getDeviceListArray(SalesChannelContext $salesChannelContext) : array
    {
        $return = [];
        $deviceIds = [];
        if($salesChannelContext->getCustomer() && !($salesChannelContext->getCustomer()->getGuest())) {
            $deviceIdsAssoc =  $this->connection->createQueryBuilder()
                ->select ('LOWER(HEX(device_id)) as device_id')
                ->from('topdata_device_to_customer')
                ->where('customer_id = 0x' . $salesChannelContext->getCustomer()->getId() )
                ->execute()
                ->fetchAllAssociative();
            foreach ($deviceIdsAssoc as $val) {
                $deviceIds[] = $val['device_id'];
            }
        }
        if($deviceIds) {
            $criteria = (new Criteria($deviceIds))
                    ->addFilter(new EqualsFilter('enabled', true))
                    ->addAssociations(['media','brand','series','type'])
                    ->addSorting(new FieldSorting('brand.name', FieldSorting::ASCENDING))
                    ->addSorting(new FieldSorting('model', FieldSorting::ASCENDING));

            $devices = $this->deviceRepository->search($criteria, $salesChannelContext->getContext())->getEntities();
            
            foreach ($devices as $device) {
                $return[$device->getId()] = [
                    'id'=>$device->getId(),
                    'model'=>$device->getModel(),
                    'code'=>$device->getCode(),
                    'brand'=>[
                        'id'=>$device->getBrand()->getId(),
                        'code'=>$device->getBrand()->getCode(),
                        'name'=>$device->getBrand()->getName(),
                    ]
                ];
            }
        }
        
        return $return;
    }
    
    
    public static function getDeviceListInfo(SalesChannelContext $salesChannelContext, $connection) : array
    {
        $return = [];
        
        if(!$salesChannelContext->getCustomer() || $salesChannelContext->getCustomer()->getGuest()) {
            return $return;
        }
            
        $rez = $connection
            ->createQueryBuilder()
            ->select ('LOWER(HEX(device_id)) as device_id, extra_info')
            ->from('topdata_device_to_customer')
            ->where('customer_id = 0x' . $salesChannelContext->getCustomer()->getId() )
            //')AND(is_dealer_managed=0)'
            ->execute()
            ->fetchAllAssociative();
        
        foreach ($rez as $val) {
            $return[$val['device_id']] = $val['extra_info'] ? json_decode($val['extra_info'], true) : DeviceCustomerEntity::defaultExtraInfo();
        }
        
        return $return;
    }
    
    
    public static function filterTerm(string $term, int $minLength = 3, int $maxLength = 25) : string
    {
        $term = str_replace(['%', '\'', '"'], '', $term);
        $term = str_replace(['_', '/', '\\'], ' ', $term);
        $term = trim($term);
        
        $term = mb_substr($term, 0, $maxLength);
        
        return (mb_strlen($term) < $minLength) ? '' : $term;
    }
    
    
    public static function getTermsFromString(string $string, int $min=2, int $max=10, int $maxCount=4) : array
    {
        $rez = [];
        $string = str_replace(['-', '/', '+', '&', '.', ','], ' ', $string);
        $words = explode(' ', $string);
        if(count($words)>1) {
            $firstWord = $words[0];
            unset($words[0]);
            $words = array_reverse($words);
            array_unshift($words, $firstWord);
        }
        
        foreach ($words as $word) {
            if(mb_strlen(trim($word))>=$min) {
                $rez[] = mb_substr(trim($word),0,$max);
                if(count($rez)>=$maxCount) {
                    break;
                }
            }
        }
        return $rez;
    }
    
    
    public static function findDevices(string $term, Connection $connection, int $limit = 0) : array
    {
        $filteredTerm = static::filterTerm($term);
        if($filteredTerm !== $term) {
            $term = $filteredTerm;
        }
        
        if(!$term) {
            return [];
        }
        
        $SQLadd =  ($limit>0) ? " LIMIT $limit" : '';
        
        $terms = static::getTermsFromString($term);
        
        $WhereAdd = '';
        if(count($terms)) {
            $WhereAdd = 'OR(';
            foreach ($terms as $key=>$value) {
                $terms[$key] = '(topdata_device.keywords LIKE "%'.$value.'%")';
            }
            $WhereAdd .= implode('AND', $terms);
            $WhereAdd .= ')';
        }
        
        $devices = $connection->fetchAllAssociative('
SELECT topdata_device.model as name, 
       LOWER(HEX(topdata_device.id)) as id, 
       LOWER(HEX(topdata_device.brand_id)) as brand_id, 
       LOWER(HEX(topdata_device.series_id)) as series_id, 
       LOWER(HEX(topdata_device.type_id)) as type_id, 
       topdata_device.code as code, 
       topdata_brand.label as brand_name,
       topdata_brand.code as brand_code,
       topdata_series.code as series_code,
       LOWER(HEX(topdata_device.media_id)) as media_id,
       topdata_series.label as series_name,
       topdata_device_type.label as type_name
  FROM topdata_device 
  LEFT JOIN topdata_brand ON (topdata_brand.id=topdata_device.brand_id)
  LEFT JOIN topdata_series ON (topdata_series.id=topdata_device.series_id)
  LEFT JOIN topdata_device_type ON (topdata_device_type.id=topdata_device.type_id)
  WHERE (topdata_device.is_enabled=1)AND((topdata_device.keywords LIKE "%'.$term.'%")'.$WhereAdd.')
        ORDER BY `topdata_device`.`code` ASC '.$SQLadd);
        
        return count($devices) ? $devices : [];
    }
    
    
    public static function countFindDevices(string $term, Connection $connection) : int
    {
        $filteredTerm = static::filterTerm($term);
        if($filteredTerm !== $term) {
            $term = $filteredTerm;
        }
        
        if(!$term) {
            return 0;
        }
        
        $terms = static::getTermsFromString($term);
        
        $WhereAdd = '';
        if(count($terms)) {
            $WhereAdd = 'OR(';
            foreach ($terms as $key=>$value) {
                $terms[$key] = '(topdata_device.keywords LIKE "%'.$value.'%")';
            }
            $WhereAdd .= implode('AND', $terms);
            $WhereAdd .= ')';
        }
        
        $devicesCount = $connection->fetchAllAssociative('
SELECT COUNT(*) as cnt
  FROM topdata_device 
  WHERE (topdata_device.is_enabled=1)AND((topdata_device.keywords LIKE "%'.$term.'%")'.$WhereAdd.')
            ');
        
        return isset($devicesCount[0]['cnt']) ? (int)$devicesCount[0]['cnt'] : 0;
    }
}