<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\Popup;

use Psr\Log\LoggerInterface;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataTopFinderProSW6\Component\BreadcrumbsTrait;
use Topdata\TopdataTopFinderProSW6\Service\TopdataDeviceDataService;
use Topdata\TopdataTopFinderProSW6\Service\SeriesDataService;
use Topdata\TopdataTopFinderProSW6\Service\SettingsService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\Topfinder\MyList\ListPage;
use Topdata\TopdataTopFinderProSW6\Storefront\PageLoader\Topfinder\MyList\TRASH____ListPageLoader;

/**
 * Class SeriesListPageLoader
 *
 * This class is responsible for loading a specific series page.
 * It retrieves series data and devices based on the series code from the request.
 */
class SeriesListPageLoader
{
    /**
     * @readonly
     */
    private GenericPageLoader $genericPageLoader;
    /**
     * @readonly
     */
    private SeriesDataService $seriesDataService;
    /**
     * @readonly
     */
    private TopdataDeviceDataService $deviceDataService;
    /**
     * @readonly
     */
    private SettingsService $settingsService;
    /**
     * @readonly
     */
    private Translator $translator;
    /**
     * @readonly
     */
    private UrlGeneratorInterface $router;
    /**
     * @readonly
     */
    private LoggerInterface $logger;
    use BreadcrumbsTrait;

    public function __construct(GenericPageLoader        $genericPageLoader, SeriesDataService        $seriesDataService, TopdataDeviceDataService $deviceDataService, SettingsService          $settingsService, Translator               $translator, UrlGeneratorInterface    $router, LoggerInterface          $logger)
    {
        $this->genericPageLoader = $genericPageLoader;
        $this->seriesDataService = $seriesDataService;
        $this->deviceDataService = $deviceDataService;
        $this->settingsService = $settingsService;
        $this->translator = $translator;
        $this->router = $router;
        $this->logger = $logger;
    }

    /**
     * Load the series page for a specific series code
     *
     * This method loads the generic page, retrieves the series data,
     * and configures the page with appropriate devices, metadata, and breadcrumbs.
     *
     * @param Request $request The current request
     * @param SalesChannelContext $salesChannelContext The sales channel context
     * @return ListPage The loaded series page
     * @throws \Exception If an error occurs during loading
     */
    public function load(Request $request, SalesChannelContext $salesChannelContext): ListPage
    {
        try {
            // Get the series code from the request
            $seriesCode = $request->attributes->get('code');

            // Load the generic page
            $page = $this->genericPageLoader->load($request, $salesChannelContext);

            // Create the list page from the generic page
            $listPage = new ListPage();
            // Copy all properties from the generic page to the list page
            foreach (get_object_vars($page) as $property => $value) {
                $listPage->$property = $value;
            }

            // Set the page type to series
            $listPage->setPageType(TRASH____ListPageLoader::TYPE_SERIES);

            // Get the series by code
            $series = $this->seriesDataService->getSeriesByCode($seriesCode, $salesChannelContext);

            // If the series is not found, throw an exception
            if (!$series) {
                $this->logger->error('Series not found', ['code' => $seriesCode]);
                throw new \Exception('Series not found: ' . $seriesCode);
            }

            // Set the series model to the page
            $listPage->model = $series;

            // Get the associated brand
            $brand = $series->getBrand();

            // If the brand is not loaded, log a warning
            if (!$brand) {
                $this->logger->warning('Brand association not loaded for series', ['seriesCode' => $seriesCode]);
                throw new \Exception('Brand association not loaded for series: ' . $seriesCode);
            }

            // Get devices for this series
            $devices = $this->deviceDataService->getDevicesArray($brand->getCode(), $series->getId());

            // Set devices to the page
            $listPage->setDevices($devices);

            // Set the page to show devices
            $listPage->setShow(['devices' => true]);

            // Set the SEO title for the series page
            $listPage->setTitle($this->translator->trans('topdata-topfinder.SEO.seriesPageTitle', [
                '%brand%'  => $brand->getName(),
                '%series%' => $series->getLabel(),
            ]));

            // Set the SEO meta information
            // Set the SEO meta information
            // Note: We're logging a warning because we're not sure if the meta information methods
            // are available on the ListPage class. This needs to be verified in a real implementation.
            try {
                if (method_exists($listPage, 'getMetaInformation')) {
                    // Set the SEO meta title for the series page
                    $listPage->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.seriesMetaTitle', [
                        '%brand%'  => $brand->getName(),
                        '%series%' => $series->getLabel(),
                    ]));

                    // Set the SEO meta description for the series page
                    $listPage->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.seriesMetaDescription', [
                        '%brand%'  => $brand->getName(),
                        '%series%' => $series->getLabel(),
                    ]));

                    // Set the SEO robots meta tag for the series page
                    $listPage->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.seriesMetaRobots'));
                } else {
                    $this->logger->warning('Meta information methods not available on ListPage class');
                }
            } catch (\Exception $e) {
                $this->logger->warning('Error setting meta information: ' . $e->getMessage());
            }

            // Add breadcrumbs for the series page
            $listPage->addBreadcrumb(
                $this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'),
                $this->router->generate('frontend.top-finder.brands')
            );
            $listPage->addBreadcrumb(
                $this->translator->trans('topdata-topfinder.SEO.brandPageTitle', ['%brand%' => $brand->getName()]),
                $this->router->generate('frontend.top-finder.brandq', ['code' => $brand->getCode()])
            );
            $listPage->addBreadcrumb(
                $listPage->getTitle(),
                $this->router->generate('frontend.top-finder.series', ['code' => $series->getCode()])
            );

            // Apply compact mode limit if configured
            if ($this->settingsService->getInt('searchCompactLimit', true)) {
                $listPage->setCompactModeLimit($this->settingsService->getInt('searchCompactLimit'));
            }

            // If devices are shown, load their content
            if (isset($listPage->getDevices()[0])) {
                $devices = $listPage->getDevices();
                $this->deviceDataService->loadDeviceContent($devices, $salesChannelContext, 'series');
                $listPage->setDevices($devices);
            }

            return $listPage;
        } catch (\Exception $e) {
            $this->logger->error('Error loading series page: ' . $e->getMessage(), [
                'exception'  => $e,
                'seriesCode' => $request->attributes->get('code')
            ]);
            throw $e;
        }
    }
}