<?php

declare(strict_types=1);

namespace Topdata\TopdataTopFinderProSW6\Storefront\PageLoader;

use Doctrine\DBAL\Connection;
use Psr\Log\LoggerInterface;
use Shopware\Core\Checkout\Customer\CustomerEntity;
use Shopware\Core\Content\Media\DataAbstractionLayer\MediaRepositoryDecorator;
use Shopware\Core\Framework\Adapter\Translation\Translator;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Page\GenericPageLoader;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Topdata\TopdataConnectorSW6\Core\Content\Device\Agregate\DeviceCustomer\DeviceCustomerEntity;
use Topdata\TopdataTopFinderProSW6\Service\DeviceToCustomerService;
use Topdata\TopdataTopFinderProSW6\Storefront\Page\OneBrandLetter\OneBrandLetterPage;


/**
 * This class is responsible for loading the data required for the OneBrandLetter page.
 * It fetches brand details, devices associated with the brand, and prepares the page for display.
 */
class OneBrandLetterPageLoader
{

    /**
     * @readonly
     */
    private GenericPageLoader $genericLoader;
    /**
     * @readonly
     */
    private Connection $connection;
    /**
     * @readonly
     */
    private UrlGeneratorInterface $router;
    /**
     * @readonly
     * @var Translator $translator
     */
    private $translator;
    /**
     * @readonly
     * @var EntityRepository|MediaRepositoryDecorator $mediaRepository
     */
    private $mediaRepository;
    /**
     * @readonly
     */
    private LoggerInterface $logger;
    /**
     * @readonly
     */
    private DeviceToCustomerService $deviceToCustomerService;
    public function __construct(
        GenericPageLoader       $genericLoader,
        Connection              $connection,
        UrlGeneratorInterface   $router,
        $translator,
        $mediaRepository,
        LoggerInterface         $logger,
        DeviceToCustomerService $deviceToCustomerService
    )
    {
        $this->genericLoader = $genericLoader;
        $this->connection = $connection;
        $this->router = $router;
        $this->translator = $translator;
        $this->mediaRepository = $mediaRepository;
        $this->logger = $logger;
        $this->deviceToCustomerService = $deviceToCustomerService;
    }

    /**
     * Loads the OneBrandLetterPage with the necessary data.
     *
     * @param Request $request The current HTTP request.
     * @param SalesChannelContext $salesChannelContext The current sales channel context.
     * @param string $brandCode The code of the brand to load.
     * @return OneBrandLetterPage The populated OneBrandLetterPage.
     * @throws \Exception
     */
    public function load(Request $request, SalesChannelContext $salesChannelContext, string $brandCode): OneBrandLetterPage
    {
        // ---- Load generic page
        $page = $this->genericLoader->load($request, $salesChannelContext);
        $page = OneBrandLetterPage::createFrom($page);

        // ---- Set brand and device elements
        $this->setElements($page, $brandCode);

        // ---- Set SEO information if not an AJAX request
        if ($request->isXmlHttpRequest() === false) {
            $page->setTitle($this->translator->trans('topdata-topfinder.SEO.brandPageTitle', [
                '%brand%' => $page->brand['name'],
            ]));

            $page->getMetaInformation()->setMetaTitle($this->translator->trans('topdata-topfinder.SEO.brandMetaTitle', [
                '%brand%' => $page->brand['name'],
            ]));

            $page->getMetaInformation()->setMetaDescription($this->translator->trans('topdata-topfinder.SEO.brandMetaDescription', [
                '%brand%' => $page->brand['name'],
            ]));

            $page->getMetaInformation()->setRobots($this->translator->trans('topdata-topfinder.SEO.brandMetaRobots'));

            $page->addBreadcrumb(
                $this->translator->trans('topdata-topfinder.SEO.brandsPageTitle'),
                $this->router->generate('frontend.top-finder.brands')
            );

            $page->addBreadcrumb(
                $page->getTitle(),
                $this->router->generate('frontend.top-finder.show-brand', ['brandCode' => $page->brand['code']])
            );
        }

        return $page;
    }

    /**
     * Sets the brand, devices, series, and types elements for the page.
     *
     * @param OneBrandLetterPage $page The page to set the elements on.
     * @param string $brandCode The code of the brand.
     * @throws \Exception
     */
    private function setElements(OneBrandLetterPage $page, string $brandCode): void
    {
        $startTime = microtime(true);
        $this->logger->info('OneBrandLetterPageLoader::setElements - Start', ['brandCode' => $brandCode]);

        $return = [];

        // ---- Get brand information
        $page->brand = $this->getBrand($brandCode);
        if (!$page->brand) {
            throw new \Exception('Device brand not found!');
        }

        $this->logger->info('OneBrandLetterPageLoader::setElements - Brand loaded', ['brandCode' => $brandCode, 'duration_ms' => (microtime(true) - $startTime) * 1000]);
        $queryStartTime = microtime(true);

        // ---- Fetch devices associated with the brand
        $devices = $this->connection->fetchAllAssociative('
SELECT topdata_device.model as name, 
       LOWER(HEX(topdata_device.id)) as id, 
       LOWER(HEX(topdata_device.brand_id)) as brand_id, 
       LOWER(HEX(topdata_device.series_id)) as series_id, 
       LOWER(HEX(topdata_device.type_id)) as type_id, 
       topdata_device.code as code, 
       topdata_brand.label as brand_name,
       topdata_brand.code as brand_code,
       LOWER(HEX(topdata_device.media_id)) as media_id
  FROM topdata_brand,
       topdata_device
  WHERE (topdata_brand.id=topdata_device.brand_id)
        AND(topdata_brand.code = "' . $brandCode . '")
        AND(topdata_device.is_enabled=1)
  ORDER BY topdata_device.code
            ');

        $queryEndTime = microtime(true);
        $deviceCount = count($devices);
        $this->logger->info('OneBrandLetterPageLoader::setElements - Device query finished', [
            'brandCode'   => $brandCode,
            'deviceCount' => $deviceCount,
            'duration_ms' => ($queryEndTime - $queryStartTime) * 1000
        ]);

        if ($deviceCount === 0) {
            $this->logger->warning('OneBrandLetterPageLoader::setElements - No devices found for brand', ['brandCode' => $brandCode]);
            // Set empty arrays and return early if no devices found to avoid errors
            $page->series = [];
            $page->types = [];
            $page->devices = [];
            $endTime = microtime(true);
            $this->logger->info('OneBrandLetterPageLoader::setElements - End (no devices)', ['brandCode' => $brandCode, 'total_duration_ms' => ($endTime - $startTime) * 1000]);
            return;
        }

        $current_brand = [
            'brand'      => $devices[0]['brand_name'],
            'brand_id'   => $devices[0]['brand_id'],
            'brand_code' => $devices[0]['brand_code'],
            'devices'    => [],
            'series'     => [],
            'types'      => [],
        ];

        // ---- Get series and types
        $series = $this->getSeries();
        $types = $this->getTypes();

        $uniqueSeries = [];
        $uniqueTypes = [];

        $processingStartTime = microtime(true);
        $this->logger->info('OneBrandLetterPageLoader::setElements - Starting device processing loop', ['brandCode' => $brandCode, 'deviceCount' => $deviceCount]);

        // ---- Process devices to include series and type information
        foreach ($devices as $device) {
            if ($device['series_id'] && isset($series[$device['series_id']])) {
                $device['series_name'] = $series[$device['series_id']]['name'];
                $device['series_code'] = $series[$device['series_id']]['code'];
            } else {
                $device['series_name'] = '';
            }

            if ($device['type_id'] && isset($types[$device['type_id']])) {
                $device['type_name'] = $types[$device['type_id']]['name'];
                $device['type_code'] = $types[$device['type_id']]['code'];
            } else {
                $device['type_name'] = '';
            }

            if (!in_array($device['series_name'], $uniqueSeries)) {
                $uniqueSeries[] = $device['series_name'];
            }

            if (!in_array($device['type_name'], $uniqueTypes)) {
                $uniqueTypes[] = $device['type_name'];
            }
            $return[] = $device;
        }

        $processingEndTime = microtime(true);
        $this->logger->info('OneBrandLetterPageLoader::setElements - Finished device processing loop', [
            'brandCode'   => $brandCode,
            'deviceCount' => $deviceCount,
            'duration_ms' => ($processingEndTime - $processingStartTime) * 1000
        ]);

        // ---- Sort and clean up series and types
        sort($uniqueSeries, SORT_STRING);
        if ($uniqueSeries[0] == '') {
            unset($uniqueSeries[0]);
            $uniqueSeries[] = '';
        }
        sort($uniqueTypes, SORT_STRING);
        if ($uniqueTypes[0] == '') {
            unset($uniqueTypes[0]);
            $uniqueTypes[] = '';
        }

        // ---- Assign data to the page
        $page->series = $uniqueSeries;
        $page->types = $uniqueTypes;
        $page->devices = $return;

        $endTime = microtime(true);
        $this->logger->info('OneBrandLetterPageLoader::setElements - End', ['brandCode' => $brandCode, 'total_duration_ms' => ($endTime - $startTime) * 1000]);
    }

    /**
     * Retrieves all enabled series from the database.
     *
     * @return array An array of series data.
     */
    private function getSeries(): array
    {
        $return = [];

        // ---- Fetch series data
        $series = $this->connection->fetchAllAssociative('
SELECT label as name, 
       LOWER(HEX(id)) as id, 
       code as code
  FROM topdata_series
  WHERE is_enabled=1
  ORDER BY label
        ');

        // ---- Format series data
        foreach ($series as $serie) {
            $return[$serie['id']] = $serie;
        }

        return $return;
    }

    /**
     * Retrieves all enabled device types from the database.
     *
     * @return array An array of device type data.
     */
    private function getTypes(): array
    {
        $return = [];

        // ---- Fetch device type data
        $types = $this->connection->fetchAllAssociative('
SELECT label as name, 
       LOWER(HEX(id)) as id, 
       code as code
  FROM topdata_device_type
  WHERE is_enabled=1
  ORDER BY label
        ');

        // ---- Format device type data
        foreach ($types as $type) {
            $return[$type['id']] = $type;
        }

        return $return;
    }

    /**
     * Retrieves a brand from the database by its code.
     *
     * @param string $brandCode The code of the brand to retrieve.
     * @return array The brand data, or an empty array if not found.
     */
    private function getBrand(string $brandCode): array
    {
        // ---- Fetch brand data
        $brand = $this->connection->fetchAllAssociative('
SELECT LOWER(HEX(id)) as id, 
       code, 
       label as name
  FROM topdata_brand
  WHERE (is_enabled=1)
        AND(code = "' . $brandCode . '")
            ');

        return isset($brand[0]) ? $brand[0] : [];
    }

    /**
     * Loads the OneBrandLetterPage for JSON response, including device-specific data and media URLs.
     *
     * @param Request $request The current HTTP request.
     * @param SalesChannelContext $salesChannelContext The current sales channel context.
     * @param string $brandCode The code of the brand to load.
     * @param string $displayMode The display mode ('all', 'series', or 'types').
     * @return OneBrandLetterPage The populated OneBrandLetterPage.
     */
    public function loadJson(
        Request             $request,
        SalesChannelContext $salesChannelContext,
        string              $brandCode,
        string              $displayMode = 'all'
    ): OneBrandLetterPage
    {
        // ---- Load generic page and set elements
        $page = $this->genericLoader->load($request, $salesChannelContext);
        /** @var OneBrandLetterPage $page */
        $page = OneBrandLetterPage::createFrom($page);
        $this->setElements($page, $brandCode);

        $page->setTitle($this->translator->trans('topdata-topfinder.SEO.brandPageTitle', [
            '%brand%' => $page->brand['name'],
        ]));

        // ---- Determine the letter for the popup path
        $letter = $page->brand['code'][0];

        if (preg_match('/^[0-9]{1}$/', $letter)) {
            $letter = '0';
            $letterStr = '0-9';
        } else {
            $letterStr = strtoupper($letter);
        }

        $page->popupPath[] = [
            'name' => $letterStr,
            'path' => $this->router->generate('frontend.top-finder-api.popup-letter', ['letter' => $letter]),
        ];

        $page->popupPath[] = [
            'name' => $page->brand['name'],
        ];

        // ---- Determine the list type based on display mode
        $listType = 'brand';
        if ($displayMode == 'series') {
            usort($page->devices, function ($a, $b) {
                if ($a['series_name'] === '' && $b['series_name']) {
                    return 1;
                }
                if ($a['series_name'] && $b['series_name'] === '') {
                    return -1;
                }

                return $a['series_name'] . $a['code'] <=> $b['series_name'] . $b['code'];
            });

            $page->popupPath[] = [
                'name' => $this->translator->trans('topdata-topfinder.popup.brandSeries'),
            ];
            $listType = 'series';
        } elseif ($displayMode == 'types') {
            usort($page->devices, function ($a, $b) {
                if ($a['type_name'] === '' && $b['type_name']) {
                    return 1;
                }
                if ($a['type_name'] && $b['type_name'] === '') {
                    return -1;
                }

                return $a['type_name'] . $a['code'] <=> $b['type_name'] . $b['code'];
            });
            $page->popupPath[] = [
                'name' => $this->translator->trans('topdata-topfinder.popup.brandTypes'),
            ];
            $listType = 'types';
        } else {
            $page->popupPath[] = [
                'name' => $this->translator->trans('topdata-topfinder.popup.allModels'),
            ];
        }

        $mediaIds = [];
        $devicelist = $this->deviceToCustomerService->getDevicesOfCustomer($salesChannelContext->getCustomer());

        // ---- Process devices to include path, media URL, and device list count
        foreach ($page->devices as $key => $device) {
            $page->devices[$key]['path'] = $this->router->generate('frontend.top-finder-api.popup-device-new', [
                'deviceCode' => $device['code'],
                'listType'   => $listType,
            ]);
            if ($device['media_id']) {
                $mediaIds[] = $device['media_id'];
            }

            if (isset($devicelist[$device['id']]) && isset($devicelist[$device['id']]['devices'])) {
                $page->devices[$key]['devicelist'] = count($devicelist[$device['id']]['devices']);
            } else {
                $page->devices[$key]['devicelist'] = -1;
            }
        }

        // ---- Fetch media URLs for devices
        if (count($mediaIds)) {
            $medias = $this->mediaRepository->search(new Criteria($mediaIds), $salesChannelContext->getContext());

            foreach ($page->devices as $key => $device) {
                if ($device['media_id'] && $medias->get($device['media_id'])) {
                    $page->devices[$key]['media_url'] = $medias->get($device['media_id'])->getUrl();
                }
            }
        }

        return $page;
    }



}